<?php
/**
 * Process POS Transaction
 * Loops through all sold items and updates stock silently
 */
function processPOSTransaction($conn, $transactionItems, $receiptNumber, $userloggedid, $orgid, $transaction_date) {
    // Ensure $transactionItems is an array
    if (!is_array($transactionItems) || empty($transactionItems)) return;

    foreach ($transactionItems as $item) {
        // Call inventory update for each item
        updateInventoryAfterSale(
            $conn,
            $item,
            $receiptNumber,
            $userloggedid,
            $orgid,
            $transaction_date
        );
        // Nothing is returned, and products not in stock are silently skipped
    }
}

// The updateInventoryAfterSale function (silent)
function updateInventoryAfterSale($conn, $item, $receiptNumber, $userloggedid, $orgid, $transaction_date) {
    $productId = $item['id'] ?? null;
    $quantity = floatval($item['count'] ?? 0);

    // Invalid product/quantity, silently skip
    if (!$productId || $quantity <= 0) return;

    // Determine warehouse if provided
    $warehouseId = $item['warehouse_id'] ?? null;

    // Update stock (silent)
    updateStockForProduct(
        $conn,
        $productId,
        $quantity,
        $warehouseId,
        $receiptNumber,
        $transaction_date,
        $userloggedid,
        $orgid
    );

    // Done: nothing returned
}

// The updateStockForProduct function (silent)
function updateStockForProduct($conn, $productId, $quantity, $warehouseId, $receiptNumber, $saleDate, $userId, $orgId) {
    // Pick warehouse if not given
    if (empty($warehouseId)) {
        $stmt = $conn->prepare("
            SELECT warehouse_id
            FROM stock_balance
            WHERE product_id = :product_id
              AND org_id = :org_id
            ORDER BY quantity_available DESC
            LIMIT 1
        ");
        $stmt->execute([
            ':product_id' => $productId,
            ':org_id' => $orgId
        ]);
        $warehouse = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$warehouse) return; // Product not in stock, skip

        $warehouseId = $warehouse['warehouse_id'];
    }

    // Fetch existing stock_balance
    $stmt = $conn->prepare("
        SELECT id, quantity_on_hand, quantity_available
        FROM stock_balance
        WHERE product_id = :product_id
          AND org_id = :org_id
          AND warehouse_id = :warehouse_id
        LIMIT 1
    ");
    $stmt->execute([
        ':product_id' => $productId,
        ':org_id' => $orgId,
        ':warehouse_id' => $warehouseId
    ]);
    $stock = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$stock) return; // No stock record, skip

    // Deduct stock
    $newOnHand = max(0, floatval($stock['quantity_on_hand']) - $quantity);
    $newAvailable = max(0, floatval($stock['quantity_available']) - $quantity);

    $update = $conn->prepare("
        UPDATE stock_balance
        SET quantity_on_hand = :new_on_hand,
            quantity_available = :new_available,
            last_movement_date = :movement_date
        WHERE id = :id
    ");
    $update->execute([
        ':new_on_hand' => $newOnHand,
        ':new_available' => $newAvailable,
        ':movement_date' => $saleDate,
        ':id' => $stock['id']
    ]);
}
?>
