<?php
/**
 * Invoice Check Functions
 * This file contains only functions - no direct execution or database connections
 */

/**
 * Check if user exists or needs to be created based on charge_rate
 * Returns user ID if user exists or was created, null otherwise
 */
function checkChargeOnce($conn, $cartArray, $phone, $userloggedid = null, $orgid = null, $fname = null) {
    if (empty($phone)) return null;

    // 1) If user already exists, return id
    $userPhoneQuery = $conn->prepare("SELECT id FROM users WHERE phone = :phone LIMIT 1");
    $userPhoneQuery->execute([':phone' => $phone]);
    $userPhoneExists = $userPhoneQuery->fetch(PDO::FETCH_ASSOC);
    if ($userPhoneExists) {
        return (int)$userPhoneExists['id'];
    }

    // 2) Decide whether we need to create a user (if any product charge_rate != 'once')
    $needsCreate = false;
    foreach ($cartArray as $item) {
        $productid = $item['id'] ?? null;
        if (!$productid) continue;
        $q = $conn->prepare("SELECT charge_rate FROM tblproduct WHERE id = :productid LIMIT 1");
        $q->execute([':productid' => $productid]);
        $res = $q->fetch(PDO::FETCH_ASSOC);
        $charge_rate = strtolower($res['charge_rate'] ?? '');
        if ($charge_rate !== 'once' && $charge_rate !== 'none') {
        $needsCreate = true;
        break;
        }

    }

    if (!$needsCreate) return null;

    // 3) Create a lightweight user record
    $name = $fname ?: 'Unknown';
    $email = 'user' . (preg_replace('/\D/', '', $phone) ?: uniqid()) . '@cyberdesk.live';
    $address = '';
    $nrc = '0000';
    $status = 'Active';
    $avatar = 'xd-uploads/user-pictures/avatar.png';
    $sessionUserId = $userloggedid ?? 0;

    $insertUser = $conn->prepare("
        INSERT INTO users (email, names, phone, address, nrc, avatar, status, sessionUserId)
        VALUES (:email, :name, :phone, :address, :nrc, :avatar, :status, :sessionUserId)
    ");
    $insertUser->execute([
        ':email' => $email,
        ':name' => $name,
        ':phone' => $phone,
        ':address' => $address,
        ':nrc' => $nrc,
        ':avatar' => $avatar,
        ':status' => $status,
        ':sessionUserId' => $sessionUserId
    ]);
    $userid = (int)$conn->lastInsertId();

    // Add role
    $conn->prepare("INSERT INTO user_roles (user_id, role_id, status, created_by, org_id) 
                    VALUES (:user_id, 22, 'Active', :created_by, :org_id)")
         ->execute([':user_id' => $userid, ':created_by' => $userloggedid, ':org_id' => $orgid]);

    // Add clients entry
    $conn->prepare("INSERT INTO clients (user_id, status, business_category, account_type, depot_id, created_by, org_id)
                    VALUES (:user_id, 'Active', 1, 'Business', 1, :created_by, :org_id)")
         ->execute([':user_id' => $userid, ':created_by' => $userloggedid, ':org_id' => $orgid]);

    return $userid;
}

/**
 * Create invoices for products with recurring charge rates
 * Returns array of created invoice details
 */
function createRecurringInvoices($conn, $cartArray, $phone, $userloggedid, $orgid, $orgname, $fname = null) {
    $postClientId = filter_input(INPUT_POST, 'userid', FILTER_VALIDATE_INT);
    
    // Get or create user ID
    $userId = $postClientId ? (int)$postClientId : null;
    if (!$userId) {
        $userId = checkChargeOnce($conn, $cartArray, $phone, $userloggedid, $orgid, $fname);
    } else {
        // Verify user exists
        $uCheck = $conn->prepare("SELECT id FROM users WHERE id = :id LIMIT 1");
        $uCheck->execute([':id' => $userId]);
        if (!$uCheck->fetch(PDO::FETCH_ASSOC)) {
            $userId = checkChargeOnce($conn, $cartArray, $phone, $userloggedid, $orgid, $fname);
        }
    }

    // Determine invoice number (lock latest row to prevent duplicates)
    $stmtLast = $conn->prepare("SELECT b_id, invoice_no FROM billing_subscription 
                                WHERE org_id = :orgid ORDER BY b_id DESC LIMIT 1 FOR UPDATE");
    $stmtLast->execute([':orgid' => $orgid]);
    if ($stmtLast->rowCount() === 0) {
        $invoiceNumber = 1;
    } else {
        $row = $stmtLast->fetch(PDO::FETCH_ASSOC);
        $invoiceNumber = (int)$row['invoice_no'] + 1;
    }

    // Fetch user details if we have a userId
    $fname = $lname = $email = $deliveryaddress = $plotnumber = $area = $nrc = $businesscategory = null;
    if ($userId) {
        $userQuery = $conn->prepare("
            SELECT u.names, u.email, u.phone, u.address, u.nrc,
                   c.business_category
            FROM users u
            LEFT JOIN clients c ON u.id = c.user_id
            WHERE u.id = :userid
            LIMIT 1
        ");
        $userQuery->execute([':userid' => $userId]);
        $userData = $userQuery->fetch(PDO::FETCH_ASSOC);
        if ($userData) {
            $fname = $userData['names'] ?? '';
            $lname = '';
            $email = $userData['email'] ?? '';
            $deliveryaddress = $userData['address'] ?? '';
            $plotnumber = $userData['address'] ?? '';
            $area = $userData['address'] ?? '';
            $nrc = $userData['nrc'] ?? '';
            $businesscategory = $userData['business_category'] ?? '';
        }
    }

    // Prepare statements
    $selectChargeRateStmt = $conn->prepare("SELECT charge_rate FROM tblproduct WHERE id = :pid LIMIT 1");
    $checkSubscriptionStmt = $conn->prepare("SELECT COUNT(*) FROM billing_subscription 
                                             WHERE client_id = :cid AND product_id = :pid");
    $insertBillingStmt = $conn->prepare("
        INSERT INTO billing_subscription 
        (company_name, company_registration_number, tpin, business_category, client_name, client_id, 
         product_name, product_id, product_code, units, unit_price, total_charge, product_category, 
         product_category_code, billing_interval, balance_bf, paid, balance, balance_cf, invoice_no, due_date, 
         due_time, b_status, phone_contact, email_contact, delivery_address, plot_number, area, nrc, 
         user_id, document_type, depot, depot_id, depot_code, original_receipt_details, comment, 
         org_name, org_id, session_id)
        VALUES
        (:company_name, :company_reg, :tpin, :business_category, :client_name, :client_id,
         :product_name, :product_id, :product_code, :units, :unit_price, :total_charge, :product_category, 
         :product_category_code, :billing_interval, :balance_bf, :paid, :balance, :balance_cf, :invoice_no, 
         :due_date, :due_time, :b_status, :phone_contact, :email_contact, :delivery_address, :plot_number, 
         :area, :nrc, :user_id, :document_type, :depot, :depot_id, :depot_code, :original_receipt_details, 
         :comment, :org_name, :org_id, :session_id)
    ");

    // Get current time in Lusaka timezone
    $tz = new DateTimeZone('Africa/Lusaka');
    $now = new DateTimeImmutable('now', $tz);
    $transaction_date = $now->format('Y-m-d');
    $transaction_time = $now->format('H:i:s');

    $created = [];

    // Process each cart item
    foreach ($cartArray as $item) {
        $productid = (int)($item['id'] ?? 0);
        if (!$productid) continue;

        // Get charge_rate from database
        $selectChargeRateStmt->execute([':pid' => $productid]);
        $p = $selectChargeRateStmt->fetch(PDO::FETCH_ASSOC);
        $charge_rate = strtolower($p['charge_rate'] ?? '');

        // Skip 'once' products
        if ($charge_rate === 'once' || $charge_rate === 'none') {
        continue;
        }

        // If no user yet, try to create
        if (!$userId) {
            $userId = checkChargeOnce($conn, $cartArray, $phone, $userloggedid, $orgid, $fname);
            if ($userId && empty($fname)) {
                $userQuery->execute([':userid' => $userId]);
                $userData = $userQuery->fetch(PDO::FETCH_ASSOC);
                if ($userData) {
                    $fname = $userData['names'] ?? '';
                    $email = $userData['email'] ?? '';
                    $deliveryaddress = $userData['address'] ?? '';
                    $plotnumber = $userData['address'] ?? '';
                    $area = $userData['address'] ?? '';
                    $nrc = $userData['nrc'] ?? '';
                    $businesscategory = $userData['business_category'] ?? '';
                }
            }
            if (!$userId) {
                continue;
            }
        }

        // Check if subscription already exists
        $checkSubscriptionStmt->execute([':cid' => $userId, ':pid' => $productid]);
        $exists = (int)$checkSubscriptionStmt->fetchColumn();
        if ($exists > 0) {
            continue;
        }

        // Prepare invoice data
        $units = (int)($item['count'] ?? 1);
        $price = number_format((float)($item['price'] ?? 0), 2, '.', '');
        $totalcharge = number_format($units * (float)$price, 2, '.', '');
        $totalbalance = 0;
        $balance_brought_forward = $item['balancebroughtforward'] ?? 0;
        $balance_cf = $item['balanceCF'] ?? 0;
        $product_code = $item['productcode'] ?? '';
        $product_category = $item['category'] ?? '';
        $product_category_code = $item['categorycode'] ?? '';
        $depot = $item['depot'] ?? '';
        $depotid = (int)($item['depotid'] ?? 0);
        $depotcode = $item['depotcode'] ?? '';
        $cartObjectRaw = json_encode($cartArray);

        // Insert billing subscription
        $insertBillingStmt->execute([
            ':company_name' => $fname,
            ':company_reg' => $item['pacra'] ?? '',
            ':tpin' => $item['tpin'] ?? '',
            ':business_category' => $businesscategory ?? 1,
            ':client_name' => $fname,
            ':client_id' => $userId,
            ':product_name' => $item['name'] ?? '',
            ':product_id' => $productid,
            ':product_code' => $product_code,
            ':units' => $units,
            ':unit_price' => $price,
            ':total_charge' => $totalcharge,
            ':product_category' => $product_category,
            ':product_category_code' => $product_category_code,
            ':billing_interval' => $charge_rate,
            ':balance_bf' => $balance_brought_forward,
            ':paid' => $totalcharge,
            ':balance' => $totalbalance,
            ':balance_cf' => $balance_cf,
            ':invoice_no' => $invoiceNumber,
            ':due_date' => $transaction_date,
            ':due_time' => $transaction_time,
            ':b_status' => 'Active',
            ':phone_contact' => $phone,
            ':email_contact' => $email,
            ':delivery_address' => $deliveryaddress,
            ':plot_number' => $plotnumber,
            ':area' => $area,
            ':nrc' => $nrc,
            ':user_id' => $userloggedid,
            ':document_type' => 'INVOICE',
            ':depot' => $depot,
            ':depot_id' => $depotid,
            ':depot_code' => $depotcode,
            ':original_receipt_details' => $cartObjectRaw,
            ':comment' => $item['comment'] ?? 'Auto created invoice',
            ':org_name' => $orgname,
            ':org_id' => $orgid,
            ':session_id' => $userloggedid
        ]);

        $created[] = [
            'product_id' => $productid,
            'product_name' => $item['name'] ?? '',
            'invoice_no' => $invoiceNumber,
            'b_id' => $conn->lastInsertId()
        ];
    }

    return [
        'status' => 'created',
        'invoice_no' => $invoiceNumber,
        'created_count' => count($created),
        'created' => $created,
        'user_id' => $userId
    ];
}
?>