<?php
/**
 * Create Product Backend (PDO Version)
 * File: xd-modules/inventory/backend/create_product.php
 * Purpose: Insert new product into tblproduct table with image upload
 */

// Start session and include database connection
session_start();
header('Content-Type: application/json');

// Include your database connection file
require_once '../../../../xd-assets/backend/config/dbconfig.php'; // Adjust path as needed

// Initialize response array
$response = array(
    'success' => false,
    'message' => '',
    'product_id' => null
);

try {
    // Check if this is a file upload request (multipart/form-data)
    $isFileUpload = isset($_FILES['product_image']) && $_FILES['product_image']['error'] !== UPLOAD_ERR_NO_FILE;
    
    // Get data based on request type
    if ($isFileUpload) {
        $data = $_POST;
    } else {
        $input = file_get_contents('php://input');
        $data = json_decode($input, true);
    }

    // Validate required fields
    if (empty($data['product_id'])) {
        throw new Exception('Product ID is required for update');
    }
    if (empty($data['product_code'])) {
        throw new Exception('Product code is required');
    }
    if (empty($data['name'])) {
        throw new Exception('Product name is required');
    }
    if (empty($data['category'])) {
        throw new Exception('Category is required');
    }
    if (empty($data['product_type'])) {
        throw new Exception('Product type is required');
    }
    if (empty($data['price'])) {
        throw new Exception('Selling price is required');
    }
    if (empty($data['units'])) {
        throw new Exception('Unit of measure is required');
    }
    if (empty($data['reorder_level'])) {
        throw new Exception('Reorder level is required');
    }

    // Sanitize and prepare data
    $product_id = intval($data['product_id']);
    $product_code = trim($data['product_code']);
    $name = trim($data['name']);
    $description = !empty($data['description']) ? trim($data['description']) : null;
    $category = trim($data['category']);
    $sub_category = !empty($data['sub_category']) ? trim($data['sub_category']) : null;
    $product_brand = !empty($data['product_brand']) ? trim($data['product_brand']) : null;
    $product_type = trim($data['product_type']);
    $price = floatval($data['price']);
    $cost_price = !empty($data['cost_price']) ? floatval($data['cost_price']) : 0;
    $units = trim($data['units']);
    $reorder_level = intval($data['reorder_level']);
    $color = !empty($data['color']) ? trim($data['color']) : null;
    $weight = !empty($data['weight']) ? trim($data['weight']) : null;
    $dimensions = !empty($data['dimensions']) ? trim($data['dimensions']) : null;
    $barcode = !empty($data['barcode']) ? trim($data['barcode']) : null;
    $lead_time = !empty($data['lead_time']) ? trim($data['lead_time']) : '0';
    $supplier_id = !empty($data['supplier_id']) ? intval($data['supplier_id']) : null;
    $status = isset($data['status']) ? trim($data['status']) : 'active';
    $tax_rate = !empty($data['tax_rate']) ? trim($data['tax_rate']) : null;
    $max_stock_level = !empty($data['max_stock_level']) ? intval($data['max_stock_level']) : null;
    
     $org_id = $_SESSION["orgid"]; 
    $user_id = $_SESSION["userid"];
    // Get current image path from DB
    $imagePath = null;
    $img_sql = "SELECT image FROM tblproduct WHERE id = :product_id AND org_id = :org_id";
    $img_stmt = $conn->prepare($img_sql);
    $img_stmt->execute([':product_id' => $product_id, ':org_id' => $org_id]);
    $img_row = $img_stmt->fetch(PDO::FETCH_ASSOC);
    if ($img_row) {
        $imagePath = $img_row['image'];
    } else {
        throw new Exception('Product not found');
    }

    // Handle image upload
    if ($isFileUpload && $_FILES['product_image']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = '../../../../xd-uploads/products/images/';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        $fileInfo = pathinfo($_FILES['product_image']['name']);
        $fileExtension = strtolower($fileInfo['extension']);
        $allowedExtensions = array('jpg', 'jpeg', 'png', 'gif', 'webp');
        if (!in_array($fileExtension, $allowedExtensions)) {
            throw new Exception('Invalid image format. Only JPG, PNG, GIF, and WEBP are allowed.');
        }
        $maxFileSize = 5 * 1024 * 1024;
        if ($_FILES['product_image']['size'] > $maxFileSize) {
            throw new Exception('Image size too large. Maximum 5MB allowed.');
        }
        $newFileName = 'product_' . uniqid() . '_' . time() . '.' . $fileExtension;
        $uploadPath = $uploadDir . $newFileName;
        if (move_uploaded_file($_FILES['product_image']['tmp_name'], $uploadPath)) {
            $imagePath = 'xd-uploads/products/images/' . $newFileName;
        } else {
            throw new Exception('Failed to upload image. Please try again.');
        }
    }

    // Get supplier name if supplier_id provided
    $supplier_name = null;
    if ($supplier_id) {
        $supplier_sql = "SELECT supplier_name FROM supplier WHERE id = :supplier_id";
        $supplier_stmt = $conn->prepare($supplier_sql);
        $supplier_stmt->execute([':supplier_id' => $supplier_id]);
        $supplier_row = $supplier_stmt->fetch(PDO::FETCH_ASSOC);
        if ($supplier_row) {
            $supplier_name = $supplier_row['supplier_name'];
        }
    }

    // Prepare SQL UPDATE statement
    $sql = "UPDATE tblproduct SET
        name = :name,
        product_code = :product_code,
        image = :image,
        price = :price,
        charge_rate = :charge_rate,
        reorder_level = :reorder_level,
        cost = :cost_price,
        max_stock_level = :max_stock_level,
        warranty = :warranty,
        description = :description,
        color = :color,
        weight = :weight,
        dimensions = :dimensions,
        category = :category,
        sub_category = :sub_category,
        units = :units,
        product_type = :product_type,
        product_brand = :product_brand,
        barcode = :barcode,
        lead_time = :lead_time,
        tax_rate = :tax_rate,
        supplier_id = :supplier_id,
        supplier_name = :supplier_name,
        user_id = :user_id,
        status = :status,
        org_id = :org_id,
        sessionUserId = :sessionUserId,
        datemodified = NOW()
    WHERE id = :product_id AND org_id = :org_id";

    $stmt = $conn->prepare($sql);
    $charge_rate = 'Once off'; // Default
    $warranty = null; // Not provided
    $stmt->execute([
        ':name' => $name,
        ':product_code' => $product_code,
        ':image' => $imagePath,
        ':price' => $price,
        ':charge_rate' => $charge_rate,
        ':reorder_level' => $reorder_level,
        ':description' => $description,
        ':color' => $color,
        ':weight' => $weight,
        ':dimensions' => $dimensions,
        ':category' => $category,
        ':sub_category' => $sub_category,
        ':units' => $units,
        ':product_type' => $product_type,
        ':product_brand' => $product_brand,
        ':barcode' => $barcode,
        ':lead_time' => $lead_time,
        ':supplier_id' => $supplier_id,
        ':supplier_name' => $supplier_name,
        ':user_id' => $user_id,
        ':status' => $status,
        ':org_id' => $org_id,
        ':sessionUserId' => $user_id,
        ':cost_price' => $cost_price,
        ':tax_rate' => $tax_rate,
        ':max_stock_level' => $max_stock_level,
        ':warranty' => $warranty,
        ':product_id' => $product_id
    ]);

    // Log the activity
    logProductActivity($conn, $product_id, 'update', $user_id, "Updated product: $name", $org_id);

    $response['success'] = true;
    $response['message'] = 'Product updated successfully';
    $response['product_id'] = $product_id;
    $response['product_code'] = $product_code;
    $response['image_path'] = $imagePath;

} catch (PDOException $e) {
    $response['success'] = false;
    $response['message'] = 'Database error: ' . $e->getMessage();
    error_log("Product Update DB Error: " . $e->getMessage());
    
} catch (Exception $e) {
    $response['success'] = false;
    $response['message'] = $e->getMessage();
    error_log("Product Update Error: " . $e->getMessage());
}

// Connection is automatically closed when script ends (PDO)

// Return JSON response
echo json_encode($response);

/**
 * Log product activity to audit log (PDO version)
 * Optional function - only use if you want audit trail
 */
function logProductActivity($conn, $product_id, $action, $user_id, $description, $org_id) {
    try {
        $sql = "INSERT INTO inventory_audit_log (
            entity_type,
            entity_id,
            action,
            old_value,
            new_value,
            user_id,
            ip_address,
            user_agent,
            reason,
            org_id,
            created_at
        ) VALUES (
            :entity_type,
            :entity_id,
            :action,
            :old_value,
            :new_value,
            :user_id,
            :ip_address,
            :user_agent,
            :reason,
            :org_id,
            NOW()
        )";
        
        $stmt = $conn->prepare($sql);
        
        $entity_type = 'product';
        $old_value = null;
        $new_value = json_encode(['product_id' => $product_id]);
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? null;
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? null;
        
        $stmt->execute([
            ':entity_type' => $entity_type,
            ':entity_id' => $product_id,
            ':action' => $action,
            ':old_value' => $old_value,
            ':new_value' => $new_value,
            ':user_id' => $user_id,
            ':ip_address' => $ip_address,
            ':user_agent' => $user_agent,
            ':reason' => $description,
            ':org_id' => $org_id
        ]);
        
    } catch (PDOException $e) {
        // Silently fail - don't break main flow if audit logging fails
        error_log("Audit Log Error: " . $e->getMessage());
    }
}
?>