<?php
header('Content-Type: application/json');
require_once '../../../../xd-assets/backend/config/dbconfig.php';

$response = ['success' => false, 'message' => ''];

try {
    // Get input data
    $movement_id = $_POST['movement_id'] ?? '';
    $status = $_POST['status'] ?? '';
    $user_id = $_POST['user_id'] ?? '';
    
    if (empty($movement_id) || empty($status)) {
        throw new Exception('Movement ID and status are required');
    }
    
    // Validate status
    $valid_statuses = ['pending', 'approved', 'completed', 'cancelled'];
    if (!in_array($status, $valid_statuses)) {
        throw new Exception('Invalid status');
    }
    
    // Use your existing connection
    $pdo = $conn;
    
    // Check if movement exists
    $check_sql = "SELECT * FROM stock_movement WHERE id = ?";
    $check_stmt = $pdo->prepare($check_sql);
    $check_stmt->execute([$movement_id]);
    $movement = $check_stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$movement) {
        throw new Exception('Movement not found');
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    try {
        // Update movement status
        $update_sql = "UPDATE stock_movement SET status = ?, processed_by = ? WHERE id = ?";
        $update_stmt = $pdo->prepare($update_sql);
        $update_stmt->execute([$status, $user_id, $movement_id]);
        
        // If movement is being completed, update stock balance
        if ($status === 'completed') {
            updateStockBalance($pdo, $movement);
        }
        
        // Log the status change
        logMovementStatusChange($pdo, $movement_id, $movement['status'], $status, $user_id);
        
        $pdo->commit();
        
        $response['success'] = true;
        $response['message'] = "Movement status updated successfully to " . strtoupper($status);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
    
} catch (PDOException $e) {
    $response['error'] = "Database error: " . $e->getMessage();
} catch (Exception $e) {
    $response['error'] = $e->getMessage();
}

echo json_encode($response);

// Function to update stock balance when movement is completed
function updateStockBalance($pdo, $movement) {
    $product_id = $movement['product_id'];
    $quantity = $movement['quantity'];
    $movement_type = $movement['movement_type'];
    
    switch ($movement_type) {
        case 'inbound':
            $warehouse_id = $movement['to_warehouse_id'];
            updateStockQuantity($pdo, $product_id, $warehouse_id, $quantity, 'increase', $movement);
            break;
            
        case 'outbound':
            $warehouse_id = $movement['from_warehouse_id'];
            updateStockQuantity($pdo, $product_id, $warehouse_id, $quantity, 'decrease', $movement);
            break;
            
        case 'transfer':
            $from_warehouse_id = $movement['from_warehouse_id'];
            $to_warehouse_id = $movement['to_warehouse_id'];
            updateStockQuantity($pdo, $product_id, $from_warehouse_id, $quantity, 'decrease', $movement);
            updateStockQuantity($pdo, $product_id, $to_warehouse_id, $quantity, 'increase', $movement);
            break;
            
        case 'adjustment':
            $warehouse_id = $movement['to_warehouse_id'] ?: $movement['from_warehouse_id'];
            if ($quantity > 0) {
                updateStockQuantity($pdo, $product_id, $warehouse_id, $quantity, 'increase', $movement);
            } else {
                updateStockQuantity($pdo, $product_id, $warehouse_id, abs($quantity), 'decrease', $movement);
            }
            break;
    }
}

// Helper function to update stock quantity
function updateStockQuantity($pdo, $product_id, $warehouse_id, $quantity, $operation, $movement = null) {
    // Check if stock balance record exists
    $check_sql = "SELECT * FROM stock_balance WHERE product_id = ? AND warehouse_id = ?";
    $check_stmt = $pdo->prepare($check_sql);
    $check_stmt->execute([$product_id, $warehouse_id]);
    $stock_balance = $check_stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($stock_balance) {
        // Update existing record
        $current_qty = $stock_balance['quantity_on_hand'];
        $current_allocated = $stock_balance['quantity_allocated'];
        $average_cost = $stock_balance['average_cost'];
        
        if ($operation === 'increase') {
            $new_qty = $current_qty + $quantity;
        } else {
            $new_qty = $current_qty - $quantity;
            if ($new_qty < 0) $new_qty = 0;
        }
        
        $update_sql = "
            UPDATE stock_balance 
            SET quantity_on_hand = ?,
                quantity_available = ? - quantity_allocated,
                last_movement_date = NOW(),
                updated_at = NOW()
            WHERE product_id = ? AND warehouse_id = ?
        ";
        $update_stmt = $pdo->prepare($update_sql);
        $update_stmt->execute([$new_qty, $new_qty, $product_id, $warehouse_id]);
        
    } else {
        // Create new stock balance record
        if ($operation === 'increase') {
            $insert_sql = "
                INSERT INTO stock_balance 
                (product_id, warehouse_id, quantity_on_hand, quantity_available, last_movement_date, created_at, updated_at)
                VALUES (?, ?, ?, ?, NOW(), NOW(), NOW())
            ";
            $insert_stmt = $pdo->prepare($insert_sql);
            $insert_stmt->execute([$product_id, $warehouse_id, $quantity, $quantity]);
        }
    }
}

// Function to log status changes
function logMovementStatusChange($pdo, $movement_id, $old_status, $new_status, $user_id) {
    $log_sql = "
        INSERT INTO inventory_audit_log 
        (entity_type, entity_id, action, field_name, old_value, new_value, user_id, reason, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
    ";
    $log_stmt = $pdo->prepare($log_sql);
    $log_stmt->execute([
        'stock_movement',
        $movement_id,
        'update',
        'status',
        $old_status,
        $new_status,
        $user_id,
        "Movement status updated from $old_status to $new_status"
    ]);
}
?>