<?php
/**
 * Create Initial Stock / Handle Stock Transactions
 * Handles: Purchases, Transfers, Returns
 */

header('Content-Type: application/json');
session_start();
require_once '../../../../xd-assets/backend/config/dbconfig.php';

$response = ['success' => false, 'message' => '', 'details' => []];

try {
    $input = json_decode(file_get_contents("php://input"), true);

    if (!$input || !isset($input['warehouse_id'], $input['products']) || !is_array($input['products'])) {
        throw new Exception("Invalid input data received");
    }

    $warehouse_id = $input['warehouse_id'];
    //$supplier_id = $input['supplier_id'];
    $org_id = $_SESSION["orgid"]; 
    $user_id = $_SESSION["userid"];  
    $notes = $input['notes'] ?? null;
         

    // ✅ Start Transaction
    $conn->beginTransaction();

    foreach ($input['products'] as $p) {
        $product_id = $p['product_id'] ?? null;
        $quantity = $p['quantity'] ?? 0;
        $unit_cost = $p['unit_cost'] ?? 0;
         $reorder_level = $p['reorder_level'] ?? 0;
         $reorder_point = $p['reorder_point'] ?? 0; 
         $quantity_allocated = $p['stock_allocated'] ?? 0;
        $max_stock_level = $p['max_stock_level'] ?? 0;
        $batch_number = $p['batch_number'] ?? null;
        $expiry_date = !empty($p['expiry_date']) ? $p['expiry_date'] : null;
        //$stockavailable = $p['avalible_stocK'] ?? 0;
        $stockavailable =   $quantity-$quantity_allocated;
        
        if (!$product_id || $quantity <= 0) {
            throw new Exception("Missing or invalid product details.");
        }

        // 1️⃣ — STOCK BATCH (if batch number provided)
        if (!empty($batch_number)) {
            $stmt = $conn->prepare("
                INSERT INTO stock_batch 
                (product_id, batch_number, warehouse_id, quantity, unit_cost, expiry_date, org_id)
                VALUES (:product_id, :batch_number, :warehouse_id, :quantity, :unit_cost, :expiry_date, :org_id)
                ON DUPLICATE KEY UPDATE 
                    quantity = quantity + VALUES(quantity),
                    unit_cost = VALUES(unit_cost)
            ");
            $stmt->execute([
                ':product_id' => $product_id,
                ':batch_number' => $batch_number,
                ':warehouse_id' => $warehouse_id,
                ':quantity' => $quantity,
                ':unit_cost' => $unit_cost,
                ':expiry_date' => $expiry_date,
                ':org_id' => $org_id
            ]);
        }

        // 2️⃣ — STOCK BALANCE (update or insert)
        $stmt = $conn->prepare("SELECT id, quantity_on_hand FROM stock_balance WHERE product_id = ? AND warehouse_id = ?");
        $stmt->execute([$product_id, $warehouse_id]);
        $existingBalance = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($existingBalance) {
            $new_qty = $existingBalance['quantity_on_hand'] + $quantity;
            $stmt = $conn->prepare("
                UPDATE stock_balance 
                SET quantity_on_hand = ?, quantity_available = ?, last_movement_date = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$new_qty, $new_qty, $existingBalance['id']]);
        } else {
            $stmt = $conn->prepare("
                INSERT INTO stock_balance 
                (product_id, warehouse_id, quantity_on_hand, quantity_allocated, quantity_available, reorder_level, reorder_point, max_stock_level, average_cost, org_id, last_movement_date)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?,?, NOW())
            ");
            $stmt->execute([$product_id, $warehouse_id, $quantity, $quantity_allocated,$stockavailable, $reorder_level, $reorder_point, $max_stock_level,$unit_cost, $org_id]);
        }

        // 3️⃣ — STOCK MOVEMENT (record inbound movement)
        $movement_code = uniqid("MOV-");
        $stmt = $conn->prepare("
            INSERT INTO stock_movement
            (movement_code, product_id, movement_type, reference_type, quantity, unit_cost, total_cost, 
             to_warehouse_id, batch_number, expiry_date, movement_date, notes, processed_by, org_id, status)
            VALUES 
            (:movement_code, :product_id, 'inbound', 'purchase_order', :quantity, :unit_cost, :total_cost,
             :to_warehouse_id, :batch_number, :expiry_date, NOW(), :notes, :processed_by, :org_id, 'completed')
        ");
        $stmt->execute([
            ':movement_code' => $movement_code,
            ':product_id' => $product_id,
            ':quantity' => $quantity,
            ':unit_cost' => $unit_cost,
            ':total_cost' => $unit_cost * $quantity,
            ':to_warehouse_id' => $warehouse_id,
            ':batch_number' => $batch_number,
            ':expiry_date' => $expiry_date,
            ':notes' => $notes,
            ':processed_by' => $user_id,
            ':org_id' => $org_id
        ]);

        // 4️⃣ — UPDATE tblproduct.quantity
        $stmt = $conn->prepare("UPDATE tblproduct SET quantity = quantity + ? WHERE id = ?");
        $stmt->execute([$quantity, $product_id]);

        // 5️⃣ — INVENTORY AUDIT LOG
        $stmt = $conn->prepare("
            INSERT INTO inventory_audit_log 
            (entity_type, entity_id, action, user_id, org_id, reason) 
            VALUES ('stock_movement', ?, 'create', ?, ?, 'Initial stock or purchase entry')
        ");
        $stmt->execute([$product_id, $user_id, $org_id]);
    }

    // ✅ Commit Transaction
    $conn->commit();
    $response['success'] = true;
    $response['message'] = "Stock operation successful.";
} catch (Exception $e) {
    if ($conn->inTransaction()) {
        $conn->rollBack();
    }
    $response['message'] = "Error: " . $e->getMessage();
}

echo json_encode($response);
