<?php
/**
 * Get All Goods Receipts (List)
 * File: xd-modules/inventory/procurement/backend/get_goods_receipts.php
 * Purpose: Lists all GRNs with summary information
 */

session_start();
require_once '../../../../xd-assets/backend/config/dbconfig.php';
header('Content-Type: application/json');

$response = ['success' => false, 'data' => [], 'message' => ''];

try {
    $orgId = $_SESSION['orgid'] ?? null;

    if (!$orgId) {
        throw new Exception('Organization ID not found in session');
    }

    // Optional filters
    $search = $_GET['search'] ?? null;
    $startDate = $_GET['start_date'] ?? null;
    $endDate = $_GET['end_date'] ?? null;

    // Base query
    $sql = "
        SELECT 
            gr.id,
            gr.grn_number,
            gr.po_id,
            gr.warehouse_id,
            gr.receipt_date,
            gr.received_by,
            gr.total_amount,
            gr.notes,
            gr.status,
            gr.created_at,
            po.po_number,
            s.supplier_name,
            w.warehouse_name,
            u.names AS received_by_name,
            COUNT(DISTINCT gri.id) AS item_count,
            SUM(gri.quantity_received) AS total_quantity
        FROM goods_receipt gr
        LEFT JOIN purchase_order po ON gr.po_id = po.id
        LEFT JOIN supplier s ON po.supplier_id = s.id
        LEFT JOIN warehouse w ON gr.warehouse_id = w.id
        LEFT JOIN users u ON gr.received_by = u.id
        LEFT JOIN goods_receipt_item gri ON gr.id = gri.grn_id
        WHERE gr.org_id = :org_id
    ";

    $params = [':org_id' => $orgId];

    // Add search filter
    if ($search) {
        $sql .= " AND (gr.grn_number LIKE :search OR s.supplier_name LIKE :search OR w.warehouse_name LIKE :search)";
        $params[':search'] = '%' . $search . '%';
    }

    // Add date filters
    if ($startDate) {
        $sql .= " AND gr.receipt_date >= :start_date";
        $params[':start_date'] = $startDate;
    }

    if ($endDate) {
        $sql .= " AND gr.receipt_date <= :end_date";
        $params[':end_date'] = $endDate;
    }

    $sql .= " GROUP BY gr.id ORDER BY gr.id DESC";

    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $receipts = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Format data
    foreach ($receipts as &$receipt) {
        $receipt['total_amount'] = floatval($receipt['total_amount'] ?? 0);
        $receipt['item_count'] = intval($receipt['item_count'] ?? 0);
        $receipt['total_quantity'] = floatval($receipt['total_quantity'] ?? 0);
    }

    $response = [
        'success' => true,
        'data' => $receipts,
        'count' => count($receipts),
        'message' => 'Goods receipts retrieved successfully'
    ];

} catch (Exception $e) {
    $response['message'] = $e->getMessage();
    error_log("Get GRNs Error: " . $e->getMessage());
}

$conn = null;
echo json_encode($response);
?>