<?php
/**
 * Create Goods Receipt (GRN)
 * File: xd-modules/inventory/procurement/backend/create_goods_receipt.php
 * 
 * THIS IS THE MOST CRITICAL FILE - IT INCREASES STOCK!
 */

session_start();
require_once '../../../../xd-assets/backend/config/dbconfig.php';
header('Content-Type: application/json');

$response = ['success' => false, 'message' => ''];

try {
    // Get JSON input
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    if (!$data) {
        throw new Exception("Invalid JSON data received");
    }

    $poId = $data['po_id'] ?? null;
    $warehouseId = $data['warehouse_id'] ?? null;
    $receiptDate = $data['receipt_date'] ?? date('Y-m-d');
    $notes = $data['notes'] ?? '';
    $items = $data['items'] ?? [];

    $orgId = $_SESSION['orgid'] ?? null;
    $userId = $_SESSION['userid'] ?? null;

    // Validation
    if (!$poId) throw new Exception("Purchase Order is required");
    if (!$warehouseId) throw new Exception("Warehouse is required");
    if (empty($items)) throw new Exception("At least one item is required");

    // Get next GRN number
    $stmt = $conn->query("SELECT grn_number FROM goods_receipt WHERE org_id = '$orgId' ORDER BY id DESC LIMIT 1");
    $lastGRN = $stmt->fetch(PDO::FETCH_ASSOC);
    $nextNumber = ($lastGRN && isset($lastGRN['grn_number'])) ? ((int)$lastGRN['grn_number'] + 1) : 1;

    // Begin transaction
    $conn->beginTransaction();

    // Calculate total amount
    $totalAmount = 0;
    foreach ($items as $item) {
        $totalAmount += ($item['received_qty'] * $item['unit_price']);
    }

    // Insert goods_receipt
    $stmt = $conn->prepare("
        INSERT INTO goods_receipt (
            grn_number, po_id, warehouse_id, receipt_date, 
            received_by, total_amount, notes, status, org_id
        ) VALUES (
            :grn_number, :po_id, :warehouse_id, :receipt_date, 
            :received_by, :total_amount, :notes, 'Completed', :org_id
        )
    ");

    $stmt->execute([
        ':grn_number' => $nextNumber,
        ':po_id' => $poId,
        ':warehouse_id' => $warehouseId,
        ':receipt_date' => $receiptDate,
        ':received_by' => $userId,
        ':total_amount' => $totalAmount,
        ':notes' => $notes,
        ':org_id' => $orgId
    ]);

    $grnId = $conn->lastInsertId();

    // Process each item
    foreach ($items as $item) {
        $productId = $item['product_id'];
        $receivedQty = floatval($item['received_qty']);
        $unitPrice = floatval($item['unit_price']);
        $batchNumber = $item['batch_number'] ?? null;
        $expiryDate = $item['expiry_date'] ?? null;
        $lineTotal = $receivedQty * $unitPrice;

        // 1. Insert goods_receipt_item
        $stmt = $conn->prepare("
            INSERT INTO goods_receipt_item (
                grn_id, po_item_id, product_id, quantity_received, 
                unit_cost, line_total, batch_number, expiry_date
            ) VALUES (
                :grn_id, :po_item_id, :product_id, :qty, 
                :unit_cost, :line_total, :batch_number, :expiry_date
            )
        ");

        $stmt->execute([
            ':grn_id' => $grnId,
            ':po_item_id' => $item['po_item_id'] ?? null,
            ':product_id' => $productId,
            ':qty' => $receivedQty,
            ':unit_cost' => $unitPrice,
            ':line_total' => $lineTotal,
            ':batch_number' => $batchNumber,
            ':expiry_date' => $expiryDate
        ]);

        // 2. Create stock_movement (THIS RECORDS THE RECEIPT)
        $stmt = $conn->prepare("
            INSERT INTO stock_movement (
                product_id, warehouse_id, movement_type, quantity, 
                reference_type, reference_id, movement_date, notes, org_id
            ) VALUES (
                :product_id, :warehouse_id, 'IN', :qty, 
                'GRN', :grn_id, :movement_date, :notes, :org_id
            )
        ");

        $stmt->execute([
            ':product_id' => $productId,
            ':warehouse_id' => $warehouseId,
            ':qty' => $receivedQty,
            ':grn_id' => $grnId,
            ':movement_date' => $receiptDate,
            ':notes' => "Goods received - GRN #$nextNumber",
            ':org_id' => $orgId
        ]);

        // 3. Update or create stock_balance (THIS INCREASES STOCK!)
        $stmt = $conn->prepare("
            SELECT id, quantity_on_hand, average_cost 
            FROM stock_balance 
            WHERE product_id = :product_id 
            AND warehouse_id = :warehouse_id 
            AND org_id = :org_id
        ");
        $stmt->execute([
            ':product_id' => $productId,
            ':warehouse_id' => $warehouseId,
            ':org_id' => $orgId
        ]);
        $balance = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($balance) {
            // Update existing balance
            $oldQty = floatval($balance['quantity_on_hand']);
            $oldCost = floatval($balance['average_cost']);
            $newQty = $oldQty + $receivedQty;
            
            // Calculate weighted average cost
            $newAvgCost = (($oldQty * $oldCost) + ($receivedQty * $unitPrice)) / $newQty;

            $stmt = $conn->prepare("
                UPDATE stock_balance 
                SET quantity_on_hand = :new_qty,
                    quantity_available = quantity_available + :received_qty,
                    average_cost = :new_avg_cost,
                    last_movement_date = :movement_date
                WHERE id = :balance_id
            ");

            $stmt->execute([
                ':new_qty' => $newQty,
                ':received_qty' => $receivedQty,
                ':new_avg_cost' => $newAvgCost,
                ':movement_date' => $receiptDate,
                ':balance_id' => $balance['id']
            ]);
        } else {
            // Create new balance
            $stmt = $conn->prepare("
                INSERT INTO stock_balance (
                    product_id, warehouse_id, quantity_on_hand, 
                    quantity_available, quantity_allocated, average_cost, 
                    last_movement_date, org_id
                ) VALUES (
                    :product_id, :warehouse_id, :qty, 
                    :qty, 0, :unit_cost, 
                    :movement_date, :org_id
                )
            ");

            $stmt->execute([
                ':product_id' => $productId,
                ':warehouse_id' => $warehouseId,
                ':qty' => $receivedQty,
                ':unit_cost' => $unitPrice,
                ':movement_date' => $receiptDate,
                ':org_id' => $orgId
            ]);
        }

        // 4. Create stock_batch (if batch number provided)
        if ($batchNumber) {
            $stmt = $conn->prepare("
                INSERT INTO stock_batch (
                    product_id, warehouse_id, batch_number, 
                    quantity, expiry_date, received_date, org_id
                ) VALUES (
                    :product_id, :warehouse_id, :batch_number, 
                    :qty, :expiry_date, :received_date, :org_id
                )
            ");

            $stmt->execute([
                ':product_id' => $productId,
                ':warehouse_id' => $warehouseId,
                ':batch_number' => $batchNumber,
                ':qty' => $receivedQty,
                ':expiry_date' => $expiryDate,
                ':received_date' => $receiptDate,
                ':org_id' => $orgId
            ]);
        }

        // 5. Update tblproduct quantity (for backward compatibility)
        $stmt = $conn->prepare("
            UPDATE tblproduct 
            SET quantity = COALESCE(quantity, 0) + :qty 
            WHERE id = :product_id
        ");
        $stmt->execute([
            ':qty' => $receivedQty,
            ':product_id' => $productId
        ]);
    }

    // Update PO status
    $stmt = $conn->prepare("
        SELECT 
            COUNT(*) as total_items,
            SUM(CASE WHEN poi.quantity <= COALESCE(gri.total_received, 0) THEN 1 ELSE 0 END) as fully_received
        FROM purchase_order_item poi
        LEFT JOIN (
            SELECT po_item_id, SUM(quantity_received) as total_received
            FROM goods_receipt_item
            GROUP BY po_item_id
        ) gri ON poi.id = gri.po_item_id
        WHERE poi.po_id = :po_id
    ");
    $stmt->execute([':po_id' => $poId]);
    $poStats = $stmt->fetch(PDO::FETCH_ASSOC);

    $poStatus = 'Sent';
    if ($poStats['fully_received'] == $poStats['total_items']) {
        $poStatus = 'Received';
    } else if ($poStats['fully_received'] > 0) {
        $poStatus = 'Partially Received';
    }

    $stmt = $conn->prepare("UPDATE purchase_order SET status = :status WHERE id = :po_id");
    $stmt->execute([':status' => $poStatus, ':po_id' => $poId]);

    // Audit log
    $stmt = $conn->prepare("
        INSERT INTO inventory_audit_log (
            action_type, table_name, record_id, user_id, org_id, details
        ) VALUES (
            'CREATE', 'goods_receipt', :grn_id, :user_id, :org_id, :details
        )
    ");
    $stmt->execute([
        ':grn_id' => $grnId,
        ':user_id' => $userId,
        ':org_id' => $orgId,
        ':details' => "Goods received - GRN #$nextNumber, " . count($items) . " items, Total: K" . number_format($totalAmount, 2)
    ]);

    $conn->commit();

    $response = [
        'success' => true,
        'grn_number' => $nextNumber,
        'grn_id' => $grnId,
        'po_status' => $poStatus,
        'items_processed' => count($items),
        'total_amount' => $totalAmount,
        'message' => 'Goods receipt created successfully. Stock updated!'
    ];

} catch (Exception $e) {
    if ($conn->inTransaction()) $conn->rollBack();
    $response['message'] = $e->getMessage();
    error_log("Create GRN Error: " . $e->getMessage());
}

$conn = null;
echo json_encode($response);
?>