<?php
/**
 * Update Warehouse Backend
 * File: xd-modules/admin-modules/inventory/backend/update_warehouse.php
 * Purpose: Update existing warehouse details in the warehouse table
 */

session_start();
header('Content-Type: application/json');
require_once '../../../xd-assets/backend/config/dbconfig.php';

$response = [
    'success' => false,
    'message' => '',
    'warehouse_id' => null
];

try {
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    // --- Validate required fields ---
    if (empty($data['id'])) throw new Exception('Warehouse ID is required.');
    if (empty($data['warehouse_code'])) throw new Exception('Warehouse code is required.');
    if (empty($data['warehouse_name'])) throw new Exception('Warehouse name is required.');
    if (empty($data['warehouse_type'])) throw new Exception('Warehouse type is required.');
    if (empty($data['lat']) || empty($data['lng'])) throw new Exception('GPS coordinates are required. Please select location on map.');

    $warehouse_id   = intval($data['id']);
    $warehouse_code = trim($data['warehouse_code']);
    $warehouse_name = trim($data['warehouse_name']);
    $warehouse_type = trim($data['warehouse_type']);
    $address        = !empty($data['address']) ? trim($data['address']) : null;
    $city           = !empty($data['city']) ? trim($data['city']) : null;
    $state          = !empty($data['state']) ? trim($data['state']) : null;
    $country        = !empty($data['country']) ? trim($data['country']) : null;
    $postal_code    = !empty($data['postal_code']) ? trim($data['postal_code']) : null;
    $lat            = floatval($data['lat']);
    $lng            = floatval($data['lng']);
    $phone          = !empty($data['phone']) ? trim($data['phone']) : null;
    $email          = !empty($data['email']) ? trim($data['email']) : null;
    $manager_id     = !empty($data['manager_id']) ? intval($data['manager_id']) : null;
    $capacity       = !empty($data['capacity']) ? floatval($data['capacity']) : null;
    $is_active      = isset($data['is_active']) ? intval($data['is_active']) : 1;

    $org_id    = !empty($data['org_id']) ? trim($data['org_id']) : ($_SESSION['org_id'] ?? null);
    $updated_by = !empty($data['updated_by']) ? intval($data['updated_by']) : ($_SESSION['userid'] ?? null);

    // --- Check if warehouse exists ---
    $check_sql = "SELECT * FROM warehouse WHERE id = ? AND org_id = ?";
    $check_stmt = $conn->prepare($check_sql);
    $check_stmt->execute([$warehouse_id, $org_id]);
    $existingWarehouse = $check_stmt->fetch(PDO::FETCH_ASSOC);

    if (!$existingWarehouse) {
        throw new Exception('Warehouse not found or not associated with this organization.');
    }

    // --- Prevent duplicate warehouse codes ---
    $code_check_sql = "SELECT id FROM warehouse WHERE warehouse_code = ? AND org_id = ? AND id != ?";
    $code_check_stmt = $conn->prepare($code_check_sql);
    $code_check_stmt->execute([$warehouse_code, $org_id, $warehouse_id]);
    if ($code_check_stmt->fetch()) {
        throw new Exception('Warehouse code already exists. Please use a different code.');
    }

    // --- Update warehouse record ---
    $sql = "UPDATE warehouse SET 
                warehouse_code = ?,
                warehouse_name = ?,
                warehouse_type = ?,
                address = ?,
                city = ?,
                state = ?,
                country = ?,
                postal_code = ?,
                lat = ?,
                lng = ?,
                phone = ?,
                email = ?,
                manager_id = ?,
                capacity = ?,
                is_active = ?,
                created_by = ?,
                updated_at = NOW()
            WHERE id = ? AND org_id = ?";

    $stmt = $conn->prepare($sql);
    $stmt->execute([
        $warehouse_code,
        $warehouse_name,
        $warehouse_type,
        $address,
        $city,
        $state,
        $country,
        $postal_code,
        $lat,
        $lng,
        $phone,
        $email,
        $manager_id,
        $capacity,
        $is_active,
        $updated_by,
        $warehouse_id,
        $org_id
    ]);

    // --- Log update ---
    logWarehouseActivity(
        $conn,
        $warehouse_id,
        'update',
        $updated_by,
        "Updated warehouse: $warehouse_name",
        $existingWarehouse,
        $data
    );

    $response['success'] = true;
    $response['message'] = 'Warehouse updated successfully';
    $response['warehouse_id'] = $warehouse_id;

} catch (Exception $e) {
    $response['success'] = false;
    $response['message'] = $e->getMessage();
    error_log("Warehouse Update Error: " . $e->getMessage());
}

$conn = null;
echo json_encode($response);

/**
 * Log warehouse update activity
 */
function logWarehouseActivity($conn, $warehouse_id, $action, $user_id, $description, $old_value = null, $new_value = null) {
    try {
        $sql = "INSERT INTO inventory_audit_log (
            entity_type,
            entity_id,
            action,
            old_value,
            new_value,
            user_id,
            ip_address,
            user_agent,
            reason,
            created_at
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";

        $stmt = $conn->prepare($sql);

        $entity_type = 'warehouse';
        $ip_address  = $_SERVER['REMOTE_ADDR'] ?? null;
        $user_agent  = $_SERVER['HTTP_USER_AGENT'] ?? null;

        $stmt->execute([
            $entity_type,
            $warehouse_id,
            $action,
            json_encode($old_value),
            json_encode($new_value),
            $user_id,
            $ip_address,
            $user_agent,
            $description
        ]);
    } catch (Exception $e) {
        error_log("Audit Log Error: " . $e->getMessage());
    }
}
?>
