<?php
/**
 * List/Get All Warehouses (Dynamic Filters)
 * File: xd-modules/admin-modules/inventory/backend/list_warehouses.php
 */

session_start();
header('Content-Type: application/json');
require_once '../../../xd-assets/backend/config/dbconfig.php';

$response = [
    'success' => false,
    'message' => '',
    'details' => []
];

try {
    $org_id = $_SESSION['orgid'] ?? null;
    if (!$org_id) throw new Exception('Organization ID not found in session');

    // Optional filters from GET
    $filters = [
        'is_active' => isset($_GET['is_active']) ? intval($_GET['is_active']) : null,
        'warehouse_type' => isset($_GET['warehouse_type']) ? trim($_GET['warehouse_type']) : null,
        'search' => isset($_GET['search']) ? trim($_GET['search']) : null
    ];

    // Base SQL
    $sql = "SELECT 
                w.id, w.warehouse_code, w.warehouse_name, w.warehouse_type,
                w.address, w.city, w.state, w.country, w.postal_code,
                w.lat, w.lng, w.phone, w.email, w.manager_id, w.capacity,
                w.current_utilization, w.is_active, w.created_at, w.updated_at,
                names AS manager_name,
                (SELECT COUNT(*) FROM stock_balance WHERE warehouse_id = w.id) AS product_count,
                (SELECT SUM(quantity_on_hand * average_cost) FROM stock_balance WHERE warehouse_id = w.id) AS total_stock_value
            FROM warehouse w
            LEFT JOIN users u ON w.manager_id = u.id
            WHERE w.org_id = :org_id";

    $params = ['org_id' => $org_id];

    // Dynamically add filters
    if ($filters['is_active'] !== null) {
        $sql .= " AND w.is_active = :is_active";
        $params['is_active'] = $filters['is_active'];
    }

    if (!empty($filters['warehouse_type'])) {
        $sql .= " AND w.warehouse_type = :warehouse_type";
        $params['warehouse_type'] = $filters['warehouse_type'];
    }

    if (!empty($filters['search'])) {
        $sql .= " AND (w.warehouse_name LIKE :search OR w.warehouse_code LIKE :search OR w.city LIKE :search OR w.address LIKE :search)";
        $params['search'] = '%' . $filters['search'] . '%';
    }

    $sql .= " ORDER BY w.created_at DESC";

    // Prepare and execute
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);

    $warehouses = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $warehouses[] = [
            'id' => $row['id'],
            'warehouse_code' => $row['warehouse_code'],
            'warehouse_name' => $row['warehouse_name'],
            'warehouse_type' => $row['warehouse_type'],
            'warehouse_type_label' => ucfirst($row['warehouse_type']),
            'address' => $row['address'],
            'city' => $row['city'],
            'state' => $row['state'],
            'country' => $row['country'],
            'postal_code' => $row['postal_code'],
            'full_address' => implode(', ', array_filter([$row['address'], $row['city'], $row['state'], $row['country']])),
            'lat' => floatval($row['lat']),
            'lng' => floatval($row['lng']),
            'phone' => $row['phone'],
            'email' => $row['email'],
            'manager_id' => $row['manager_id'],
            'manager_name' => $row['manager_name'],
            'capacity' => floatval($row['capacity']),
            'current_utilization' => floatval($row['current_utilization']),
            'utilization_percentage' => $row['capacity'] > 0 
                ? round(($row['current_utilization'] / $row['capacity']) * 100, 2) 
                : 0,
            'is_active' => intval($row['is_active']),
            'status' => intval($row['is_active']) ? 'Active' : 'Inactive',
            'status_class' => intval($row['is_active']) ? 'success' : 'danger',
            'product_count' => intval($row['product_count']),
            'total_stock_value' => floatval($row['total_stock_value']),
            'created_at' => $row['created_at'],
            'updated_at' => $row['updated_at']
        ];
    }

    $response['success'] = true;
    $response['message'] = 'Warehouses fetched successfully';
    $response['details'] = $warehouses;
    $response['count'] = count($warehouses);

} catch (Exception $e) {
    $response['success'] = false;
    $response['message'] = $e->getMessage();
    error_log("List Warehouses Error: " . $e->getMessage());
}

$conn = null;
echo json_encode($response);
