<?php
/**
 * Create Warehouse Backend
 * File: xd-modules/admin-modules/inventory/backend/create_warehouse.php
 * Purpose: Insert new warehouse into warehouse table
 */

// Start session and include database connection
session_start();
header('Content-Type: application/json');

// Include your PDO database connection
require_once '../../../xd-assets/backend/config/dbconfig.php';

// Initialize response array
$response = [
    'success' => false,
    'message' => '',
    'warehouse_id' => null
];

try {
    // Get JSON input
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    // Validate required fields
    if (empty($data['warehouse_code'])) throw new Exception('Warehouse code is required');
    if (empty($data['warehouse_name'])) throw new Exception('Warehouse name is required');
    if (empty($data['warehouse_type'])) throw new Exception('Warehouse type is required');
    if (empty($data['lat']) || empty($data['lng'])) throw new Exception('GPS coordinates are required. Please select location on map.');

    // Sanitize and prepare data
    $warehouse_code = trim($data['warehouse_code']);
    $warehouse_name = trim($data['warehouse_name']);
    $warehouse_type = trim($data['warehouse_type']);
    $address = !empty($data['address']) ? trim($data['address']) : null;
    $city = !empty($data['city']) ? trim($data['city']) : null;
    $state = !empty($data['state']) ? trim($data['state']) : null;
    $country = !empty($data['country']) ? trim($data['country']) : null;
    $postal_code = !empty($data['postal_code']) ? trim($data['postal_code']) : null;
    $lat = floatval($data['lat']);
    $lng = floatval($data['lng']);
    $phone = !empty($data['phone']) ? trim($data['phone']) : null;
    $email = !empty($data['email']) ? trim($data['email']) : null;
    $manager_id = !empty($data['manager_id']) ? intval($data['manager_id']) : null;
    $capacity = !empty($data['capacity']) ? floatval($data['capacity']) : null;
    $is_active = isset($data['is_active']) ? intval($data['is_active']) : 1;

    // Get org_id and user_id from session or data
    $org_id = ($_SESSION['orgid'] ?? null);
    $created_by = !empty($data['created_by']) ? intval($data['created_by']) : ($_SESSION['userid'] ?? null);

    // Check if warehouse code already exists
    $check_sql = "SELECT id FROM warehouse WHERE warehouse_code = ? AND org_id = ?";
    $check_stmt = $conn->prepare($check_sql);
    $check_stmt->execute([$warehouse_code, $org_id]);
    $check_result = $check_stmt->fetch(PDO::FETCH_ASSOC);

    if ($check_result) {
        throw new Exception('Warehouse code already exists. Please use a different code.');
    }

    // Prepare SQL INSERT statement
    $sql = "INSERT INTO warehouse (
        warehouse_code,
        warehouse_name,
        warehouse_type,
        address,
        city,
        state,
        country,
        postal_code,
        lat,
        lng,
        phone,
        email,
        manager_id,
        capacity,
        current_utilization,
        is_active,
        org_id,
        created_by,
        created_at
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";

    $stmt = $conn->prepare($sql);

    // Default current utilization
    $current_utilization = 0.00;

    // Execute insert
    $stmt->execute([
        $warehouse_code,
        $warehouse_name,
        $warehouse_type,
        $address,
        $city,
        $state,
        $country,
        $postal_code,
        $lat,
        $lng,
        $phone,
        $email,
        $manager_id,
        $capacity,
        $current_utilization,
        $is_active,
        $org_id,
        $created_by
    ]);

    $warehouse_id = $conn->lastInsertId();

    // Optional: log warehouse creation
    logWarehouseActivity($conn, $warehouse_id, 'create', $created_by, "Created warehouse: $warehouse_name");

    // Success response
    $response['success'] = true;
    $response['message'] = 'Warehouse created successfully';
    $response['warehouse_id'] = $warehouse_id;
    $response['warehouse_code'] = $warehouse_code;

} catch (Exception $e) {
    $response['success'] = false;
    $response['message'] = $e->getMessage();
    error_log("Warehouse Creation Error: " . $e->getMessage());
}

// Close PDO connection
$conn = null;

// Return JSON response
echo json_encode($response);

/**
 * Log warehouse activity to audit log
 */
function logWarehouseActivity($conn, $warehouse_id, $action, $user_id, $description) {
    try {
        $sql = "INSERT INTO inventory_audit_log (
            entity_type,
            entity_id,
            action,
            old_value,
            new_value,
            user_id,
            ip_address,
            user_agent,
            reason,
            created_at
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";

        $stmt = $conn->prepare($sql);

        $entity_type = 'warehouse';
        $old_value = null;
        $new_value = json_encode(['warehouse_id' => $warehouse_id]);
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? null;
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? null;

        $stmt->execute([
            $entity_type,
            $warehouse_id,
            $action,
            $old_value,
            $new_value,
            $user_id,
            $ip_address,
            $user_agent,
            $description
        ]);
    } catch (Exception $e) {
        error_log("Audit Log Error: " . $e->getMessage());
    }
}
?>
