class ManualBilling {
    constructor() {
        this.products = [];
        this.selectedProduct = '';
        this.billingHistory = [];
        this.loading = false;
        this.currentRunId = null;
    }

    initialize() {
        this.createBillingContainer();
        this.loadProducts();
        this.loadBillingHistory();
        this.attachEventListeners();
    }

    createBillingContainer() {
        const mainBody = document.getElementById('main-body');
        if (!mainBody) {
            console.error('main-body element not found');
            return;
        }

        // Clear existing content and create billing container
        mainBody.innerHTML = '';
        
        const billingContainer = document.createElement('div');
        billingContainer.id = 'billing-container';
        billingContainer.className = 'billing-container';
        billingContainer.innerHTML = this.getBillingHTML();
        
        mainBody.appendChild(billingContainer);
    }

    getBillingHTML() {
        return `
        <div class="billing-content">
            <!-- Header Section -->
            <div class="billing-header">
                <div class="billing-title-section">
                    <i class="fas fa-file-invoice-dollar billing-title-icon"></i>
                    <h2 class="billing-title">Automatic Billing</h2>
                </div>
                <div class="billing-actions">
                     <button id="billing-schedule-btn" class="billing-btn billing-btn-outline">
                        <i class="fas fa-clock"></i> Schedule Billing
                    </button>
                    
                </div>
            </div>

            <!-- Control Panel -->
            <div class="billing-control-panel">
                <div class="billing-card">
                    <div class="billing-card-header">
                        <h3 class="billing-card-title">
                            <i class="fas fa-play-circle"></i> Run Manual Billing
                        </h3>
                    </div>
                    <div class="billing-card-body">
                        <div class="billing-form-group">
                            <label for="billing-product-select" class="billing-form-label">
                                Select Product to Bill
                            </label>
                            <select id="billing-product-select" class="billing-form-select">
                                <option value="">Loading products...</option>
                            </select>
                        </div>
                        
                        <div class="billing-product-info" id="billing-product-info" style="display: none;">
                            <div class="billing-info-grid">
                                <div class="billing-info-item">
                                    <span class="billing-info-label">Product Code:</span>
                                    <span id="billing-product-code" class="billing-info-value"></span>
                                </div>
                                <div class="billing-info-item">
                                    <span class="billing-info-label">Price:</span>
                                    <span id="billing-product-price" class="billing-info-value"></span>
                                </div>
                                <div class="billing-info-item">
                                    <span class="billing-info-label">Billing Interval:</span>
                                    <span id="billing-product-interval" class="billing-info-value"></span>
                                </div>
                            </div>
                        </div>

                        <div class="billing-action-buttons">
                            <button id="billing-run-btn" class="billing-btn billing-btn-primary" disabled>
                                <i class="fas fa-play"></i> Run Billing Now
                            </button>
                            
                        </div>
                    </div>
                </div>
            </div>

            <!-- Loading State -->
            <div id="billing-loading" class="billing-loading" style="display: none;">
                <div class="billing-loading-content">
                    <div class="billing-spinner"></div>
                    <p class="billing-loading-text">Processing billing run...</p>
                </div>
            </div>

            <!-- Results Section -->
            <div id="billing-results" class="billing-results" style="display: none;">
                <div class="billing-card">
                    <div class="billing-card-header">
                        <h3 class="billing-card-title">
                            <i class="fas fa-chart-bar"></i> Billing Results
                        </h3>
                    </div>
                    <div class="billing-card-body">
                        <div id="billing-results-content"></div>
                    </div>
                </div>
            </div>

            <!-- Billing History -->
            <div class="billing-history-section">
                <div class="billing-card">
                    <div class="billing-card-header">
                        <h3 class="billing-card-title">
                            <i class="fas fa-history"></i> Billing History
                        </h3>
                    </div>
                    <div class="billing-card-body">
                        <div id="billing-history-content">
                            <div class="billing-empty-state">
                                <i class="fas fa-history billing-empty-icon"></i>
                                <p>No billing history available</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Billing Details Modal -->
        <div id="billing-details-modal" class="billing-modal">
            <div class="billing-modal-content">
                <div class="billing-modal-header">
                    <h3 id="billing-modal-title">Billing Run Details</h3>
                    <span class="billing-modal-close">&times;</span>
                </div>
                <div class="billing-modal-body">
                    <div id="billing-modal-content"></div>
                </div>
            </div>
        </div>
        `;
    }

    attachEventListeners() {
        // Product selection
        const productSelect = document.getElementById('billing-product-select');
        if (productSelect) {
            productSelect.addEventListener('change', (e) => {
                this.handleProductSelect(e.target.value);
            });
        }

        // Run billing button
        const runBtn = document.getElementById('billing-run-btn');
        if (runBtn) {
            runBtn.addEventListener('click', () => {
                this.runManualBilling();
            });
        }

       

        // Modal close
        const modal = document.getElementById('billing-details-modal');
        const closeBtn = modal.querySelector('.billing-modal-close');
        closeBtn.addEventListener('click', () => {
            modal.style.display = 'none';
        });

        window.addEventListener('click', (e) => {
            if (e.target === modal) {
                modal.style.display = 'none';
            }
        });
    }

    async loadProducts() {
        try {
            const response = await fetch('xd-modules/billing-module/backend/get-billing-products.php');
            const data = await response.json();
            
            if (data.success) {
                this.products = data.products;
                this.populateProductSelect();
            } else {
                this.showError('Failed to load products');
            }
        } catch (error) {
            console.error('Error loading products:', error);
            this.showError('Error loading products');
        }
    }

    populateProductSelect() {
        const select = document.getElementById('billing-product-select');
        select.innerHTML = '<option value="">Select a product...</option>';
        
        this.products.forEach(product => {
            const option = document.createElement('option');
            option.value = product.id;
            option.textContent = `${product.name} (${product.product_code}) - $${product.price}`;
            option.setAttribute('data-product', JSON.stringify(product));
            select.appendChild(option);
        });
    }

    handleProductSelect(productId) {
        const select = document.getElementById('billing-product-select');
        const selectedOption = select.options[select.selectedIndex];
        const runBtn = document.getElementById('billing-run-btn'); 
        const productInfo = document.getElementById('billing-product-info');

        if (productId && selectedOption) {
            const product = JSON.parse(selectedOption.getAttribute('data-product'));
            this.selectedProduct = product;
            
            // Update product info
            document.getElementById('billing-product-code').textContent = product.product_code;
            document.getElementById('billing-product-price').textContent = `$${product.price}`;
            document.getElementById('billing-product-interval').textContent = product.charge_rate || 'Once off';
            
            productInfo.style.display = 'block';
            runBtn.disabled = false;
            scheduleBtn.disabled = false;
        } else {
            productInfo.style.display = 'none';
            runBtn.disabled = true;
            scheduleBtn.disabled = true;
            this.selectedProduct = '';
        }
    }

    async runManualBilling() {
        if (!this.selectedProduct) {
            this.showError('Please select a product first');
            return;
        }

        this.showLoading(true);
        
        try {
            const formData = new FormData();
            formData.append('product_id', this.selectedProduct.id);
            formData.append('action', 'run_manual_billing');

            const response = await fetch('xd-modules/billing-module/backend/run-manual-billing.php', {
                method: 'POST',
                body: formData
            });

            const data = await response.json();
            
            if (data.success) {
                this.showBillingResults(data);
                this.loadBillingHistory(); // Refresh history
            } else {
                this.showError(data.message || 'Billing run failed');
            }
        } catch (error) {
            console.error('Error running billing:', error);
            this.showError('Error running billing');
        } finally {
            this.showLoading(false);
        }
    }

   showBillingResults(data) {
    const resultsSection = document.getElementById('billing-results');
    const resultsContent = document.getElementById('billing-results-content');
    
    let runIdDisplay = data.run_id || 'N/A';
    if (!data.run_id) {
        runIdDisplay = 'Not recorded (table missing)';
    }
    
    resultsContent.innerHTML = `
        <div class="billing-results-success">
            <div class="billing-results-header">
                <i class="fas fa-check-circle billing-success-icon"></i>
                <h4>Billing Run Completed Successfully</h4>
            </div>
            <div class="billing-results-stats">
                <div class="billing-stat-item">
                    <span class="billing-stat-label">Total Clients Billed:</span>
                    <span class="billing-stat-value">${data.total_clients_billed || 0}</span>
                </div>
                <div class="billing-stat-item">
                    <span class="billing-stat-label">Invoices Generated:</span>
                    <span class="billing-stat-value">${data.total_invoices || 0}</span>
                </div>
                <div class="billing-stat-item">
                    <span class="billing-stat-label">Total Amount:</span>
                    <span class="billing-stat-value">$${data.total_amount || 0}</span>
                </div>
                <div class="billing-stat-item">
                    <span class="billing-stat-label">Run ID:</span>
                    <span class="billing-stat-value">${runIdDisplay}</span>
                </div>
            </div>
            ${data.failed_clients > 0 ? `
                <div class="billing-results-warning">
                    <i class="fas fa-exclamation-triangle"></i>
                    ${data.failed_clients} clients failed to bill
                </div>
            ` : ''}
            <div class="billing-results-message">
                <p><strong>Product:</strong> ${this.selectedProduct.name}</p>
                <p><strong>Message:</strong> ${data.message}</p>
            </div>
        </div>
    `;
    
    resultsSection.style.display = 'block';
    this.currentRunId = data.run_id;
}

    async loadBillingHistory() {
        try {
            const response = await fetch('xd-modules/billing-module/backend/get-billing-history.php');
            const data = await response.json();
            
            if (data.success) {
                this.billingHistory = data.history;
                this.renderBillingHistory();
            }
        } catch (error) {
            console.error('Error loading billing history:', error);
        }
    }

    renderBillingHistory() {
        const historyContent = document.getElementById('billing-history-content');
        
        if (this.billingHistory.length === 0) {
            historyContent.innerHTML = `
                <div class="billing-empty-state">
                    <i class="fas fa-history billing-empty-icon"></i>
                    <p>No billing history available</p>
                </div>
            `;
            return;
        }

        historyContent.innerHTML = `
            <div class="billing-history-table">
                <table class="billing-table">
                    <thead>
                        <tr>
                            <th>Run Date</th>
                            <th>Product</th>
                            <th>Clients Billed</th>
                            <th>Invoices</th>
                            <th>Total Amount</th>
                            <th>Status</th>
                            <th>Run By</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${this.billingHistory.map(run => `
                            <tr>
                                <td>${this.formatDate(run.run_date)}</td>
                                <td>${run.product_name}</td>
                                <td>${run.total_clients_billed}</td>
                                <td>${run.total_invoices_generated}</td>
                                <td>$${run.total_amount_billed}</td>
                                <td>
                                    <span class="billing-status billing-status-${run.status}">
                                        ${run.status}
                                    </span>
                                </td>
                                <td>
                                <span class="billing-status billing-run-bys-${run.run_by_user_name}">
                                        ${run.run_by_user_name}
                                    </span>
                                </td>
                                <td>
                                    <button class="billing-btn-icon" onclick="manualBilling.viewRunDetails(${run.id})">
                                        <i class="fontx fx-more-vertical" style="float:right;font-size:2em;"></i>
                                    </button>
                                </td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            </div>
        `;
    }

    viewRunDetails(runId) {
        // Implement view details functionality
        console.log('View run details:', runId);
        // You can fetch and display detailed run information in the modal
    }

    showLoading(show) {
        const loadingElement = document.getElementById('billing-loading');
        const runBtn = document.getElementById('billing-run-btn');
        
        if (show) {
            loadingElement.style.display = 'block';
            runBtn.disabled = true;
            runBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
        } else {
            loadingElement.style.display = 'none';
            runBtn.disabled = false;
            runBtn.innerHTML = '<i class="fas fa-play"></i> Run Billing Now';
        }
    }

    

    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `billing-notification billing-notification-${type}`;
        notification.innerHTML = `
            <div class="billing-notification-content">
                <i class="fas fa-${type === 'error' ? 'exclamation-triangle' : 'check-circle'}"></i>
                <span>${message}</span>
            </div>
        `;

        // Add to container
        const container = document.getElementById('billing-container');
        container.appendChild(notification);

        // Remove after 5 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 5000);
    }

    formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
    }
}

// Initialize manual billing when DOM is loaded
/*
 $(document).on('click', '.billingDashBoard', function(event) {
     window.manualBilling = new ManualBilling();
    window.manualBilling.initialize();
});
*/