 class CustomerStatements {
  constructor() {
    this.currentView = 'statements';
    this.statements = [];
    this.clients = [];
    this.isInitialized = false;
  }

  // Initialize standalone
  init() {
    if (this.isInitialized) {
      console.log('Customer Statements already initialized');
      return;
    }
    
    try {
      this.createStatementsContainer();
      this.createNavigationButtons(); // Create nav buttons like your example
      this.attachEventListeners();
      this.showStatementsList(); // Show default view
      this.isInitialized = true;
      
      console.log('Customer Statements initialized successfully');
    } catch (error) {
      console.error('Error initializing Customer Statements:', error);
    }
  }

  // Create navigation buttons like your example
  createNavigationButtons() {
    // Clear existing navigation
    $('#sub_top_nav').empty();
    
    // Add navigation buttons like your revenue report example
    $('#sub_top_nav').append(`
      <div class="cyber-horizontal-scroll hideOnPrint">
        <button id="statements-list-btn" class="statements-nav-btn statements-nav-active">
          <i class="fontx fx-list"></i> Statements List
        </button>
        <button id="statements-generate-btn" class="statements-nav-btn">
          <i class="fontx fx-plus"></i> Generate Statement
        </button>
      </div>
    `);
  }

  // Create main container
  createStatementsContainer() {
    // Remove existing container if any
    const existingContainer = document.getElementById('customer-statements-container');
    if (existingContainer) {
      existingContainer.remove();
    }

    // Create new container
    const container = document.createElement('div');
    container.id = 'customer-statements-container';
    container.className = 'customer-statements-container';
    container.innerHTML = this.getStatementsHTML();
    
    // Insert into main body
    const mainBody = document.getElementById('main-body');
    if (mainBody) {
      mainBody.innerHTML = '';
      mainBody.appendChild(container);
    } else {
      // Fallback if main-body doesn't exist
      const appContainer = document.querySelector('.app-container') || document.body;
      appContainer.appendChild(container);
    }
  }

  // Main HTML template
  getStatementsHTML() {
    return `
    <div class="customer-statements-content">
      <!-- Content Area -->
      <div class="statements-card">
        <div class="statements-card-body">
          <div id="statements-content-area">
            <div class="statements-loading-state">
              <div class="statements-spinner"></div>
              <p>Loading customer statements...</p>
            </div>
          </div>
        </div>
      </div>

      <!-- Loading Overlay -->
      <div id="statements-loading-overlay" class="statements-loading-overlay" style="display: none;">
        <div class="statements-loading-content">
          <div class="statements-spinner-large"></div>
          <p class="statements-loading-text">Processing...</p>
        </div>
      </div>
    </div>`;
  }

  // Attach event listeners
  attachEventListeners() {
    // Use setTimeout to ensure DOM is ready
    setTimeout(() => {
      const listBtn = document.getElementById('statements-list-btn');
      const generateBtn = document.getElementById('statements-generate-btn');
      
      if (listBtn) {
        listBtn.addEventListener('click', () => {
          this.showStatementsList();
          this.setActiveNav('statements-list-btn');
        });
      }
      
      if (generateBtn) {
        generateBtn.addEventListener('click', () => {
          this.showGenerateStatement();
          this.setActiveNav('statements-generate-btn');
        });
      }
    }, 100);
  }

  // Set active navigation button
  setActiveNav(activeId) {
    const buttons = document.querySelectorAll('.statements-nav-btn');
    buttons.forEach(btn => {
      btn.classList.remove('statements-nav-active');
    });
    
    const activeBtn = document.getElementById(activeId);
    if (activeBtn) {
      activeBtn.classList.add('statements-nav-active');
    }
  }

  // Show statements list view
  async showStatementsList() {
    const contentArea = document.getElementById('statements-content-area');
    if (!contentArea) return;
    
    contentArea.innerHTML = `
      <div class="statements-list-view">
        <div class="statements-view-header">
          <h3><i class="fontx fx-file-text"></i> Customer Statements</h3>
          <div class="statements-view-actions">
            <button class="statements-btn statements-btn-primary" onclick="window.customerStatementsInstance.refreshStatements()">
              <i class="fontx fx-refresh-cw"></i> Refresh
            </button>
            <button class="statements-btn statements-btn-success" onclick="window.customerStatementsInstance.showGenerateStatement()">
              <i class="fontx fx-plus"></i> New Statement
            </button>
          </div>
        </div>
        
        <div class="statements-table-container">
          <div id="statements-list-loading" class="statements-loading-state">
            <div class="statements-spinner"></div>
            <p>Loading statements...</p>
          </div>
          <div id="statements-table-content">
            <div class="statements-empty-state">
              <i class="fontx fx-file-text"></i>
              <h4>No Statements Found</h4>
              <p>Get started by generating your first customer statement</p>
              <button class="statements-btn statements-btn-primary" onclick="window.customerStatementsInstance.showGenerateStatement()">
                Generate Statement
              </button>
            </div>
          </div>
        </div>
      </div>`;

    // Load statements after a short delay to show the UI first
    setTimeout(() => {
      this.loadStatements();
    }, 500);
  }

  // Show generate statement view
  showGenerateStatement() {
    const contentArea = document.getElementById('statements-content-area');
    if (!contentArea) return;
    
    contentArea.innerHTML = `
      <div class="generate-statement-view">
        <div class="generate-header">
          <h3><i class="fontx fx-file-plus"></i> Generate New Statement</h3>
          <p>Create professional statements for your customers</p>
        </div>
        
        <div class="generate-form">
          <div class="form-group">
            <label class="form-label">Select Client</label>
            <select id="generate-client-select" class="statements-select">
              <option value="">Loading clients...</option>
            </select>
          </div>
          
          <div class="form-row">
            <div class="form-group">
              <label class="form-label">Statement Period Start</label>
              <input type="date" id="generate-period-start" class="statements-input">
            </div>
            <div class="form-group">
              <label class="form-label">Statement Period End</label>
              <input type="date" id="generate-period-end" class="statements-input">
            </div>
          </div>
          
          <div class="form-group">
            <label class="form-label">Due Date</label>
            <input type="date" id="generate-due-date" class="statements-input">
          </div>
          
          <div class="form-group">
            <label class="form-label">Additional Notes (Optional)</label>
            <textarea id="generate-notes" class="statements-textarea" 
                      placeholder="Add any special notes for this statement..." rows="3"></textarea>
          </div>
          
          <div class="form-actions">
            <button type="button" class="statements-btn statements-btn-outline" onclick="window.customerStatementsInstance.showStatementsList()">
              <i class="fontx fx-arrow-left"></i> Back to List
            </button>
            <button type="button" class="statements-btn statements-btn-primary" onclick="window.customerStatementsInstance.generateStatement()">
              <i class="fontx fx-file-text"></i> Generate Statement
            </button>
          </div>
        </div>
      </div>`;

    this.loadClientsForGeneration();
    this.setDefaultDates();
  }

  // Load clients for statement generation using your PHP endpoint
  async loadClientsForGeneration() {
    try {
      const select = document.getElementById('generate-client-select');
      if (!select) return;
      
      select.innerHTML = '<option value="">Loading clients...</option>';
      
      // Use your PHP endpoint to get clients
      const clients = await this.fetchClientsFromAPI();
      
      if (clients && clients.length > 0) {
        select.innerHTML = '<option value="">Select a client...</option>';
        clients.forEach(client => {
          const option = document.createElement('option');
          option.value = client.id;
          option.textContent = `${client.names} (${client.email || client.phone || 'No contact'})`;
          select.appendChild(option);
        });
      } else {
        select.innerHTML = '<option value="">No clients found</option>';
        this.showNotification('No clients found in the system', 'warning');
      }
      
    } catch (error) {
      console.error('Error loading clients:', error);
      this.showNotification('Error loading clients: ' + error.message, 'error');
      
      // Fallback to empty select
      const select = document.getElementById('generate-client-select');
      if (select) {
        select.innerHTML = '<option value="">Error loading clients</option>';
      }
    }
  }

  // Fetch clients from your PHP endpoint
  async fetchClientsFromAPI() {
    try {
      const formData = new FormData();
      // You can add filters if needed, or leave empty to get all clients
      // formData.append('searchKey', '');
      // formData.append('status', 'active');
      
      const response = await fetch('xd-modules/admin-modules/backend/allClients.php', {
        method: 'POST',
        body: formData
      });
      
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      
      const data = await response.json();
      
      if (data.error) {
        throw new Error(data.error);
      }
      
      if (data.freelancedetails && Array.isArray(data.freelancedetails)) {
        return data.freelancedetails;
      } else {
        throw new Error('Invalid response format from server');
      }
      
    } catch (error) {
      console.error('Error fetching clients from API:', error);
      
      // Fallback: return mock data if API fails
      return [
        { id: 1, names: 'Bosco Chinyemba', email: 'user0972587141@cyberdesk.live', phone: '0972587141' },
        { id: 2, names: 'Bobliz Solar shop', email: 'user0972566072@cyberdesk.live', phone: '0972566072' },
        { id: 3, names: 'Chitengi Chinyama', email: 'user0960520937@cyberdesk.live', phone: '0960520937' }
      ];
    }
  }

  // Set default dates for generation
  setDefaultDates() {
    const today = new Date();
    const firstDay = new Date(today.getFullYear(), today.getMonth(), 1);
    const lastDay = new Date(today.getFullYear(), today.getMonth() + 1, 0);
    const dueDate = new Date(today.getFullYear(), today.getMonth() + 1, 15);
    
    const startInput = document.getElementById('generate-period-start');
    const endInput = document.getElementById('generate-period-end');
    const dueInput = document.getElementById('generate-due-date');
    
    if (startInput) startInput.valueAsDate = firstDay;
    if (endInput) endInput.valueAsDate = lastDay;
    if (dueInput) dueInput.valueAsDate = dueDate;
  }

  // Generate statement
async generateStatement() {
    const clientId = document.getElementById('generate-client-select')?.value;
    const periodStart = document.getElementById('generate-period-start')?.value;
    const periodEnd = document.getElementById('generate-period-end')?.value;
    const dueDate = document.getElementById('generate-due-date')?.value;
    const notes = document.getElementById('generate-notes')?.value;

    if (!clientId || !periodStart || !periodEnd || !dueDate) {
        this.showNotification('Please fill in all required fields', 'error');
        return;
    }

    try {
        this.showLoading(true);
        
        const response = await this.apiCall('generate_statement', {
            client_id: parseInt(clientId),
            period_start: periodStart,
            period_end: periodEnd,
            due_date: dueDate,
            notes: notes || ''
        });

        if (response.success) {
            this.showNotification(`Statement ${response.statement_number} generated successfully!`, 'success');
            this.showStatementsList();
        } else {
            this.showNotification('Error generating statement: ' + response.message, 'error');
        }
    } catch (error) {
        this.showNotification('Error generating statement: ' + error.message, 'error');
    } finally {
        this.showLoading(false);
    }
}

  // Load statements list
 
async loadStatements() {
    try {
        this.showListLoading(true);
        
        const response = await this.apiCall('get_statements');
        if (response.success) {
            this.statements = response.data || [];
            this.renderStatementsTable();
        } else {
            this.showNotification('Error loading statements: ' + response.message, 'error');
            this.statements = [];
            this.renderStatementsTable();
        }
        
        this.showListLoading(false);
        
    } catch (error) {
        this.showNotification('Error loading statements: ' + error.message, 'error');
        this.showListLoading(false);
    }
}

  // Render statements table
  renderStatementsTable() {
    const container = document.getElementById('statements-table-content');
    if (!container) return;
    
    if (!this.statements.length) {
      container.innerHTML = `
        <div class="statements-empty-state">
          <i class="fontx fx-file-text"></i>
          <h4>No Statements Found</h4>
          <p>Get started by generating your first customer statement</p>
          <button class="statements-btn statements-btn-primary" onclick="window.customerStatementsInstance.showGenerateStatement()">
            Generate Statement
          </button>
        </div>`;
      return;
    }

    // This will be used when we have actual statements
    container.innerHTML = `
      <table class="statements-table">
        <thead>
          <tr>
            <th>Statement #</th>
            <th>Client</th>
            <th>Period</th>
            <th>Amount Due</th>
            <th>Due Date</th>
            <th>Status</th>
            <th>Actions</th>
          </tr>
        </thead>
        <tbody>
          ${this.statements.map(statement => `
            <tr>
              <td>${statement.statement_number || 'N/A'}</td>
              <td>${statement.client_name || 'Unknown Client'}</td>
              <td>${this.formatDate(statement.statement_period_start)} - ${this.formatDate(statement.statement_period_end)}</td>
              <td>$${this.formatAmount(statement.amount_due)}</td>
              <td>${this.formatDate(statement.due_date)}</td>
              <td>
                <span class="statement-badge statement-badge-${statement.status || 'draft'}">
                  ${statement.status || 'draft'}
                </span>
              </td>
              <td>
                <div class="statement-actions">
                  <button class="statements-btn statements-btn-sm" 
                          onclick="window.customerStatementsInstance.viewStatement(${statement.id})">
                    <i class="fontx fx-eye"></i> View
                  </button>
                  <button class="statements-btn statements-btn-sm statements-btn-outline"
                          onclick="window.customerStatementsInstance.downloadStatement(${statement.id})">
                    <i class="fontx fx-download"></i> PDF
                  </button>
                </div>
              </td>
            </tr>
          `).join('')}
        </tbody>
      </table>`;
  }

  // Utility methods
  showLoading(show) {
    const overlay = document.getElementById('statements-loading-overlay');
    if (overlay) {
      overlay.style.display = show ? 'flex' : 'none';
    }
  }

  showListLoading(show) {
    const loading = document.getElementById('statements-list-loading');
    const content = document.getElementById('statements-table-content');
    if (loading && content) {
      loading.style.display = show ? 'flex' : 'none';
      content.style.display = show ? 'none' : 'block';
    }
  }

  formatDate(dateString) {
    if (!dateString) return 'N/A';
    try {
      return new Date(dateString).toLocaleDateString();
    } catch (error) {
      return 'Invalid Date';
    }
  }

  formatAmount(amount) {
    return parseFloat(amount || 0).toFixed(2);
  }

  showNotification(message, type = 'info') {
    // Use your existing notification system
    const notification = document.getElementById('bottom_toast345');
    if (notification) {
      notification.textContent = message;
      notification.className = `bottom_toast345 ${type}`;
      notification.style.display = 'block';
      
      setTimeout(() => {
        notification.style.display = 'none';
      }, 5000);
    } else {
      // Fallback to alert
      alert(`${type.toUpperCase()}: ${message}`);
    }
  }

  // Action methods
 async viewStatement(statementId) {
    try {
        const response = await this.apiCall('get_statement_details', {
            statement_id: statementId
        });

        if (response.success) {
            this.showStatementDetails(response);
        } else {
            this.showNotification('Error loading statement: ' + response.message, 'error');
        }
    } catch (error) {
        this.showNotification('Error loading statement details: ' + error.message, 'error');
    }
}

// Show statement details modal
showStatementDetails(statementData) {
    const { statement, items, settings } = statementData;
    
    const modal = document.createElement('div');
    modal.id = 'statement-details-modal';
    modal.className = 'statements-modal';
    modal.style.display = 'block';
    modal.innerHTML = `
        <div class="statements-modal-content" style="max-width: 900px;">
            <div class="statements-modal-header">
                <h3><i class="fontx fx-file-text"></i> Statement: ${statement.statement_number}</h3>
                <span class="statements-modal-close" onclick="this.parentElement.parentElement.parentElement.remove()">&times;</span>
            </div>
            <div class="statements-modal-body">
                <div class="statement-details-view">
                    <div class="statement-header-info">
                        <div class="company-info">
                            <h4>${settings?.org_name || this.orgname}</h4>
                            <p>${settings?.company_address || 'Lukulu, Zambia'}</p>
                        </div>
                        <div class="statement-meta">
                            <p><strong>Statement #:</strong> ${statement.statement_number}</p>
                            <p><strong>Date:</strong> ${this.formatDate(statement.statement_date)}</p>
                            <p><strong>Due Date:</strong> ${this.formatDate(statement.due_date)}</p>
                        </div>
                    </div>
                    
                    <div class="client-info-section">
                        <h5>Bill To:</h5>
                        <p><strong>${statement.client_name}</strong></p>
                        <p>${statement.client_email || ''}</p>
                        <p>${statement.client_phone || ''}</p>
                        ${statement.client_address ? `<p>${statement.client_address}</p>` : ''}
                    </div>
                    
                    <div class="statement-summary-cards">
                        <div class="summary-card">
                            <span class="summary-label">Opening Balance</span>
                            <span class="summary-value">$${this.formatAmount(statement.opening_balance)}</span>
                        </div>
                        <div class="summary-card">
                            <span class="summary-label">Total Charges</span>
                            <span class="summary-value">$${this.formatAmount(statement.total_charges)}</span>
                        </div>
                        <div class="summary-card">
                            <span class="summary-label">Total Payments</span>
                            <span class="summary-value">$${this.formatAmount(statement.total_payments)}</span>
                        </div>
                        <div class="summary-card highlight">
                            <span class="summary-label">Amount Due</span>
                            <span class="summary-value">$${this.formatAmount(statement.amount_due)}</span>
                        </div>
                    </div>
                    
                    <div class="statement-transactions">
                        <h5>Transaction History</h5>
                        <table class="statements-detail-table">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Type</th>
                                    <th>Document #</th>
                                    <th>Description</th>
                                    <th>Debit</th>
                                    <th>Credit</th>
                                    <th>Balance</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${items.map(item => `
                                    <tr>
                                        <td>${this.formatDate(item.transaction_date)}</td>
                                        <td>
                                            <span class="transaction-type transaction-type-${item.transaction_type}">
                                                ${item.transaction_type}
                                            </span>
                                        </td>
                                        <td>${item.document_number || 'N/A'}</td>
                                        <td>${item.description}</td>
                                        <td>${item.debit_amount > 0 ? '$' + this.formatAmount(item.debit_amount) : '-'}</td>
                                        <td>${item.credit_amount > 0 ? '$' + this.formatAmount(item.credit_amount) : '-'}</td>
                                        <td>$${this.formatAmount(item.balance)}</td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    </div>
                    
                    ${statement.statement_notes ? `
                        <div class="statement-notes-section">
                            <h5>Notes</h5>
                            <p>${statement.statement_notes}</p>
                        </div>
                    ` : ''}
                </div>
            </div>
            <div class="statements-modal-footer">
                <button type="button" class="statements-btn statements-btn-outline" 
                        onclick="document.getElementById('statement-details-modal').remove()">
                    <i class="fontx fx-x"></i> Close
                </button>
                <button type="button" class="statements-btn statements-btn-primary" 
                        onclick="window.customerStatementsInstance.downloadStatement(${statement.id})">
                    <i class="fontx fx-download"></i> Download PDF
                </button>
                ${statement.status === 'draft' ? `
                    <button type="button" class="statements-btn statements-btn-success" 
                            onclick="window.customerStatementsInstance.sendStatement(${statement.id})">
                        <i class="fontx fx-send"></i> Send to Client
                    </button>
                ` : ''}
            </div>
        </div>
    `;

    document.body.appendChild(modal);
}

// API call helper - UPDATED
async apiCall(action, data = {}) {
    try {
        const response = await fetch('xd-modules/billing-module/backend/customer-statements.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: action,
                ...data
            })
        });
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        return await response.json();
    } catch (error) {
        console.error('API Call failed:', error);
        return { 
            success: false, 
            message: 'Network error: ' + error.message 
        };
    }
}

  downloadStatement(statementId) {
    this.showNotification(`Downloading statement ${statementId}`, 'info');
  }

  refreshStatements() {
    this.loadStatements();
  }

  showGenerateModal() {
    this.showGenerateStatement();
    this.setActiveNav('statements-generate-btn');
  }
}

// Global instance and initialization function
window.customerStatementsInstance = new CustomerStatements();

function initializeCustomerStatements() {
  console.log('Initializing Customer Statements...');
  window.customerStatementsInstance.init();
}

// Simple jQuery click handler
$(document).on('click', '.customerStatements', function(event) {
  event.preventDefault();
  console.log('Customer Statements clicked!');
  initializeCustomerStatements();
}); 