// billing_system.js - Dynamic Billing System Interface

class BillingSystemUI {
    constructor() {
        this.currentTab = 'dashboard';
        this.apiBase = 'xd-modules/billing-module/backend/billing_system_api.php'; // Adjust to your API endpoint
        this.mainBody = document.getElementById('main-body');
        this.init();
    }

    init() {
        this.renderUI();
        this.loadDashboardData();
    }

    renderUI() {
        const html = `
            <div id="billing-main-container"> 
                <div class="billing-tabs-container" id="billing-tabs-container"></div>
                
                <div id="billing-dashboard-tab" class="billing-tab-content billing-tab-active"></div>
                <div id="billing-settings-tab" class="billing-tab-content"></div>
                <div id="billing-products-tab" class="billing-tab-content"></div>
                <div id="billing-manual-tab" class="billing-tab-content"></div>
                <div id="billing-history-tab" class="billing-tab-content"></div>
            </div>

            <!-- Results Modal -->
            <div id="billing-results-modal" class="billing-modal">
                <div class="billing-modal-content">
                    <span class="billing-modal-close" id="billing-modal-close">&times;</span>
                    <div id="billing-modal-body"></div>
                </div>
            </div>
        `;

        this.mainBody.innerHTML = html;
        
        // Add event listener for modal close
        document.getElementById('billing-modal-close').addEventListener('click', () => this.closeModal());
        document.getElementById('billing-results-modal').addEventListener('click', (e) => {
            if (e.target.id === 'billing-results-modal') {
                this.closeModal();
            }
        });

        this.renderTabs();
        this.renderDashboard();
    }

   renderTabs() {
    const tabs = [
        { id: 'dashboard', label: '📊 Dashboard' },
        { id: 'settings', label: '⚙️ Settings' },
        { id: 'products', label: '📦 Products' },
        { id: 'manual', label: '🎯 Manual Billing' },
        { id: 'history', label: '📜 History' }
    ];

    const tabsHtml = tabs.map(tab => `
        <button class="billing-tab-btn ${tab.id === this.currentTab ? 'billing-tab-active' : ''}" 
                data-tab="${tab.id}">
            ${tab.label}
        </button>
    `).join('');

    const container = document.getElementById('billing-tabs-container');
    container.innerHTML = tabsHtml;

    // Add event listeners
    container.querySelectorAll('.billing-tab-btn').forEach(btn => {
        btn.addEventListener('click', (e) => this.switchTab(e.target.dataset.tab));
    });
}

switchTab(tabId) {
    this.currentTab = tabId;
    
    // Update active tab buttons
    document.querySelectorAll('.billing-tab-btn').forEach(btn => {
        btn.classList.toggle('billing-tab-active', btn.dataset.tab === tabId);
    });
    
    // Update tab content visibility
    document.querySelectorAll('.billing-tab-content').forEach(content => {
        content.classList.toggle('billing-tab-active', content.id === `billing-${tabId}-tab`);
    });

    // Load tab-specific content
    switch(tabId) {
        case 'dashboard':
            this.loadDashboardData();
            break;
        case 'settings':
            this.renderSettings();
            break;
        case 'products':
            this.renderProducts();
            break;
        case 'manual':
            this.renderManualBilling();
            break;
        case 'history':
            this.renderBillingHistory();
            break;
    }
}

renderDashboard() {
    const dashboardTab = document.getElementById('billing-dashboard-tab');
    dashboardTab.innerHTML = `
        <div class="billing-stats-grid">
            <div class="billing-stat-card">
                <h3 id="total-revenue">0</h3>
                <p>Total Revenue</p>
            </div>
            <div class="billing-stat-card">
                <h3 id="pending-invoices">0</h3>
                <p>Pending Invoices</p>
            </div>
            <div class="billing-stat-card">
                <h3 id="active-subscriptions">0</h3>
                <p>Active Subscriptions</p>
            </div>
            <div class="billing-stat-card">
                <h3 id="today-collections">0</h3>
                <p>Today's Collections</p>
            </div>
        </div>

        <div class="billing-card">
            <h2>📈 Recent Activity</h2>
            <div id="recent-activity-content" class="billing-loading">
                <div class="billing-spinner"></div>
                <p>Loading recent activity...</p>
            </div>
        </div>

        <div class="billing-card">
            <h2>💰 Quick Actions</h2>
            <div class="billing-btn-group">
                <button class="billing-btn billing-btn-primary" onclick="billingUI.createManualInvoice()">
                    ➕ Create Invoice
                </button>
                <button class="billing-btn billing-btn-success" onclick="billingUI.runAutoBilling()">
                    🔄 Run Auto Billing
                </button>
                <button class="billing-btn billing-btn-info" onclick="billingUI.viewReports()">
                    📊 View Reports
                </button>
            </div>
        </div>
    `;
}

 async loadDashboardData() {
    try {
        // Show loading state
        document.getElementById('total-revenue').textContent = '...';
        document.getElementById('pending-invoices').textContent = '...';
        document.getElementById('active-subscriptions').textContent = '...';
        document.getElementById('today-collections').textContent = '...';

        const response = await fetch(`${this.apiBase}?action=get_dashboard_stats`);
        const data = await response.json();
        
        if (data.status === 'success') {
            // Update dashboard stats
            document.getElementById('total-revenue').textContent = data.total_revenue;
            document.getElementById('pending-invoices').textContent = data.pending_invoices;
            document.getElementById('active-subscriptions').textContent = data.active_subscriptions;
            document.getElementById('today-collections').textContent = data.today_collections;
            
            // Load recent activity separately
            this.loadRecentActivity();
        } else {
            this.showAlert('error', data.message || 'Failed to load dashboard data');
        }
    } catch (error) {
        console.error('Error loading dashboard data:', error);
        this.showAlert('error', 'Failed to load dashboard data');
    }
}

async loadRecentActivity() {
    try {
        const response = await fetch(`${this.apiBase}?action=get_recent_activity`);
        const data = await response.json();
        
        if (data.status === 'success') {
            this.renderRecentActivity(data.recent_activity);
        }
    } catch (error) {
        console.error('Error loading recent activity:', error);
    }
}

async loadProducts() {
    try {
        const response = await fetch(`${this.apiBase}?action=get_products`);
        const data = await response.json();
        
        const tbody = document.getElementById('products-table-body');
        
        if (data.status === 'success' && data.products.length > 0) {
            tbody.innerHTML = data.products.map(product => `
                <tr>
                    <td>${product.name}</td>
                    <td>${product.category}</td>
                    <td>${product.price}</td>
                    <td>
                        <span class="billing-badge billing-badge-info">
                            ${product.billing_interval || product.charge_rate}
                        </span>
                    </td>
                    <td>
                        <span class="billing-badge ${product.status === 'active' ? 'billing-badge-success' : 'billing-badge-danger'}">
                            ${product.status}
                        </span>
                    </td>
                    <td>
                        <div class="billing-action-buttons">
                            <button class="billing-btn billing-btn-warning" onclick="billingUI.editProduct(${product.id})">
                                ✏️ Edit
                            </button>
                            <button class="billing-btn billing-btn-danger" onclick="billingUI.deleteProduct(${product.id})">
                                🗑️ Delete
                            </button>
                        </div>
                    </td>
                </tr>
            `).join('');
        } else {
            tbody.innerHTML = `
                <tr>
                    <td colspan="6">
                        <div class="billing-empty-state">
                            <div class="billing-empty-state-icon">📦</div>
                            <h3>No products found</h3>
                            <p>Add your first billing product to get started</p>
                        </div>
                    </td>
                </tr>
            `;
        }
    } catch (error) {
        console.error('Error loading products:', error);
        this.showAlert('error', 'Failed to load products');
    }
}

renderRecentActivity(activities) {
    const container = document.getElementById('recent-activity-content');
    
    if (!activities || activities.length === 0) {
        container.innerHTML = `
            <div class="billing-empty-state">
                <div class="billing-empty-state-icon">📊</div>
                <h3>No recent activity</h3>
                <p>Transactions will appear here once they occur</p>
            </div>
        `;
        return;
    }

    const activitiesHtml = activities.map(activity => `
        <div class="billing-activity-item">
            <div class="billing-activity-icon">${this.getActivityIcon(activity.type)}</div>
            <div class="billing-activity-details">
                <strong>${activity.description}</strong>
                <span class="billing-activity-meta">${activity.client_name} • ${activity.amount}</span>
            </div>
            <div class="billing-activity-time">${activity.time}</div>
        </div>
    `).join('');

    container.innerHTML = activitiesHtml;
}

getActivityIcon(type) {
    const icons = {
        'payment': '💰',
        'invoice': '🧾',
        'subscription': '🔄',
        'refund': '↩️',
        'penalty': '⚠️'
    };
    return icons[type] || '📝';
}

 renderSettings() {
    const settingsTab = document.getElementById('billing-settings-tab');
    settingsTab.innerHTML = `
        <div class="billing-card">
            <h2>⚙️ Auto Billing Settings</h2>
            <div id="settings-loading" class="billing-loading">
                <div class="billing-spinner"></div>
                <p>Loading settings...</p>
            </div>
            <div id="settings-content" style="display: none;">
                <div class="billing-settings-grid">
                    <div class="billing-setting-item">
                        <label>Auto Billing</label>
                        <p>Automatically generate invoices for recurring subscriptions</p>
                        <label class="billing-toggle-switch">
                            <input type="checkbox" id="auto-billing-toggle">
                            <span class="billing-toggle-slider"></span>
                        </label>
                    </div>
                    
                    <div class="billing-setting-item">
                        <label for="billing-day">Billing Day of Month</label>
                        <p>Day of month to run automatic billing (1-28)</p>
                        <input type="number" id="billing-day" class="billing-form-input" min="1" max="28">
                    </div>
                    
                    <div class="billing-setting-item">
                        <label for="grace-period">Grace Period (Days)</label>
                        <p>Number of days before late fees apply</p>
                        <input type="number" id="grace-period" class="billing-form-input" min="0" max="30">
                    </div>
                    
                    <div class="billing-setting-item">
                        <label for="late-fee">Late Fee Percentage</label>
                        <p>Percentage charged for late payments</p>
                        <input type="number" id="late-fee" class="billing-form-input" min="0" max="100" step="0.1">
                    </div>
                </div>
                
                <div class="billing-btn-group">
                    <button class="billing-btn billing-btn-primary" onclick="billingUI.saveSettings()">
                        💾 Save Settings
                    </button>
                    <button class="billing-btn billing-btn-info" onclick="billingUI.testAutoBilling()">
                        🔄 Test Auto Billing
                    </button>
                </div>
            </div>
        </div>
    `;
    
    // Load current settings
    this.loadSettings();
}

async loadSettings() {
    try {
        const response = await fetch(`${this.apiBase}?action=get_settings`);
        const data = await response.json();
        
        if (data.status === 'success') {
            document.getElementById('settings-loading').style.display = 'none';
            document.getElementById('settings-content').style.display = 'block';
            
            // Populate settings form
            const settings = data.settings;
            document.getElementById('auto-billing-toggle').checked = settings.auto_billing_enabled === '1';
            document.getElementById('billing-day').value = settings.billing_day_of_month || '1';
            document.getElementById('grace-period').value = settings.grace_period_days || '7';
            document.getElementById('late-fee').value = settings.late_fee_percentage || '5';
        }
    } catch (error) {
        console.error('Error loading settings:', error);
        this.showAlert('error', 'Failed to load settings');
    }
}

async saveSettings() {
    try {
        const settings = {
            auto_billing_enabled: document.getElementById('auto-billing-toggle').checked ? '1' : '0',
            billing_day_of_month: document.getElementById('billing-day').value,
            grace_period_days: document.getElementById('grace-period').value,
            late_fee_percentage: document.getElementById('late-fee').value
        };

        const response = await fetch(`${this.apiBase}?action=save_settings`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(settings)
        });

        const data = await response.json();
        
        if (data.status === 'success') {
            this.showAlert('success', 'Settings saved successfully');
            // Update dashboard status
            this.checkAutoBillingStatus();
        } else {
            this.showAlert('error', data.message);
        }

    } catch (error) {
        console.error('Error saving settings:', error);
        this.showAlert('error', 'Failed to save settings');
    }
}

testAutoBilling() {
    // Force run auto-billing even if disabled (for testing)
    if (confirm('Run auto-billing in test mode? This will ignore the enabled/disabled setting.')) {
        // You could create a special test endpoint or just enable temporarily
        this.runAutoBilling();
    }
}

renderProducts() {
    const productsTab = document.getElementById('billing-products-tab');
    productsTab.innerHTML = `
        <div class="billing-card">
            <h2>📦 Billing Products</h2>
            <div class="billing-search-box">
                <input type="text" class="billing-search-input" placeholder="Search products..." id="product-search">
            </div>
            <div class="billing-table-container">
                <table class="billing-table">
                    <thead>
                        <tr>
                            <th>Product Name</th>
                            <th>Category</th>
                            <th>Price</th>
                            <th>Billing Interval</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody id="products-table-body">
                        <tr>
                            <td colspan="6" class="billing-loading">
                                <div class="billing-spinner"></div>
                                <p>Loading products...</p>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
            
            <div class="billing-btn-group">
                <button class="billing-btn billing-btn-primary" onclick="billingUI.showAddProductModal()">
                    ➕ Add Product
                </button>
                <button class="billing-btn billing-btn-success" onclick="billingUI.importProducts()">
                    📥 Import Products
                </button>
            </div>
        </div>
    `;

    this.loadProducts();
}

async loadProducts() {
    try {
        const response = await fetch(`${this.apiBase}?action=get_products`);
        const data = await response.json();
        
        const tbody = document.getElementById('products-table-body');
        
        if (data.status === 'success' && data.products.length > 0) {
            tbody.innerHTML = data.products.map(product => `
                <tr>
                    <td>${product.name}</td>
                    <td>${product.category}</td>
                    <td>${product.price}</td>
                    <td>
                        <span class="billing-badge billing-badge-info">
                            ${product.billing_interval}
                        </span>
                    </td>
                    <td>
                        <span class="billing-badge ${product.status === 'Active' ? 'billing-badge-success' : 'billing-badge-danger'}">
                            ${product.status}
                        </span>
                    </td>
                    <td>
                        <div class="billing-action-buttons">
                            <button class="billing-btn billing-btn-warning" onclick="billingUI.editProduct(${product.id})">
                                ✏️ Edit
                            </button>
                            <button class="billing-btn billing-btn-danger" onclick="billingUI.deleteProduct(${product.id})">
                                🗑️ Delete
                            </button>
                        </div>
                    </td>
                </tr>
            `).join('');
        } else {
            tbody.innerHTML = `
                <tr>
                    <td colspan="6">
                        <div class="billing-empty-state">
                            <div class="billing-empty-state-icon">📦</div>
                            <h3>No products found</h3>
                            <p>Add your first billing product to get started</p>
                        </div>
                    </td>
                </tr>
            `;
        }
    } catch (error) {
        console.error('Error loading products:', error);
        this.showAlert('error', 'Failed to load products');
    }
}

renderManualBilling() {
    const manualTab = document.getElementById('billing-manual-tab');
    manualTab.innerHTML = `
        <div class="billing-card">
            <h2>🎯 Manual Billing</h2>
            
            <div class="billing-form-group">
                <label for="client-search">Select Client</label>
                <input type="text" id="client-search" class="billing-form-input" placeholder="Search for client...">
                <div id="client-results" class="billing-search-results"></div>
            </div>
            
            <div class="billing-form-group">
                <label for="product-select">Select Product</label>
                <select id="product-select" class="billing-form-select">
                    <option value="">Select a product...</option>
                </select>
            </div>
            
            <div class="billing-form-group">
                <label for="quantity">Quantity</label>
                <input type="number" id="quantity" class="billing-form-input" value="1" min="1">
            </div>
            
            <div class="billing-form-group">
                <label for="due-date">Due Date</label>
                <input type="date" id="due-date" class="billing-form-input">
            </div>
            
            <div class="billing-form-group">
                <label for="notes">Notes (Optional)</label>
                <textarea id="notes" class="billing-form-input" rows="3" placeholder="Add any additional notes..."></textarea>
            </div>
            
            <div class="billing-btn-group">
                <button class="billing-btn billing-btn-primary" onclick="billingUI.createManualInvoice()">
                    🧾 Create Invoice
                </button>
                <button class="billing-btn billing-btn-success" onclick="billingUI.createAndSendInvoice()">
                    📧 Create & Send
                </button>
                <button class="billing-btn billing-btn-info" onclick="billingUI.previewInvoice()">
                    👀 Preview
                </button>
            </div>
        </div>
    `;

    this.loadProductsForManualBilling();
}

renderBillingHistory() {
    const historyTab = document.getElementById('billing-history-tab');
    historyTab.innerHTML = `
        <div class="billing-card">
            <h2>📜 Billing History</h2>
            
            <div class="billing-search-box">
                <input type="text" class="billing-search-input" placeholder="Search invoices, clients, or amounts..." id="history-search">
            </div>
            
            <div class="billing-table-container">
                <table class="billing-table">
                    <thead>
                        <tr>
                            <th>Invoice #</th>
                            <th>Client</th>
                            <th>Amount</th>
                            <th>Due Date</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody id="history-table-body">
                        <tr>
                            <td colspan="6" class="billing-loading">
                                <div class="billing-spinner"></div>
                                <p>Loading billing history...</p>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>
    `;

    this.loadBillingHistory();
}

async loadBillingHistory() {
    try {
        const response = await fetch(`${this.apiBase}?action=get_billing_history`);
        const data = await response.json();
        
        const tbody = document.getElementById('history-table-body');
        
        if (data.status === 'success' && data.history.length > 0) {
            tbody.innerHTML = data.history.map(item => `
                <tr>
                    <td>#${item.invoice_no}</td>
                    <td>${item.client_name}</td>
                    <td>${item.amount}</td>
                    <td>${item.due_date}</td>
                    <td>
                        <span class="billing-badge ${this.getStatusBadgeClass(item.payment_status)}">
                            ${item.payment_status}
                        </span>
                    </td>
                    <td>
                        <div class="billing-action-buttons">
                            <button class="billing-btn billing-btn-info" onclick="billingUI.viewInvoice(${item.b_id})">
                                👁️ View
                            </button>
                            ${item.payment_status === 'Pending' ? `
                                <button class="billing-btn billing-btn-success" onclick="billingUI.recordPayment(${item.b_id})">
                                    💰 Record Payment
                                </button>
                            ` : ''}
                        </div>
                    </td>
                </tr>
            `).join('');
        } else {
            tbody.innerHTML = `
                <tr>
                    <td colspan="6">
                        <div class="billing-empty-state">
                            <div class="billing-empty-state-icon">📜</div>
                            <h3>No billing history found</h3>
                            <p>Transactions will appear here once billing occurs</p>
                        </div>
                    </td>
                </tr>
            `;
        }
    } catch (error) {
        console.error('Error loading billing history:', error);
        this.showAlert('error', 'Failed to load billing history');
    }
}

getStatusBadgeClass(status) {
    const statusClasses = {
        'Paid': 'billing-badge-success',
        'Pending': 'billing-badge-warning',
        'Overdue': 'billing-badge-danger',
        'Cancelled': 'billing-badge-gray'
    };
    return statusClasses[status] || 'billing-badge-info';
}

showAlert(type, message) {
    const alert = document.createElement('div');
    alert.className = `billing-alert billing-alert-${type}`;
    alert.innerHTML = `
        <span>${this.getAlertIcon(type)}</span>
        <span>${message}</span>
    `;
    
    document.getElementById('billing-main-container').insertBefore(alert, document.querySelector('.billing-tabs-container'));
    
    setTimeout(() => {
        alert.remove();
    }, 5000);
}

getAlertIcon(type) {
    const icons = {
        'success': '✅',
        'error': '❌',
        'warning': '⚠️',
        'info': 'ℹ️'
    };
    return icons[type] || '📝';
}

closeModal() {
    document.getElementById('billing-results-modal').classList.remove('billing-modal-show');
}

// Placeholder methods for future implementation
createManualInvoice() {
    this.showAlert('info', 'Manual invoice creation feature coming soon');
}

runAutoBilling() {
    this.showAlert('info', 'Auto billing feature coming soon');
}

viewReports() {
    this.showAlert('info', 'Reports feature coming soon');
}

saveSettings() {
    this.showAlert('success', 'Settings saved successfully');
}

testNotifications() {
    this.showAlert('info', 'Test notifications feature coming soon');
}

showAddProductModal() {
    this.showAlert('info', 'Add product modal coming soon');
}

importProducts() {
    this.showAlert('info', 'Import products feature coming soon');
}

editProduct(id) {
    this.showAlert('info', `Edit product ${id} feature coming soon`);
}

deleteProduct(id) {
    if (confirm('Are you sure you want to delete this product?')) {
        this.showAlert('success', `Product ${id} deleted successfully`);
    }
}

loadProductsForManualBilling() {
    // Implementation for loading products in manual billing
}

createAndSendInvoice() {
    this.showAlert('info', 'Create and send invoice feature coming soon');
}

previewInvoice() {
    this.showAlert('info', 'Preview invoice feature coming soon');
}

viewInvoice(id) {
    this.showAlert('info', `View invoice ${id} feature coming soon`);
}

recordPayment(id) {
    this.showAlert('info', `Record payment for invoice ${id} feature coming soon`);
}


 async runAutoBilling() {
    try {
        // Show confirmation dialog
        const confirmed = confirm('Are you sure you want to run auto-billing? This will generate invoices for all eligible clients.');
        if (!confirmed) return;

        // Create progress elements if they don't exist
        this.createProgressElements();
        
        // Show progress
        this.showAutoBillingProgress(true);
        
        // Disable the run button
        const runButton = document.getElementById('run-auto-billing-btn');
        if (runButton) runButton.disabled = true;

        const response = await fetch(`${this.apiBase}?action=run_auto_billing`);
        const data = await response.json();

        if (data.status === 'success') {
            this.showAlert('success', data.message);
            
            // Show detailed results
            this.showAutoBillingResults(data.summary, data.run_id);
            
            // Refresh dashboard data
            this.loadDashboardData();
        } else {
            this.showAlert('error', data.message);
        }

    } catch (error) {
        console.error('Error running auto-billing:', error);
        this.showAlert('error', 'Failed to run auto-billing');
    } finally {
        this.showAutoBillingProgress(false);
        const runButton = document.getElementById('run-auto-billing-btn');
        if (runButton) runButton.disabled = false;
    }
}

createProgressElements() {
    // Check if progress elements already exist
    let progressContainer = document.getElementById('auto-billing-progress');
    
    if (!progressContainer) {
        // Create progress container
        progressContainer = document.createElement('div');
        progressContainer.id = 'auto-billing-progress';
        progressContainer.className = 'billing-auto-billing-progress';
        progressContainer.style.display = 'none';
        
        // Create progress bar
        const progressBar = document.createElement('div');
        progressBar.className = 'billing-progress';
        
        const progressBarInner = document.createElement('div');
        progressBarInner.id = 'auto-billing-progress-bar';
        progressBarInner.className = 'billing-progress-bar';
        
        progressBar.appendChild(progressBarInner);
        
        // Create progress text
        const progressText = document.createElement('div');
        progressText.id = 'auto-billing-progress-text';
        progressText.className = 'billing-progress-text';
        progressText.textContent = 'Processing auto-billing...';
        
        progressContainer.appendChild(progressBar);
        progressContainer.appendChild(progressText);
        
        // Add to auto-billing panel
        const autoBillingPanel = document.querySelector('.billing-auto-billing-panel');
        if (autoBillingPanel) {
            autoBillingPanel.appendChild(progressContainer);
        }
    }
}

showAutoBillingProgress(show) {
    const progressElement = document.getElementById('auto-billing-progress');
    const progressBar = document.getElementById('auto-billing-progress-bar');
    const progressText = document.getElementById('auto-billing-progress-text');
    
    if (!progressElement || !progressBar || !progressText) {
        console.warn('Progress elements not found');
        return;
    }
    
    if (show) {
        progressElement.style.display = 'block';
        progressBar.style.width = '0%';
        progressText.textContent = 'Starting auto-billing process...';
        
        // Animate progress bar
        let width = 0;
        const interval = setInterval(() => {
            if (width >= 90) {
                clearInterval(interval);
            } else {
                width++;
                progressBar.style.width = width + '%';
            }
        }, 100);
        
    } else {
        progressElement.style.display = 'none';
        const progressBar = document.getElementById('auto-billing-progress-bar');
        if (progressBar) progressBar.style.width = '0%';
    }
}

    showAutoBillingResults(summary, runId) {
        const modalBody = document.getElementById('billing-modal-body');
        modalBody.innerHTML = `
            <div class="billing-results-summary">
                <h3>✅ Auto Billing Completed Successfully</h3>
                <div class="billing-results-stats">
                    <div class="billing-result-stat">
                        <span class="billing-result-number">${summary.total_invoices}</span>
                        <span class="billing-result-label">Total Invoices</span>
                    </div>
                    <div class="billing-result-stat">
                        <span class="billing-result-number">${summary.total_amount}</span>
                        <span class="billing-result-label">Total Amount</span>
                    </div>
                    <div class="billing-result-stat">
                        <span class="billing-result-number">${summary.successful}</span>
                        <span class="billing-result-label">Successful</span>
                    </div>
                    <div class="billing-result-stat">
                        <span class="billing-result-number">${summary.failed}</span>
                        <span class="billing-result-label">Failed</span>
                    </div>
                </div>
                
                <div class="billing-results-actions">
                    <button class="billing-btn billing-btn-primary" onclick="billingUI.viewBillingRunDetails(${runId})">
                        📋 View Detailed Report
                    </button>
                    <button class="billing-btn billing-btn-info" onclick="billingUI.viewBillingRuns()">
                        📊 All Billing Runs
                    </button>
                    <button class="billing-btn billing-btn-success" onclick="billingUI.closeModal()">
                        ✅ Done
                    </button>
                </div>
            </div>
        `;
        
        document.getElementById('billing-results-modal').classList.add('billing-modal-show');
    }

    async viewBillingRuns() {
        try {
            const response = await fetch(`${this.apiBase}?action=get_billing_runs`);
            const data = await response.json();
            
            if (data.status === 'success') {
                this.showBillingRunsModal(data.runs);
            } else {
                this.showAlert('error', data.message);
            }
        } catch (error) {
            console.error('Error fetching billing runs:', error);
            this.showAlert('error', 'Failed to load billing runs');
        }
    }

    showBillingRunsModal(runs) {
        const modalBody = document.getElementById('billing-modal-body');
        
        if (runs.length === 0) {
            modalBody.innerHTML = `
                <div class="billing-empty-state">
                    <div class="billing-empty-state-icon">📊</div>
                    <h3>No Billing Runs Found</h3>
                    <p>Auto-billing runs will appear here once executed</p>
                </div>
            `;
        } else {
            modalBody.innerHTML = `
                <div class="billing-runs-history">
                    <h3>📊 Billing Runs History</h3>
                    <div class="billing-table-container">
                        <table class="billing-table">
                            <thead>
                                <tr>
                                    <th>Run Date</th>
                                    <th>Type</th>
                                    <th>Invoices</th>
                                    <th>Amount</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${runs.map(run => `
                                    <tr>
                                        <td>${run.run_date} ${run.run_time}</td>
                                        <td>
                                            <span class="billing-badge ${run.run_type === 'auto' ? 'billing-badge-info' : 'billing-badge-primary'}">
                                                ${run.run_type}
                                            </span>
                                        </td>
                                        <td>${run.total_invoices_generated}</td>
                                        <td>${run.total_amount_billed ? Number(run.total_amount_billed).toLocaleString() : '0'}</td>
                                        <td>
                                            <span class="billing-badge ${this.getRunStatusBadgeClass(run.status)}">
                                                ${run.status}
                                            </span>
                                        </td>
                                        <td>
                                            <button class="billing-btn billing-btn-info" onclick="billingUI.viewBillingRunDetails(${run.id})">
                                                👁️ View
                                            </button>
                                        </td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    </div>
                </div>
            `;
        }
        
        document.getElementById('billing-results-modal').classList.add('billing-modal-show');
    }

    getRunStatusBadgeClass(status) {
        const statusClasses = {
            'completed': 'billing-badge-success',
            'running': 'billing-badge-warning',
            'failed': 'billing-badge-danger',
            'pending': 'billing-badge-info'
        };
        return statusClasses[status] || 'billing-badge-gray';
    }

    async viewBillingRunDetails(runId) {
        // This would fetch detailed run information
        this.showAlert('info', `Viewing details for billing run #${runId} - Feature coming soon`);
    }

    async checkAutoBillingStatus() {
        try {
            const response = await fetch(`${this.apiBase}?action=get_settings`);
            const data = await response.json();
            
            if (data.status === 'success') {
                this.updateAutoBillingStatus(data.settings);
            }
        } catch (error) {
            console.error('Error checking auto-billing status:', error);
        }
    }

    updateAutoBillingStatus(settings) {
        const statusElement = document.getElementById('auto-billing-status');
        const nextRunElement = document.getElementById('next-billing-run');
        const runButton = document.getElementById('run-auto-billing-btn');
        
        const isEnabled = settings.auto_billing_enabled === '1';
        const billingDay = settings.billing_day_of_month || '1';
        
        if (statusElement) {
            const statusDot = statusElement.querySelector('.billing-status-dot');
            const statusText = statusElement.querySelector('.billing-status-text');
            
            if (isEnabled) {
                statusDot.style.backgroundColor = '#10b981';
                statusText.textContent = 'Auto-billing is ENABLED';
                runButton.disabled = false;
            } else {
                statusDot.style.backgroundColor = '#ef4444';
                statusText.textContent = 'Auto-billing is DISABLED';
                runButton.disabled = true;
            }
        }
        
        if (nextRunElement) {
            const today = new Date().getDate();
            const billingDayNum = parseInt(billingDay);
            
            if (today === billingDayNum) {
                nextRunElement.textContent = 'Next run: TODAY';
            } else if (today < billingDayNum) {
                nextRunElement.textContent = `Next run: Day ${billingDay} of month`;
            } else {
                nextRunElement.textContent = `Next run: Day ${billingDay} of next month`;
            }
        }
    }

    openAutoBillingSettings() {
        this.switchTab('settings');
    }

    // Update your loadDashboardData to include status check
    async loadDashboardData() {
        try {
            // ... your existing dashboard loading code ...
            
            // Check auto-billing status
            await this.checkAutoBillingStatus();
            
        } catch (error) {
            console.error('Error loading dashboard data:', error);
            this.showAlert('error', 'Failed to load dashboard data');
        }
    }
}

// Initialize the billing system when DOM is loaded
/*
document.addEventListener('DOMContentLoaded', function() {
    window.billingUI = new BillingSystemUI();
});
*/
 $(document).on('click', '.billingDashBoard', function(event) {
    event.preventDefault();
    
    // Hide other modules
    $('.module-content').hide();
    
    // Show or create billing container
    let billingContainer = $('#billing-container');
    if (billingContainer.length === 0) {
        $('body').append('<div id="billing-container" class="billing-module"></div>');
        billingContainer = $('#billing-container');
    }
    
    billingContainer.show();
    
    // Initialize billing system if not already done
    if (!window.billingUI) {
        window.billingUI = new BillingSystemUI('#billing-container');
    } else {
        // Refresh or show billing dashboard
        window.billingUI.switchTab('dashboard');
    }
    
    // Update active menu state
    $('.sidenav-inner-text').removeClass('active');
    $(this).addClass('active');
});