class AutomaticBilling {
  constructor() {
    this.schedules = [];
    this.products = [];
    this.billingHistory = [];
    this.loading = false;
    this.editingScheduleId = null;
    this.confirmingProductId = null;
    this.currentView = 'schedules';
  }

  initialize() {
    this.createAutomaticBillingContainer();
    this.createBillingModal();
    this.createRunConfirmationModal();
    this.createBulkRunConfirmationModal();
    this.loadSchedules();
    this.loadProducts();
    this.attachEventListeners();
    
    this.currentView = 'schedules';
    this.showBillingSchedules();
  }

  createAutomaticBillingContainer() {
    const mainBody = document.getElementById('main-body');
    if (!mainBody) return;

    mainBody.innerHTML = '';
    
    const container = document.createElement('div');
    container.id = 'automatic-billing-container';
    container.className = 'automatic-billing-container';
    container.innerHTML = this.getAutomaticBillingHTML();
    
    mainBody.appendChild(container);
  }

  createBillingModal() {
    const existingModal = document.getElementById('automatic-billing-modal');
    if (existingModal) {
      existingModal.remove();
    }

    const modal = document.createElement('div');
    modal.id = 'automatic-billing-modal';
    modal.className = 'automatic-billing-modal';
    modal.style.display = 'none';
    modal.innerHTML = `
        <div class="automatic-billing-modal-content">
            <div class="automatic-billing-modal-header">
                <h3 id="automatic-billing-modal-title">
                    <i class="fas fa-plus-circle"></i> Add Billing Schedule
                </h3>
                <span class="automatic-billing-modal-close" onclick="automaticBilling.closeModal()">&times;</span>
            </div>
            <div class="automatic-billing-modal-body" style="overflow-y: auto; max-height: 60vh;">
                <form id="automatic-billing-schedule-form">
                    <div class="automatic-billing-form-group">
                        <label class="automatic-billing-form-label">Select Product</label>
                        <select id="automatic-billing-product-select" class="automatic-billing-form-select" required>
                            <option value="">Loading products...</option>
                        </select>
                    </div>
                    <div class="automatic-billing-form-group">
                        <label class="automatic-billing-form-label">Billing Interval</label>
                        <select id="automatic-billing-interval-select" class="automatic-billing-form-select" required>
                            <option value="daily">Daily</option>
                            <option value="weekly">Weekly</option>
                            <option value="monthly" selected>Monthly</option>
                            <option value="quarterly">Quarterly</option>
                            <option value="yearly">Yearly</option>
                        </select>
                    </div>
                    <div class="automatic-billing-form-group">
                        <label class="automatic-billing-form-label">Next Billing Date</label>
                        <input type="date" id="next-automatic-billing-date" class="automatic-billing-form-input" required>
                    </div>
                    <div class="automatic-billing-form-group">
                        <label class="automatic-billing-form-label">Automated Billing</label>
                        <div class="automatic-billing-checkbox-group">
                            <input type="checkbox" id="automatic-billing-enabled" checked>
                            <label for="automatic-billing-enabled">Enable</label>
                        </div>
                    </div>
                    <div class="automatic-billing-form-group">
                        <label class="automatic-billing-form-label">Notifications</label>
                        <div class="automatic-billing-checkbox-group">
                            <input type="checkbox" id="automatic-email-enabled" checked>
                            <label for="email-notification-enabled">Email</label>
                        </div>
                        <div class="automatic-billing-checkbox-group">
                            <input type="checkbox" id="automatic-sms-enabled" checked>
                            <label for="automatic-sms-enabled">SMS</label>
                        </div>
                    </div>
                </form>
            </div>
            <div class="automatic-billing-modal-footer">
                <button type="button" class="automatic-billing-btn automatic-billing-btn-outline" onclick="automaticBilling.closeModal()">
                    <i class="fontx fx-x"></i> Cancel
                </button>
                <button type="button" class="automatic-billing-btn automatic-billing-btn-success" onclick="automaticBilling.saveSchedule()">
                    <i class="fonts fx-save"></i> Save Schedule
                </button>
            </div>
        </div>
    `;

    document.body.appendChild(modal);
  }

  createRunConfirmationModal() {
    const existingModal = document.getElementById('automatic-billing-confirm-modal');
    if (existingModal) {
      existingModal.remove();
    }

    const modal = document.createElement('div');
    modal.id = 'automatic-billing-confirm-modal';
    modal.className = 'automatic-billing-modal';
    modal.style.display = 'none';
    modal.innerHTML = `
        <div class="automatic-billing-modal-content">
            <div class="automatic-billing-modal-header">
                <h3 id="automatic-billing-confirm-modal-title">
                    <i class="fontx fx-play"></i> Confirm Billing Run
                </h3>
                <span class="automatic-billing-modal-close" onclick="automaticBilling.closeConfirmModal()">&times;</span>
            </div>
            <div class="automatic-billing-modal-body">
                <div id="automatic-billing-confirm-details">
                    <div class="automatic-billing-confirm-section">
                        <h4 class="automatic-billing-confirm-title">Billing Details</h4>
                        <div class="automatic-billing-confirm-details-grid">
                            <div class="automatic-billing-confirm-detail">
                                <span class="automatic-billing-confirm-label">Product:</span>
                                <span id="confirm-product-name" class="automatic-billing-confirm-value"></span>
                            </div>
                            <div class="automatic-billing-confirm-detail">
                                <span class="automatic-billing-confirm-label">Billing Interval:</span>
                                <span id="confirm-billing-interval" class="automatic-billing-confirm-value"></span>
                            </div>
                            <div class="automatic-billing-confirm-detail">
                                <span class="automatic-billing-confirm-label">Next Billing Date:</span>
                                <span id="confirm-next-billing-date" class="automatic-billing-confirm-value"></span>
                            </div>
                            <div class="automatic-billing-confirm-detail">
                                <span class="automatic-billing-confirm-label">Active Subscribers:</span>
                                <span id="confirm-active-subscribers" class="automatic-billing-confirm-value"></span>
                            </div>
                            <div class="automatic-billing-confirm-detail">
                                <span class="automatic-billing-confirm-label">Auto Billing:</span>
                                <span id="confirm-auto-billing" class="automatic-billing-confirm-value"></span>
                            </div>
                            <div class="automatic-billing-confirm-detail">
                                <span class="automatic-billing-confirm-label">Email Notifications:</span>
                                <span id="confirm-email-notifications" class="automatic-billing-confirm-value"></span>
                            </div>
                            <div class="automatic-billing-confirm-detail">
                                <span class="automatic-billing-confirm-label">SMS Notifications:</span>
                                <span id="confirm-sms-notifications" class="automatic-billing-confirm-value"></span>
                            </div>
                        </div>
                    </div>
                    <div class="automatic-billing-confirm-warning">
                        <i class="fontx fx-alert-triangle" style="color: #ff9800;"></i>
                        <span>This action will generate invoices for all active subscribers of this product.</span>
                    </div>
                </div>
            </div>
            <div class="automatic-billing-modal-footer">
                <button type="button" class="automatic-billing-btn automatic-billing-btn-outline" onclick="automaticBilling.closeConfirmModal()">
                    <i class="fontx fx-x"></i> Cancel
                </button>
                <button type="button" class="automatic-billing-btn automatic-billing-btn-primary" id="confirm-run-billing-btn">
                    <i class="fontx fx-play"></i> Run Billing Now
                </button>
            </div>
        </div>
    `;

    document.body.appendChild(modal);
    
    const confirmButton = document.getElementById('confirm-run-billing-btn');
    if (confirmButton) {
      confirmButton.addEventListener('click', () => {
        const productId = confirmButton.getAttribute('data-product-id');
        if (productId) {
          this.confirmRunSchedule(parseInt(productId));
        }
      });
    }
  }

  createBulkRunConfirmationModal() {
    const existingModal = document.getElementById('automatic-billing-bulk-confirm-modal');
    if (existingModal) {
      existingModal.remove();
    }

    const modal = document.createElement('div');
    modal.id = 'automatic-billing-bulk-confirm-modal';
    modal.className = 'automatic-billing-modal';
    modal.style.display = 'none';
    modal.innerHTML = `
        <div class="automatic-billing-modal-content">
            <div class="automatic-billing-modal-header">
                <h3 id="automatic-billing-bulk-confirm-modal-title">
                    <i class="fontx fx-play"></i> Confirm Bulk Billing Run
                </h3>
                <span class="automatic-billing-modal-close" onclick="automaticBilling.closeBulkConfirmModal()">&times;</span>
            </div>
            <div class="automatic-billing-modal-body">
                <div id="automatic-billing-bulk-confirm-details">
                    <div class="automatic-billing-confirm-section">
                        <h4 class="automatic-billing-confirm-title">Bulk Billing Summary</h4>
                        <div class="automatic-billing-confirm-details-grid">
                            <div class="automatic-billing-confirm-detail">
                                <span class="automatic-billing-confirm-label">Total Schedules:</span>
                                <span id="bulk-confirm-total-schedules" class="automatic-billing-confirm-value"></span>
                            </div>
                            <div class="automatic-billing-confirm-detail">
                                <span class="automatic-billing-confirm-label">Total Subscribers:</span>
                                <span id="bulk-confirm-total-subscribers" class="automatic-billing-confirm-value"></span>
                            </div>
                            <div class="automatic-billing-confirm-detail">
                                <span class="automatic-billing-confirm-label">Due Schedules:</span>
                                <span id="bulk-confirm-due-schedules" class="automatic-billing-confirm-value"></span>
                            </div>
                        </div>
                    </div>
                    <div class="automatic-billing-confirm-products">
                        <h5 class="automatic-billing-confirm-subtitle">Products to Bill:</h5>
                        <div id="bulk-confirm-products-list" class="automatic-billing-confirm-products-list"></div>
                    </div>
                    <div class="automatic-billing-confirm-warning">
                        <i class="fontx fx-alert-triangle" style="color: #ff9800;"></i>
                        <span>This action will generate invoices for all active subscribers across all scheduled products.</span>
                    </div>
                </div>
            </div>
            <div class="automatic-billing-modal-footer">
                <button type="button" class="automatic-billing-btn automatic-billing-btn-outline" onclick="automaticBilling.closeBulkConfirmModal()">
                    <i class="fontx fx-x"></i> Cancel
                </button>
                <button type="button" class="automatic-billing-btn automatic-billing-btn-primary" id="confirm-bulk-run-billing-btn">
                    <i class="fontx fx-play"></i> Run All Scheduled Billing
                </button>
            </div>
        </div>
    `;

    document.body.appendChild(modal);
    
    const confirmButton = document.getElementById('confirm-bulk-run-billing-btn');
    if (confirmButton) {
      confirmButton.addEventListener('click', () => {
        this.confirmBulkRunSchedule();
      });
    }
  }

  getAutomaticBillingHTML() {
    return `
    <div class="automatic-billing-content">
        <!-- Header -->
        <div>
              <div class="automatic-billing-title-section cyber-horizontal-scroll">
                 <button id="automatic-billing-schedules-btn">
                    <i class="fontx fx-sync"></i> Billing Schedules
                </button>
                 <button id="automatic-billing-history-btn"">
                    <i class="fontx fx-history"></i> Billing History
                </button>
              
                <button id="automatic-billing-add-btn">
                    <i class="fontx fx-plus"></i> Add Schedule
                </button>
                <button id="automatic-billing-run-btn">
                    <i class="fontx fx-play"></i> Run Scheduled Billing
                </button>
            </div>
        </div>

        <!-- Content Area -->
        <div class="automatic-billing-card">
            <div class="automatic-billing-card-body">
                <div id="automatic-billing-content-area">
                    <div class="automatic-billing-empty-state">
                        <i class="fontx fx-home automatic-billing-empty-icon"></i>
                        <p>Loading billing schedules...</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Loading State -->
        <div id="automatic-billing-loading" class="automatic-billing-loading" style="display: none;">
            <div class="automatic-billing-loading-content">
                <div class="automatic-billing-spinner"></div>
                <p class="automatic-billing-loading-text">Processing...</p>
            </div>
        </div>
    </div>
    `;
  }

  async loadBillingHistory() {
    try {
      this.showLoading(true);
      const response = await fetch('xd-modules/billing-module/backend/get-billing-history.php');
      const data = await response.json();
      
      if (data.success) {
        this.billingHistory = data.history;
        if (this.currentView === 'history') {
          this.renderBillingHistory();
          this.updateActiveButton('history');
          this.attachSearchListener();
        }
      } else {
        this.showNotification('Error loading billing history', 'error');
      }
    } catch (error) {
      this.showNotification('Error loading billing history', 'error');
      console.error('Error loading billing history:', error);
    } finally {
      this.showLoading(false);
    }
  }

  renderBillingHistory(historyToRender = null) {
    const contentArea = document.getElementById('automatic-billing-content-area');
    const history = historyToRender || this.billingHistory;
    
    if (!history || history.length === 0) {
      contentArea.innerHTML = `
        <div class="automatic-billing-empty-state">
          <i class="fas fa-history automatic-billing-empty-icon"></i>
          <p>No billing history available</p>
        </div>
      `;
      return;
    }

    contentArea.innerHTML = `
      <div class="billing-history-container"> 
        <div class="billing-history-table">
          <table class="billing-table">
            <thead>
              <tr>
                <th>Run Date</th>
                <th>Product</th>
                <th>Clients Billed</th>
                <th>Invoices</th>
                <th>Total Amount</th>
                <th>Status</th>
                <th>Run By</th>
                <th>Actions</th>
              </tr>
            </thead>
            <tbody>
              ${history.map(run => `
                <tr>
                  <td>${this.formatDate(run.run_date)}</td>
                  <td>${run.product_name || 'N/A'}</td>
                  <td>${run.total_clients_billed || 0}</td>
                  <td>${run.total_invoices_generated || 0}</td>
                  <td>$${this.formatAmount(run.total_amount_billed)}</td>
                  <td>
                    <span class="billing-status billing-status-${run.status || 'unknown'}">
                      ${run.status || 'Unknown'}
                    </span>
                  </td>
                  <td>${run.run_by_user_name || 'System'}</td>
                  <td>
                    <button class="automatic-billing-btn automatic-billing-btn-sm" onclick="automaticBilling.viewRunDetails(${run.id})">
                      <i class="fontx fx-info"></i> View
                    </button>
                  </td>
                </tr>
              `).join('')}
            </tbody>
          </table>
        </div>
      </div>
    `;
  }

  formatAmount(amount) {
    if (amount === null || amount === undefined) {
      return '0.00';
    }
    
    const numAmount = typeof amount === 'string' ? parseFloat(amount) : amount;
    
    if (isNaN(numAmount)) {
      return '0.00';
    }
    
    return numAmount.toFixed(2);
  }

  formatDate(dateString) {
    if (!dateString) return 'N/A';
    try {
      const date = new Date(dateString);
      return date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
    } catch (error) {
      return 'Invalid Date';
    }
  }

  viewRunDetails(runId) {
    this.showNotification(`Viewing details for run #${runId}`, 'info');
  }

  async loadSchedules() {
    try {
      const response = await fetch('xd-modules/billing-module/backend/get-billing-schedules.php');
      const data = await response.json();
      
      if (data.success) {
        this.schedules = data.schedules;
        if (this.currentView === 'schedules') {
          this.renderSchedules();
        }
        this.updateScheduleCount();
        this.attachSearchListener();
      }
    } catch (error) {
      this.showNotification('Error loading schedules', 'error');
    }
  }

  async loadProducts() {
    try {
      const response = await fetch('xd-modules/billing-module/backend/get-billing-products.php');
      const data = await response.json();
      
      if (data.success) {
        this.products = data.products;
        this.populateProductSelect();
      }
    } catch (error) {
      this.showNotification('Error loading products', 'error');
    }
  }

  populateProductSelect() {
    const select = document.getElementById('automatic-billing-product-select');
    if (!select) return;
    
    select.innerHTML = '<option value="">Select a product...</option>';
    
    this.products.forEach(product => {
      const option = document.createElement('option');
      option.value = product.id;
      
      const existingSchedule = this.schedules.find(s => s.product_id === product.id);
      if (existingSchedule) {
        option.textContent = `${product.name} (${product.product_code}) - $${product.price} - [Has Schedule]`;
        option.style.fontStyle = 'italic';
      } else {
        option.textContent = `${product.name} (${product.product_code}) - $${product.price}`;
      }
      
      select.appendChild(option);
    });
  }

  renderSchedules(schedulesToRender = null) {
    const contentArea = document.getElementById('automatic-billing-content-area');
    const schedules = schedulesToRender || this.schedules;
    
    if (schedules.length === 0) {
      contentArea.innerHTML = `
        <div class="automatic-billing-empty-state">
          <i class="fas fa-calendar-plus automatic-billing-empty-icon"></i>
          <p>No billing schedules found</p>
        </div>
      `;
      this.updateScheduleCount(0);
      return;
    }

    contentArea.innerHTML = `
      <div class="automatic-billing-schedules-container"> 
        <div class="automatic-billing-schedules-grid">
          ${schedules.map(schedule => `
            <div class="automatic-billing-schedule-card">
              <div class="automatic-billing-schedule-header">
                <h4 class="automatic-billing-schedule-product">${schedule.product_name}</h4>
                <span class="automatic-billing-schedule-status automatic-billing-status-${schedule.status}">
                  ${schedule.status}
                </span>
              </div>
              <div class="automatic-billing-schedule-details">
                <div class="automatic-billing-schedule-detail">
                  <span class="automatic-billing-detail-label">Interval:</span>
                  <span class="automatic-billing-detail-value">${this.formatInterval(schedule.billing_interval)}</span>
                </div>
                <div class="automatic-billing-schedule-detail">
                  <span class="automatic-billing-detail-label">Next Billing:</span>
                  <span class="automatic-billing-detail-value">${schedule.next_billing_date || 'Not scheduled'}</span>
                </div>
                <div class="automatic-billing-schedule-detail">
                  <span class="automatic-billing-detail-label">Subscribers:</span>
                  <span class="automatic-billing-detail-value">${schedule.active_subscribers || 0}</span>
                </div>
                <div class="automatic-billing-schedule-detail">
                  <span class="automatic-billing-detail-label">Auto Billing:</span>
                  <span class="automatic-billing-detail-value">
                    ${schedule.auto_billing_enabled ? 
                      '<i class="fas fa-check automatic-billing-enabled"></i> Enabled' : 
                      '<i class="fas fa-times automatic-billing-disabled"></i> Disabled'}
                  </span>
                </div>
              </div>
              <div class="automatic-billing-schedule-actions">
                <button class="automatic-billing-btn automatic-billing-btn-sm" 
                        onclick="automaticBilling.showRunConfirmation(${parseInt(schedule.product_id)})">
                  <i class="fontx fx-play"></i> Run Now
                </button>
                <button class="automatic-billing-btn automatic-billing-btn-sm automatic-billing-btn-outline"
                        onclick="automaticBilling.editSchedule(${parseInt(schedule.product_id)})">
                  <i class="fontx fx-edit"></i> Edit
                </button>
              </div>
            </div>
          `).join('')}
        </div>
      </div>
    `;
    
    this.updateScheduleCount(schedules.length);
  }

  attachEventListeners() {
    const titleSection = document.querySelector('.automatic-billing-title-section');
    
    const historyBtn = document.getElementById('automatic-billing-history-btn');
    if (historyBtn) {
      historyBtn.addEventListener('click', () => {
        this.showBillingHistory();
        if (titleSection) {
          titleSection.classList.remove('schedules-active');
        }
      });
    }

    const schedulesBtn = document.getElementById('automatic-billing-schedules-btn');
    if (schedulesBtn) {
      schedulesBtn.addEventListener('click', () => {
        this.showBillingSchedules();
        if (titleSection) {
          titleSection.classList.add('schedules-active');
        }
      });
    }

    const addBtn = document.getElementById('automatic-billing-add-btn');
    if (addBtn) {
      addBtn.addEventListener('click', () => {
        this.openModal();
      });
    }

    const runBtn = document.getElementById('automatic-billing-run-btn');
    if (runBtn) {
      runBtn.addEventListener('click', () => {
        this.showBulkRunConfirmation();
      });
    }

    this.attachSearchListener();
    this.attachProductSelectListener();
  }

  attachSearchListener() {
    const searchInput = document.getElementById('searchActiveBilling');
    if (searchInput) {
      searchInput.oninput = (event) => {
        this.handleSearch(event);
      };
    } else {
      setTimeout(() => {
        this.attachSearchListener();
      }, 100);
    }
  }

  handleSearch(event) {
    const searchTerm = event.target.value.trim();
    
    if (this.currentView === 'history') {
      this.searchBillingHistory(searchTerm);
    } else if (this.currentView === 'schedules') {
      this.searchSchedules(searchTerm);
    }
  }

  searchBillingHistory(searchTerm) {
    if (!searchTerm) {
      this.renderBillingHistory();
      return;
    }

    const term = searchTerm.toLowerCase();
    const filteredHistory = this.billingHistory.filter(run => {
      return (
        (run.product_name && run.product_name.toLowerCase().includes(term)) ||
        (run.status && run.status.toLowerCase().includes(term)) ||
        (run.run_by_user_name && run.run_by_user_name.toLowerCase().includes(term)) ||
        (run.run_date && run.run_date.toLowerCase().includes(term)) ||
        (run.total_clients_billed && run.total_clients_billed.toString().includes(term)) ||
        (run.total_invoices_generated && run.total_invoices_generated.toString().includes(term)) ||
        (run.total_amount_billed && run.total_amount_billed.toString().includes(term)) ||
        (run.id && run.id.toString().includes(term))
      );
    });

    this.renderBillingHistory(filteredHistory);
  }

  searchSchedules(searchTerm) {
    if (!searchTerm.trim()) {
      this.renderSchedules();
      return;
    }

    const term = searchTerm.toLowerCase();
    const filteredSchedules = this.schedules.filter(schedule => {
      return (
        (schedule.product_name && schedule.product_name.toLowerCase().includes(term)) ||
        (schedule.billing_interval && schedule.billing_interval.toLowerCase().includes(term)) ||
        (schedule.status && schedule.status.toLowerCase().includes(term)) ||
        (schedule.next_billing_date && schedule.next_billing_date.toLowerCase().includes(term)) ||
        (schedule.active_subscribers && schedule.active_subscribers.toString().includes(term)) ||
        (schedule.auto_email_enabled && schedule.auto_email_enabled.toString().includes(term)) ||
        (schedule.auto_sms_enabled && schedule.auto_sms_enabled.toString().includes(term)) ||
        (schedule.product_id && schedule.product_id.toString().includes(term))
      );
    });

    this.renderSchedules(filteredSchedules);
  }

  showBillingHistory() {
    this.currentView = 'history';
    this.loadBillingHistory();
    this.updateActiveButton('history');
    this.updateSearchPlaceholder('history');
    this.attachSearchListener();
    this.updateDashboardHeaderTitle('history');
  }

  showBillingSchedules() {
    this.currentView = 'schedules';
    
    if (this.schedules.length === 0) {
      const contentArea = document.getElementById('automatic-billing-content-area');
      if (contentArea) {
        contentArea.innerHTML = `
          <div class="automatic-billing-empty-state">
            <i class="fas fa-sync automatic-billing-empty-icon"></i>
            <p>Loading billing schedules...</p>
          </div>
        `;
      }
    }
    
    this.renderSchedules();
    this.updateActiveButton('schedules');
    this.updateSearchPlaceholder('schedules');
    this.attachSearchListener();
    this.updateDashboardHeaderTitle('schedules');
  }

  updateSearchPlaceholder(view) {
    const searchInput = document.getElementById('searchActiveBilling');
    if (searchInput) {
      if (view === 'history') {
        searchInput.placeholder = 'Search billing history by product, status, run by, date, amount...';
      } else {
        searchInput.placeholder = 'Search schedules by product name, interval, or status...';
      }
      searchInput.value = '';
    }
  }

  updateActiveButton(activeView) {
    const historyBtn = document.getElementById('automatic-billing-history-btn');
    const schedulesBtn = document.getElementById('automatic-billing-schedules-btn');
    
    if (historyBtn && schedulesBtn) {
      if (activeView === 'history') {
        historyBtn.classList.add('automatic-billing-btn-active');
        schedulesBtn.classList.remove('automatic-billing-btn-active');
      } else {
        schedulesBtn.classList.add('automatic-billing-btn-active');
        historyBtn.classList.remove('automatic-billing-btn-active');
      }
    }
  }

  updateScheduleCount(count = null) {
    const countElement = document.getElementById('record01');
    if (countElement) {
      const scheduleCount = count !== null ? count : this.schedules.length;
      countElement.textContent = `${scheduleCount} schedule${scheduleCount !== 1 ? 's' : ''}`;
    }
  }

  openModal() {
    this.editingScheduleId = null;
    const modal = document.getElementById('automatic-billing-modal');
    const modalTitle = document.getElementById('automatic-billing-modal-title');
    
    if (modal && modalTitle) {
      modalTitle.innerHTML = '<i class="fas fa-plus-circle"></i> Add Billing Schedule';
      modal.style.display = 'block';
      this.resetForm();
    }
  }

  closeModal() {
    const modal = document.getElementById('automatic-billing-modal');
    if (modal) {
      modal.style.display = 'none';
    }
    this.editingScheduleId = null;
    this.resetForm();
  }

  resetForm() {
    const form = document.getElementById('automatic-billing-schedule-form');
    if (form) {
      form.reset();
      const tomorrow = new Date();
      tomorrow.setDate(tomorrow.getDate() + 1);
      document.getElementById('next-automatic-billing-date').value = tomorrow.toISOString().split('T')[0];
      document.getElementById('automatic-billing-enabled').checked = true;
      document.getElementById('automatic-email-enabled').checked = true;
      document.getElementById('automatic-sms-enabled').checked = true;
    }
  }

  async saveSchedule() {
    const productId = document.getElementById('automatic-billing-product-select').value;
    const interval = document.getElementById('automatic-billing-interval-select').value;
    const autoBilling = document.getElementById('automatic-billing-enabled').checked;
    const nextBillingDate = document.getElementById('next-automatic-billing-date').value;
    const autoEmailBilling = document.getElementById('automatic-email-enabled').checked;
    const autoSMSBilling = document.getElementById('automatic-sms-enabled').checked;
  
    if (!productId) {
      this.showNotification('Please select a product', 'error');
      return;
    }

    if (!nextBillingDate) {
      this.showNotification('Please select a next billing date', 'error');
      return;
    }

    this.showLoading(true);

    try {
      const formData = new FormData();
      formData.append('product_id', productId);
      formData.append('billing_interval', interval);
      formData.append('auto_billing_enabled', autoBilling ? '1' : '0');
      formData.append('next_billing_date', nextBillingDate);
      formData.append('auto_email_enabled', autoEmailBilling ? '1' : '0');
      formData.append('auto_sms_enabled', autoSMSBilling ? '1' : '0');

      if (this.editingScheduleId) {
        formData.append('schedule_id', this.editingScheduleId);
        formData.append('action', 'update');
      } else {
        formData.append('action', 'create');
      }

      const response = await fetch('xd-modules/billing-module/backend/save-billing-schedule.php', {
        method: 'POST',
        body: formData
      });

      const data = await response.json();

      if (data.success) {
        this.showNotification('Schedule saved successfully', 'success');
        this.refreshData();
        this.closeModal();
      } else {
        this.showNotification(data.message, 'error');
      }
    } catch (error) {
      this.showNotification('Error saving schedule', 'error');
    } finally {
      this.showLoading(false);
    }
  }

  showRunConfirmation(productId) {
    console.log('showRunConfirmation called with productId:', productId);
    
    const schedule = this.schedules.find(s => s.product_id == productId);
    if (!schedule) {
      this.showNotification('Schedule not found', 'error');
      return;
    }

    const modal = document.getElementById('automatic-billing-confirm-modal');
    const confirmButton = document.getElementById('confirm-run-billing-btn');
    
    if (modal && confirmButton) {
      confirmButton.setAttribute('data-product-id', productId);
      
      document.getElementById('confirm-product-name').textContent = schedule.product_name || 'N/A';
      document.getElementById('confirm-billing-interval').textContent = this.formatInterval(schedule.billing_interval);
      document.getElementById('confirm-next-billing-date').textContent = schedule.next_billing_date || 'Not scheduled';
      document.getElementById('confirm-active-subscribers').textContent = schedule.active_subscribers || 0;
      document.getElementById('confirm-auto-billing').textContent = schedule.auto_billing_enabled ? 'Enabled' : 'Disabled';
      document.getElementById('confirm-email-notifications').textContent = schedule.auto_email_enabled ? 'Enabled' : 'Disabled';
      document.getElementById('confirm-sms-notifications').textContent = schedule.auto_sms_enabled ? 'Enabled' : 'Disabled';

      modal.style.display = 'block';
    }
  }

  closeConfirmModal() {
    const modal = document.getElementById('automatic-billing-confirm-modal');
    if (modal) {
      modal.style.display = 'none';
    }
  }

  confirmRunSchedule(productId) {
    console.log('confirmRunSchedule called with productId:', productId);
    this.closeConfirmModal();
    this.runScheduleNow(productId);
  }

  async runScheduleNow(productId) {
    this.showLoading(true);

    try {
      const formData = new FormData();
      formData.append('product_id', productId);
      formData.append('action', 'run_manual_billing');

      console.log('Running billing for product_id:', productId);

      const response = await fetch('xd-modules/billing-module/backend/run-manual-billing.php', {
        method: 'POST',
        body: formData
      });

      const data = await response.json();

      if (data.success) {
        this.showNotification(`Billing completed: ${data.message}`, 'success');
        this.refreshData();
      } else {
        this.showNotification(data.message, 'error');
      }
    } catch (error) {
      this.showNotification('Error running billing', 'error');
      console.error('Error running billing:', error);
    } finally {
      this.showLoading(false);
    }
  }

  showBulkRunConfirmation() {
    if (this.schedules.length === 0) {
      this.showNotification('No billing schedules found', 'error');
      return;
    }

    const modal = document.getElementById('automatic-billing-bulk-confirm-modal');
    
    if (modal) {
      const totalSchedules = this.schedules.length;
      const totalSubscribers = this.schedules.reduce((sum, schedule) => sum + (schedule.active_subscribers || 0), 0);
      const dueSchedules = this.schedules.filter(schedule => {
        if (!schedule.next_billing_date) return false;
        const nextDate = new Date(schedule.next_billing_date);
        const today = new Date();
        return nextDate <= today;
      }).length;

      document.getElementById('bulk-confirm-total-schedules').textContent = totalSchedules;
      document.getElementById('bulk-confirm-total-subscribers').textContent = totalSubscribers;
      document.getElementById('bulk-confirm-due-schedules').textContent = dueSchedules;

      const productsList = document.getElementById('bulk-confirm-products-list');
      productsList.innerHTML = this.schedules.map(schedule => `
        <div class="automatic-billing-confirm-product-item">
          <span class="automatic-billing-confirm-product-name">${schedule.product_name}</span>
          <span class="automatic-billing-confirm-product-details">
            ${schedule.active_subscribers || 0} subscribers • ${this.formatInterval(schedule.billing_interval)}
          </span>
        </div>
      `).join('');

      modal.style.display = 'block';
    }
  }

  closeBulkConfirmModal() {
    const modal = document.getElementById('automatic-billing-bulk-confirm-modal');
    if (modal) {
      modal.style.display = 'none';
    }
  }

  confirmBulkRunSchedule() {
    this.closeBulkConfirmModal();
    this.runScheduledBilling();
  }

  async runScheduledBilling() {
    this.showLoading(true);

    try {
      const formData = new FormData();
      formData.append('run_type', 'auto');

      const response = await fetch('xd-modules/billing-module/backend/run-scheduled-billing.php', {
        method: 'POST',
        body: formData
      });

      const data = await response.json();

      if (data.success) {
        this.showNotification(`Scheduled billing completed: ${data.message}`, 'success');
        this.refreshData();
      } else {
        this.showNotification(data.message, 'error');
      }
    } catch (error) {
      this.showNotification('Error running scheduled billing', 'error');
    } finally {
      this.showLoading(false);
    }
  }

  refreshData() {
    this.loadSchedules();
    this.loadProducts();
    if (this.currentView === 'history') {
      this.loadBillingHistory();
    }
  }

  formatInterval(interval) {
    const intervals = {
      'daily': 'Daily',
      'weekly': 'Weekly',
      'monthly': 'Monthly',
      'quarterly': 'Quarterly',
      'yearly': 'Yearly'
    };
    return intervals[interval] || interval;
  }

  showNotification(message, type = 'info') {
    const notification = document.getElementById('bottom_toast345');
    if (!notification) {
      this.createFallbackNotification(message, type);
      return;
    }
    
    notification.style.position = 'fixed';
    notification.style.bottom = '0';
    notification.style.left = '0';
    notification.style.right = '0';
    notification.style.zIndex = '999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999';
    
    notification.textContent = message;
    notification.className = `bottom_toast345 ${type}`;
    notification.style.display = 'block';
    
    const hideTime = type === 'error' ? 8000 : 5000;
    setTimeout(() => {
      this.hideNotification();
    }, hideTime);
  }

  createFallbackNotification(message, type) {
    const tempNotification = document.createElement('div');
    tempNotification.textContent = message;
    tempNotification.style.cssText = `
      position: fixed;
      bottom: 20px;
      left: 50%;
      transform: translateX(-50%);
      background-color: ${type === 'error' ? '#f44336' : type === 'success' ? '#4CAF50' : '#2196F3'};
      color: white;
      padding: 12px 24px;
      border-radius: 4px;
      z-index: 99999;
      font-family: Arial, sans-serif;
      font-size: 14px;
    `;
    
    document.body.appendChild(tempNotification);
    
    setTimeout(() => {
      if (tempNotification.parentNode) {
        tempNotification.parentNode.removeChild(tempNotification);
      }
    }, 5000);
  }

  hideNotification() {
    const notification = document.getElementById('bottom_toast345');
    if (notification) {
      notification.style.display = 'none';
    }
  }

  showLoading(show) {
    const loadingElement = document.getElementById('automatic-billing-loading');
    if (loadingElement) {
      loadingElement.style.display = show ? 'flex' : 'none';
    }
  }

  editSchedule(productId) {
    const schedule = this.schedules.find(s => s.product_id == productId);
    if (!schedule) {
      this.showNotification('Schedule not found', 'error');
      return;
    }

    this.editingScheduleId = productId;
    const modal = document.getElementById('automatic-billing-modal');
    const modalTitle = document.getElementById('automatic-billing-modal-title');
    
    if (modal && modalTitle) {
      document.getElementById('automatic-billing-product-select').value = schedule.product_id;
      document.getElementById('automatic-billing-interval-select').value = schedule.billing_interval;
      document.getElementById('next-automatic-billing-date').value = schedule.next_billing_date || '';
      document.getElementById('automatic-billing-enabled').checked = schedule.auto_billing_enabled;
      document.getElementById('automatic-email-enabled').checked = schedule.auto_email_enabled;
      document.getElementById('automatic-sms-enabled').checked = schedule.auto_sms_enabled;
      modalTitle.innerHTML = `<i class="fas fa-edit"></i> Edit Billing Schedule - ${schedule.product_name}`;
      modal.style.display = 'block';
    }
  }

  attachProductSelectListener() {
    const productSelect = document.getElementById('automatic-billing-product-select');
    if (productSelect) {
      productSelect.addEventListener('change', (e) => {
        this.handleProductSelectChange(e.target.value);
      });
    }
  }

  handleProductSelectChange(productId) {
    if (productId) {
      const existingSchedule = this.schedules.find(s => s.product_id == productId);
      if (existingSchedule && !this.editingScheduleId) {
        document.getElementById('automatic-billing-interval-select').value = existingSchedule.billing_interval;
        document.getElementById('automatic-billing-enabled').checked = existingSchedule.auto_billing_enabled;
        this.showNotification('This product already has a schedule. Updating will modify existing schedule.', 'info');
      }
    }
  }

  updateDashboardHeaderTitle(view) {
    const select = document.querySelector('.billingDashBoardHeaderSelectKey101');
    if (select) {
      if (view === 'history') {
        select.options[0].text = 'Billing History';
      } else {
        select.options[0].text = 'Billing Schedules';
      }
    }
  }
}

function billingDashBoardHeader(){
  let today = new Date().toISOString().slice(0, 10);
  $('#page-options').empty(); 
  $('#page-options').append('<select class="billingDashBoardHeaderSelectKey101" style="border:none;outline:none;background-color:transparent;text-transform:uppercase;color:gray;"><option value="billingDashBoardHeader">Billing System</option></select> <span id="record01" class="topNavTotals"></span>');
  $('#right_nav_options0').empty();
  $('#right_nav_options0').append('<div class="top-right-navigation-icons101 w3-hide-small"><input type="checkbox" class="checkThemAll fontx" style="margin-left:2.5px;margin-bottom:5px;"/></div>');
  $('#right_nav_options0').append('<div class="w3-hide-small mainPrintBtn102  w3-hide-small top-right-navigation-icons101"><span class="fontx fx-printer"></span></div>');
  $('#right_nav_options0').append('<div id="exportbillingDashBoard" class="w3-hide-small top-right-navigation-icons101"><span class="fontx fx-share-2"></span></div>');
  $('#right_nav_options0').append('<div  class="downloadData w3-hide-small top-right-navigation-icons101"><span class="fontx fx-download"></span></div>');
  
  $('#sub_top_nav').empty(); 
  $('#sub_top_nav').append('<div id="searchContainer0" class="w3-col l12 m12 s12 w3-center"><div class="w3-col l1 m1 s1 w3-center w3-hide-small" style="padding-top: 5px;"><span class="fontx fx-search" style="color:gray;font-size:1.5em"></span></div><div class="w3-col l4 m4 s7 w3-center"><input id="searchActiveBilling" class="automatic-billing-search-input" placeholder="Search billing history by product, status, run by, date, amount..." style="font-weight:400;outline:none;color:black;border:none;border-bottom:none;width:100%;height:30px;background-color:transparent;padding-left:7px;"/></div><div class="w3-col l2 m2 s2 w3-center" style="padding-top:5px;"><span style="float:right"><span class="fontx fx-pie-chart" style="color:var(--primary-color);margin-top:5px;"></span><select id="revenueReportDisplayType" class="reduceSizeOnSmallScreen"  style="background-color:transparent;border:none;outline:none;color:gray;"><option value="Charts">Charts</option><option value="Normal">Normal</option><option value="Display All">Display All</option><option value="Map">Map</option></select></span><div/></div><div class="w3-col l5 m5 s3 w3-center" style="padding-left:7px;padding-top:3px;color:gray;background-color:;"><span style="float:right;"><input type="date" value="'+today+'" id="revenueReportStartDate" class="reduceSizeOnSmallScreen" style="color:var(--primary-color);border:none;outline:none;font-weight:100;background-color:transparent;"/><input type="date" value="'+today+'" id="revenueReportEndDate" class="reduceSizeOnSmallScreen" style="color:var(--primary-color);border:none;outline:none;font-weight:100;background-color:transparent;"/></span></div>');
}


$(document).on('click', '.billingDashBoard', function(event) {
  window.automaticBilling = new AutomaticBilling();
  window.automaticBilling.initialize();
  billingDashBoardHeader();
});

$(document).on('click', '.billingSchedules', function(event) {
  window.automaticBilling = new AutomaticBilling();
  window.automaticBilling.initialize();
  billingDashBoardHeader();
});



//// HORIZONTAL SCROLL BAR////
