<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PATCH, PUT, DELETE, OPTIONS');
header('Content-Type: application/json');

session_start();
require_once '../../../xd-assets/backend/config/dbconfig.php';

$runType = filter_input(INPUT_POST, 'run_type');
$userId = $_SESSION["userid"] ?? 0;
$username = $_SESSION["username"] ?? 'System';
$orgId = $_SESSION["orgid"] ?? '100';

function generateInvoiceNumber($conn, $orgId) {
    $getLastInvoiceQuery = "SELECT b_id, invoice_no FROM billing_subscription WHERE org_id = ? ORDER BY b_id DESC LIMIT 1";
    $getLastInvoice = $conn->prepare($getLastInvoiceQuery);
    $getLastInvoice->execute([$orgId]);
    
    if ($getLastInvoice->rowCount() == 0) {
        $invoiceNumber = 1;
    } else {
        $row = $getLastInvoice->fetch(PDO::FETCH_ASSOC);
        $last_invoice_no = $row['invoice_no'];
        $invoiceNumber = ++$last_invoice_no;
    }
    return $invoiceNumber;
}

try {
    $conn->beginTransaction();
    
    // Get products that are due for billing TODAY from billing_schedules
    $productsQuery = "SELECT 
                        bs.id as schedule_id,
                        bs.product_id,
                        p.name as product_name,
                        p.price,
                        p.product_code,
                        p.category,
                        p.category_code,
                        bs.billing_interval,
                        bs.next_billing_date
                      FROM billing_schedules bs
                      INNER JOIN tblproduct p ON bs.product_id = p.id
                      WHERE bs.status = 'active' 
                      AND bs.auto_billing_enabled = 1
                      AND bs.next_billing_date <= CURDATE()
                      AND p.status = 'active'
                      AND bs.org_id = ?";
    
    $productsStmt = $conn->prepare($productsQuery);
    $productsStmt->execute([$orgId]);
    $products = $productsStmt->fetchAll(PDO::FETCH_ASSOC);
    
    error_log("DEBUG: Found " . count($products) . " scheduled products due for billing");
    
    $totalRuns = 0;
    $successfulRuns = 0;
    $failedRuns = 0;
    $results = [];
    
    foreach ($products as $product) {
        try {
            $billingResult = runRecurringBilling($conn, $product, $orgId, $userId);
            
            if ($billingResult['success']) {
                $successfulRuns++;
                
                // Update next billing date for successful runs
                updateNextBillingDate($conn, $product['schedule_id'], $product['billing_interval']);
                
                $results[] = [
                    'product' => $product['product_name'],
                    'schedule_id' => $product['schedule_id'],
                    'status' => 'success',
                    'invoices' => $billingResult['total_invoices'],
                    'amount' => $billingResult['total_amount'],
                    'clients' => $billingResult['total_clients']
                ];
                
                error_log("SUCCESS: Billed {$billingResult['total_clients']} existing clients for {$product['product_name']}");
            } else {
                $failedRuns++;
                $results[] = [
                    'product' => $product['product_name'],
                    'schedule_id' => $product['schedule_id'],
                    'status' => 'failed',
                    'error' => $billingResult['message']
                ];
                error_log("FAILED: {$product['product_name']} - {$billingResult['message']}");
            }
            
            $totalRuns++;
            
        } catch (Exception $e) {
            $failedRuns++;
            $results[] = [
                'product' => $product['product_name'],
                'schedule_id' => $product['schedule_id'],
                'status' => 'failed',
                'error' => $e->getMessage()
            ];
            error_log("ERROR: {$product['product_name']} - {$e->getMessage()}");
        }
    }
    
    // Record billing run
    $runId = null;
    if ($totalRuns > 0) {
        $runQuery = "INSERT INTO billing_runs (
            run_type, run_date, run_time, total_clients_billed, 
            total_invoices_generated, total_amount_billed, 
            successful_invoices, failed_invoices, status,
            run_by_user_id, run_by_user_name, org_id, summary_details
        ) VALUES (?, CURDATE(), CURTIME(), ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $totalClients = array_sum(array_column(array_filter($results, function($r) { 
            return $r['status'] === 'success'; 
        }), 'clients'));
        
        $totalInvoices = array_sum(array_column(array_filter($results, function($r) { 
            return $r['status'] === 'success'; 
        }), 'invoices'));
        
        $totalAmount = array_sum(array_column(array_filter($results, function($r) { 
            return $r['status'] === 'success'; 
        }), 'amount'));
        
        $runStmt = $conn->prepare($runQuery);
        $runStmt->execute([
            $runType, 
            $totalClients, 
            $totalInvoices, 
            $totalAmount,
            $successfulRuns, 
            $failedRuns, 
            'completed',
            $userId, 
            $username, 
            $orgId, 
            json_encode($results)
        ]);
        
        $runId = $conn->lastInsertId();
    }
    
    $conn->commit();
    
    echo json_encode([
        'success' => true,
        'run_id' => $runId,
        'total_products_processed' => $totalRuns,
        'successful_runs' => $successfulRuns,
        'failed_runs' => $failedRuns,
        'results' => $results,
        'message' => "Processed $totalRuns scheduled products ($successfulRuns successful, $failedRuns failed)"
    ]);
    
} catch (Exception $e) {
    $conn->rollBack();
    error_log("BILLING RUN FAILED: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Scheduled billing run failed: ' . $e->getMessage()
    ]);
}

function runRecurringBilling($conn, $schedule, $orgId, $userId) {
    try {
        $productId = $schedule['product_id'];
        $scheduleId = $schedule['schedule_id'];
        
        // CORRECTED: Get ONLY CLIENTS WHO ALREADY HAVE THIS PRODUCT in billing_subscription
        $clientsQuery = "SELECT DISTINCT 
                            bs.client_id,
                            bs.client_name,
                            bs.phone_contact,
                            bs.email_contact,
                            bs.company_name,
                            bs.delivery_address,
                            bs.plot_number,
                            bs.area,
                            bs.nrc
                         FROM billing_subscription bs
                         WHERE bs.product_id = ? 
                         AND bs.org_id = ?
                         AND bs.b_status = 'Active'
                         AND bs.document_type = 'INVOICE'
                         AND NOT EXISTS (
                             SELECT 1 FROM billing_subscription bs2 
                             WHERE bs2.client_id = bs.client_id 
                             AND bs2.product_id = bs.product_id
                             AND bs2.document_type = 'INVOICE'
                             AND bs2.b_status = 'Active'
                             AND DATE(bs2.timestamp) = CURDATE()  // Prevent duplicate billing today
                         )";
        
        $clientsStmt = $conn->prepare($clientsQuery);
        $clientsStmt->execute([$productId, $orgId]);
        $clients = $clientsStmt->fetchAll(PDO::FETCH_ASSOC);
        
        error_log("DEBUG: Found " . count($clients) . " existing subscription clients for product: {$schedule['product_name']}");
        
        if (empty($clients)) {
            return [
                'success' => true, 
                'total_invoices' => 0, 
                'total_amount' => 0, 
                'total_clients' => 0,
                'message' => 'No existing subscription clients found or invoices already generated today'
            ];
        }
        
        $totalInvoices = 0;
        $totalAmount = 0;
        $failedClients = 0;
        
        foreach ($clients as $client) {
            try {
                $invoiceNumber = generateInvoiceNumber($conn, $orgId);
                $dueDate = date('Y-m-d', strtotime('+30 days'));
                $dueTime = date('H:i:s');
                
                $insertQuery = "INSERT INTO billing_subscription (
                    company_name, client_name, client_id, 
                    product_name, product_id, product_code, 
                    units, unit_price, total_charge, 
                    product_category, product_category_code, billing_interval,
                    balance_bf, paid, balance, balance_cf, invoice_no, 
                    due_date, due_time, b_status, phone_contact, email_contact, 
                    delivery_address, plot_number, area, nrc, user_id, document_type,
                    billing_schedule_id, org_name, org_id, session_id, timestamp
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP)";
                
                $insertStmt = $conn->prepare($insertQuery);
                
                $totalCharge = $schedule['price'] ?? 0;
                
                $insertStmt->execute([
                    $client['company_name'] ?? $client['client_name'],
                    $client['client_name'],
                    $client['client_id'],
                    $schedule['product_name'],
                    $productId,
                    $schedule['product_code'] ?? '',
                    1, // units
                    $schedule['price'] ?? 0,
                    $totalCharge,
                    $schedule['category'] ?? '',
                    $schedule['category_code'] ?? '',
                    $schedule['billing_interval'] ?? '',
                    0, // balance_bf
                    0, // paid
                    $totalCharge, // balance
                    $totalCharge, // balance_cf
                    $invoiceNumber,
                    $dueDate,
                    $dueTime,
                    'Active',
                    $client['phone_contact'],
                    $client['email_contact'],
                    $client['delivery_address'] ?? '',
                    $client['plot_number'] ?? '',
                    $client['area'] ?? '',
                    $client['nrc'] ?? '',
                    $userId,
                    'INVOICE',
                    $scheduleId,
                    'Lukulu Town Council',
                    $orgId,
                    $userId
                ]);
                
                $totalInvoices++;
                $totalAmount += $totalCharge;
                
            } catch (Exception $e) {
                $failedClients++;
                error_log("Failed to bill existing client {$client['client_id']}: " . $e->getMessage());
            }
        }
        
        return [
            'success' => true,
            'total_invoices' => $totalInvoices,
            'total_amount' => $totalAmount,
            'total_clients' => count($clients),
            'message' => "Recurring billing completed: $totalInvoices invoices generated for existing clients ($failedClients failed)"
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Recurring billing process failed: ' . $e->getMessage()
        ];
    }
}

function updateNextBillingDate($conn, $scheduleId, $billingInterval) {
    $nextDate = null;
    
    switch ($billingInterval) {
        case 'daily':
            $nextDate = date('Y-m-d', strtotime('+1 day'));
            break;
        case 'weekly':
            $nextDate = date('Y-m-d', strtotime('+1 week'));
            break;
        case 'monthly':
            $nextDate = date('Y-m-d', strtotime('+1 month'));
            break;
        case 'quarterly':
            $nextDate = date('Y-m-d', strtotime('+3 months'));
            break;
        case 'yearly':
            $nextDate = date('Y-m-d', strtotime('+1 year'));
            break;
        case '3years':
            $nextDate = date('Y-m-d', strtotime('+3 years'));
            break;
        case 'Semester':
            $nextDate = date('Y-m-d', strtotime('+6 months'));
            break;
        default:
            $nextDate = date('Y-m-d', strtotime('+1 month'));
    }
    
    $updateQuery = "UPDATE billing_schedules SET last_billing_date = CURDATE(), next_billing_date = ? WHERE id = ?";
    $updateStmt = $conn->prepare($updateQuery);
    $updateStmt->execute([$nextDate, $scheduleId]);
    
    error_log("DEBUG: Updated schedule $scheduleId - next billing: $nextDate");
}
?>