<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PATCH, PUT, DELETE, OPTIONS');
header('Content-Type: application/json');

session_start();
require_once '../../../xd-assets/backend/config/dbconfig.php';

if ($_POST['action'] !== 'run_manual_billing') {
    echo json_encode(['success' => false, 'message' => 'Invalid action']);
    exit;
}

$productId = filter_input(INPUT_POST, 'product_id', FILTER_VALIDATE_INT);
$userId = $_SESSION["userid"] ?? 0;
$username = $_SESSION["username"] ?? 'System';
$orgId = $_SESSION["orgid"] ?? '100';

if (!$productId) {
    echo json_encode(['success' => false, 'message' => 'Invalid product ID']);
    exit;
}

// Function to generate invoice number - SAME METHOD AS createInvoice.php
function generateInvoiceNumber($conn, $orgId) {
    $getLastInvoiceQuery = "SELECT b_id, invoice_no FROM billing_subscription WHERE org_id = ? ORDER BY b_id DESC LIMIT 1";
    $getLastInvoice = $conn->prepare($getLastInvoiceQuery);
    $getLastInvoice->execute([$orgId]);
    
    if ($getLastInvoice->rowCount() == 0) {
        $invoiceNumber = 1;
    } else {
        $row = $getLastInvoice->fetch(PDO::FETCH_ASSOC);
        $last_invoice_no = $row['invoice_no'];
        $invoiceNumber = ++$last_invoice_no;
    }
    return $invoiceNumber;
}

try {
    $conn->beginTransaction();
    
    // Get product details
    $productQuery = "SELECT * FROM tblproduct WHERE id = ? AND status = 'active'";
    $productStmt = $conn->prepare($productQuery);
    $productStmt->execute([$productId]);
    $product = $productStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$product) {
        throw new Exception('Product not found or inactive');
    }
    
    // Get clients who ALREADY HAVE this product in billing_subscription and are active
    $clientsQuery = "SELECT DISTINCT 
                        bs.client_id,
                        bs.client_name,
                        bs.phone_contact,
                        bs.email_contact,
                        bs.company_name,
                        bs.delivery_address,
                        bs.plot_number,
                        bs.area,
                        bs.nrc
                     FROM billing_subscription bs
                     WHERE bs.product_id = ? 
                     AND bs.org_id = ?
                     AND bs.b_status = 'Active'
                     AND bs.document_type = 'INVOICE'
                     AND NOT EXISTS (
                         SELECT 1 FROM billing_subscription bs2 
                         WHERE bs2.client_id = bs.client_id 
                         AND bs2.product_id = bs.product_id
                         AND bs2.document_type = 'INVOICE'
                         AND bs2.b_status = 'Active'
                         AND DATE(bs2.timestamp) = CURDATE()
                     )";
    
    $clientsStmt = $conn->prepare($clientsQuery);
    $clientsStmt->execute([$productId, $orgId]);
    $clients = $clientsStmt->fetchAll(PDO::FETCH_ASSOC);
    
    $totalInvoices = 0;
    $totalAmount = 0;
    $failedClients = 0;
    
    // Generate NEW invoices for each client who already has this product
    foreach ($clients as $client) {
        try {
            $invoiceNumber = generateInvoiceNumber($conn, $orgId);
            $dueDate = date('Y-m-d', strtotime('+30 days'));
            $dueTime = date('H:i:s');
            
            $insertQuery = "INSERT INTO billing_subscription (
                company_name, company_registration_number, tpin, business_category,
                client_name, client_id, product_name, product_id, product_code, 
                units, unit_price, total_charge, product_category, product_category_code,
                billing_interval, balance_bf, paid, balance, balance_cf, invoice_no, 
                due_date, due_time, b_status, phone_contact, email_contact, 
                delivery_address, plot_number, area, nrc, user_id, document_type,
                org_name, org_id, session_id
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $insertStmt = $conn->prepare($insertQuery);
            $totalCharge = $product['price'] * 1; // Assuming 1 unit
            
            $insertStmt->execute([
                $client['company_name'] ?? $client['client_name'],
                '', // company_registration_number
                '', // tpin
                '', // business_category
                $client['client_name'],
                $client['client_id'],
                $product['name'],
                $product['id'],
                $product['product_code'],
                1, // units
                $product['price'],
                $totalCharge,
                $product['category'],
                $product['category_code'] ?? '',
                $product['charge_rate'],
                0, // balance_bf
                0, // paid
                $totalCharge, // balance
                $totalCharge, // balance_cf
                $invoiceNumber,
                $dueDate,
                $dueTime,
                'Active',
                $client['phone_contact'],
                $client['email_contact'],
                $client['delivery_address'] ?? '',
                $client['plot_number'] ?? '',
                $client['area'] ?? '',
                $client['nrc'] ?? '',
                $userId,
                'INVOICE',
                'Lukulu Town Council', // org_name
                $orgId,
                $userId
            ]);
            
            $totalInvoices++;
            $totalAmount += $totalCharge;
            
        } catch (Exception $e) {
            $failedClients++;
            error_log("Failed to bill client {$client['client_id']}: " . $e->getMessage());
        }
    }
    
    // Record billing run in billing_runs table
    $runId = null;
    if ($totalInvoices > 0) { // Only create billing run if invoices were created
        try {
            // FIXED: Correct column order according to your table structure
            $runQuery = "INSERT INTO billing_runs (
                    product_id, product_name, run_type, run_date, run_time,
                    total_clients_billed, total_invoices_generated, total_amount_billed, 
                    successful_invoices, failed_invoices, status, run_by_user_id, 
                    run_by_user_name, org_id
                ) VALUES (?, ?, ?, CURDATE(), CURTIME(), ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $runStmt = $conn->prepare($runQuery);
            $runStmt->execute([
                $productId,
                $product['name'],
                'manual',  
                count($clients), 
                $totalInvoices, 
                $totalAmount,
                $totalInvoices, 
                $failedClients, 
                'completed',
                $userId, 
                $username, 
                $orgId
            ]);
            
            $runId = $conn->lastInsertId();
            
        } catch (Exception $e) {
            error_log("Billing runs table error: " . $e->getMessage());
        }
    }
    
    $conn->commit();
    
    echo json_encode([
        'success' => true,
        'run_id' => $runId,
        'total_clients_billed' => count($clients),
        'total_invoices' => $totalInvoices,
        'total_amount' => $totalAmount,
        'failed_clients' => $failedClients,
        'message' => "Successfully generated $totalInvoices invoices for " . $product['name'] . " (Existing customers only)"
    ]);
    
} catch (Exception $e) {
    $conn->rollBack();
    echo json_encode([
        'success' => false,
        'message' => 'Billing run failed: ' . $e->getMessage()
    ]);
}
?>