<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PATCH, PUT, DELETE, OPTIONS');
header('Content-Type: application/json');

session_start();
require_once '../../../xd-assets/backend/config/dbconfig.php';

class CustomerStatementsBackend {
    private $conn;
    private $orgid;
    private $orgname;
    private $userid;

    public function __construct($conn) {
        $this->conn = $conn;
        $this->orgid = $_SESSION["orgid"] ?? '100';
        $this->orgname = $_SESSION["orgname"] ?? 'Lukulu Town Council';
        $this->userid = $_SESSION["userid"] ?? null;
    }

    // Generate statement number
    private function generateStatementNumber() {
        try {
            $stmt = $this->conn->prepare("SELECT statement_prefix, next_statement_number FROM statement_settings WHERE org_id = ?");
            $stmt->execute([$this->orgid]);
            $settings = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($settings) {
                $prefix = $settings['statement_prefix'];
                $nextNumber = $settings['next_statement_number'];
                
                // Update next statement number
                $updateStmt = $this->conn->prepare("UPDATE statement_settings SET next_statement_number = next_statement_number + 1 WHERE org_id = ?");
                $updateStmt->execute([$this->orgid]);
                
                return $prefix . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);
            } else {
                // Fallback if no settings found
                return 'STM' . date('Ymd') . rand(1000, 9999);
            }
            
        } catch (Exception $e) {
            // Fallback if table doesn't exist
            return 'STM' . date('YmdHis') . rand(100, 999);
        }
    }

    // CORRECT: Get client details from users table only
    private function getClientDetails($clientId) {
        $stmt = $this->conn->prepare("
            SELECT id, names, email, phone, address, nrc
            FROM users 
            WHERE id = ? AND org_id = ?
        ");
        $stmt->execute([$clientId, $this->orgid]);
        $client = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$client) {
            throw new Exception("Client with ID $clientId not found in users table");
        }

        return $client;
    }

    // Get client's complete transaction history for statement period
    private function getClientTransactionHistory($clientId, $startDate, $endDate) {
        $transactions = [];

        // 1. Get invoices from billing_subscription (charges)
        $invoiceStmt = $this->conn->prepare("
            SELECT 
                b_id as id,
                'invoice' as type,
                due_date as transaction_date,
                invoice_no as document_number,
                CONCAT(product_name, ' - Invoice #', invoice_no) as description,
                total_charge as debit_amount,
                0 as credit_amount,
                balance,
                payment_status,
                'billing_subscription' as source_table,
                timestamp as created_at
            FROM billing_subscription 
            WHERE client_id = ? 
            AND due_date BETWEEN ? AND ?
            AND org_id = ?
            AND status = 'Active'
            ORDER BY due_date, timestamp
        ");
        $invoiceStmt->execute([$clientId, $startDate, $endDate, $this->orgid]);
        $invoices = $invoiceStmt->fetchAll(PDO::FETCH_ASSOC);

        // 2. Get payments from transaction_details (credits)
        $paymentStmt = $this->conn->prepare("
            SELECT 
                id,
                'payment' as type,
                transaction_date,
                receipt_no as document_number,
                CONCAT('Payment Receipt #', receipt_no, ' - ', product_name) as description,
                0 as debit_amount,
                transaction_amount as credit_amount,
                transaction_balance as balance,
                'completed' as payment_status,
                'transaction_details' as source_table,
                transaction_date as created_at
            FROM transaction_details 
            WHERE user_id = ? 
            AND transaction_date BETWEEN ? AND ?
            AND org_id = ?
            AND transaction_status = 'active'
            AND debit_or_credit = 'Credit'
            ORDER BY transaction_date
        ");
        $paymentStmt->execute([$clientId, $startDate, $endDate, $this->orgid]);
        $payments = $paymentStmt->fetchAll(PDO::FETCH_ASSOC);

        // Combine and sort by date
        $allTransactions = array_merge($invoices, $payments);
        usort($allTransactions, function($a, $b) {
            $dateCompare = strtotime($a['transaction_date']) - strtotime($b['transaction_date']);
            if ($dateCompare === 0) {
                return strtotime($a['created_at']) - strtotime($b['created_at']);
            }
            return $dateCompare;
        });

        return $allTransactions;
    }

    // Calculate opening balance (unpaid invoices before start date)
    private function calculateOpeningBalance($clientId, $startDate) {
        $stmt = $this->conn->prepare("
            SELECT COALESCE(SUM(balance), 0) as opening_balance
            FROM billing_subscription 
            WHERE client_id = ? 
            AND due_date < ?
            AND org_id = ?
            AND status = 'Active'
            AND payment_status = 'Pending'
        ");
        $stmt->execute([$clientId, $startDate, $this->orgid]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $result['opening_balance'] ?? 0;
    }

    // Generate customer statement
    public function generateStatement($clientId, $periodStart, $periodEnd, $dueDate = null, $notes = '') {
        try {
            $this->conn->beginTransaction();

            // Get client details - CORRECT: from users table only
            $client = $this->getClientDetails($clientId);
            
            // Calculate opening balance
            $openingBalance = $this->calculateOpeningBalance($clientId, $periodStart);
            
            // Get transactions for period
            $transactions = $this->getClientTransactionHistory($clientId, $periodStart, $periodEnd);
            
            // Calculate totals and running balance
            $totalCharges = 0;
            $totalPayments = 0;
            $runningBalance = $openingBalance;

            foreach ($transactions as &$transaction) {
                $totalCharges += $transaction['debit_amount'];
                $totalPayments += $transaction['credit_amount'];
                $runningBalance += $transaction['debit_amount'] - $transaction['credit_amount'];
                $transaction['running_balance'] = $runningBalance;
            }

            $closingBalance = $runningBalance;
            $amountDue = $closingBalance;

            // Generate statement number and dates
            $statementNumber = $this->generateStatementNumber();
            $statementDate = date('Y-m-d');
            $dueDate = $dueDate ?: date('Y-m-d', strtotime($periodEnd . ' +15 days'));

            // Insert into customer_statements
            $stmt = $this->conn->prepare("
                INSERT INTO customer_statements (
                    statement_number, client_id, client_name, client_email, client_phone,
                    statement_period_start, statement_period_end, statement_date, due_date,
                    opening_balance, total_charges, total_payments, closing_balance,
                    previous_balance, amount_due, status, statement_notes, org_id, org_name, created_by
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft', ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $statementNumber, 
                $clientId, 
                $client['names'], 
                $client['email'] ?? '', 
                $client['phone'] ?? '',
                $periodStart, 
                $periodEnd, 
                $statementDate, 
                $dueDate,
                $openingBalance,
                $totalCharges,
                $totalPayments,
                $closingBalance,
                $openingBalance,
                $amountDue,
                $notes,
                $this->orgid, 
                $this->orgname, 
                $this->userid
            ]);

            $statementId = $this->conn->lastInsertId();

            // Insert statement items
            $itemStmt = $this->conn->prepare("
                INSERT INTO statement_items (
                    statement_id, transaction_date, transaction_type, document_number,
                    description, debit_amount, credit_amount, balance, reference_id,
                    reference_table, org_id
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");

            foreach ($transactions as $transaction) {
                $itemStmt->execute([
                    $statementId,
                    $transaction['transaction_date'],
                    $transaction['type'],
                    $transaction['document_number'],
                    $transaction['description'],
                    $transaction['debit_amount'],
                    $transaction['credit_amount'],
                    $transaction['running_balance'],
                    $transaction['id'],
                    $transaction['source_table'],
                    $this->orgid
                ]);
            }

            $this->conn->commit();

            return [
                'success' => true,
                'statement_id' => $statementId,
                'statement_number' => $statementNumber,
                'message' => 'Statement generated successfully',
                'summary' => [
                    'opening_balance' => $openingBalance,
                    'total_charges' => $totalCharges,
                    'total_payments' => $totalPayments,
                    'closing_balance' => $closingBalance,
                    'transaction_count' => count($transactions)
                ]
            ];

        } catch (Exception $e) {
            $this->conn->rollBack();
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    // Get all statements with filters
    public function getStatements($filters = []) {
        $query = "
            SELECT cs.*, 
                   u.names as client_name,
                   u.email as client_email,
                   u.phone as client_phone
            FROM customer_statements cs
            LEFT JOIN users u ON cs.client_id = u.id
            WHERE cs.org_id = ?
        ";
        
        $params = [$this->orgid];
        
        // Add filters
        if (!empty($filters['status'])) {
            $query .= " AND cs.status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['client_id'])) {
            $query .= " AND cs.client_id = ?";
            $params[] = $filters['client_id'];
        }
        
        if (!empty($filters['search'])) {
            $query .= " AND (cs.statement_number LIKE ? OR cs.client_name LIKE ?)";
            $searchTerm = "%{$filters['search']}%";
            $params[] = $searchTerm;
            $params[] = $searchTerm;
        }
        
        $query .= " ORDER BY cs.statement_date DESC, cs.id DESC LIMIT 100";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute($params);
        
        $statements = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Get item counts for each statement
        foreach ($statements as &$statement) {
            $countStmt = $this->conn->prepare("
                SELECT COUNT(*) as item_count 
                FROM statement_items 
                WHERE statement_id = ?
            ");
            $countStmt->execute([$statement['id']]);
            $countResult = $countStmt->fetch(PDO::FETCH_ASSOC);
            $statement['item_count'] = $countResult['item_count'] ?? 0;
        }
        
        return [
            'success' => true,
            'data' => $statements
        ];
    }

    // Get statement details with items
    public function getStatementDetails($statementId) {
        // Get statement header
        $stmt = $this->conn->prepare("
            SELECT cs.*, 
                   u.names as client_name, 
                   u.email as client_email, 
                   u.phone as client_phone,
                   u.address as client_address,
                   u.nrc as client_nrc
            FROM customer_statements cs
            LEFT JOIN users u ON cs.client_id = u.id
            WHERE cs.id = ? AND cs.org_id = ?
        ");
        $stmt->execute([$statementId, $this->orgid]);
        $statement = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$statement) {
            return ['success' => false, 'message' => 'Statement not found'];
        }

        // Get statement items
        $itemStmt = $this->conn->prepare("
            SELECT * FROM statement_items 
            WHERE statement_id = ? 
            ORDER BY transaction_date, id
        ");
        $itemStmt->execute([$statementId]);
        $items = $itemStmt->fetchAll(PDO::FETCH_ASSOC);

        return [
            'success' => true,
            'statement' => $statement,
            'items' => $items
        ];
    }
}

// Handle API requests
try {
    $conn = new PDO("mysql:host=$servername;dbname=$database", $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $backend = new CustomerStatementsBackend($conn);
    
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        $action = $input['action'] ?? '';
        
        switch ($action) {
            case 'generate_statement':
                $response = $backend->generateStatement(
                    $input['client_id'],
                    $input['period_start'],
                    $input['period_end'],
                    $input['due_date'] ?? null,
                    $input['notes'] ?? ''
                );
                break;
                
            case 'get_statements':
                $response = $backend->getStatements($input['filters'] ?? []);
                break;
                
            case 'get_statement_details':
                $response = $backend->getStatementDetails($input['statement_id']);
                break;
                
            default:
                $response = ['success' => false, 'message' => 'Invalid action'];
        }
        
        echo json_encode($response);
        exit;
    }
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Database connection failed: ' . $e->getMessage()]);
    exit;
}
?> 