<?php
header('Access-Control-Allow-Origin: *');
header('Content-Type: application/json');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

session_start();
require_once '../../../xd-assets/backend/config/dbconfig.php';

// Use your existing $conn from dbconfig.php
// No need to create new connection - it's already created in dbconfig.php

// Get the action from request
$action = $_GET['action'] ?? $_POST['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

// Check session
$user_id = $_SESSION['userid'] ?? null;
$org_id = $_SESSION['orgid'] ?? null;
$org_name = $_SESSION['orgname'] ?? null;

if (empty($user_id) || empty($org_id)) {
    echo json_encode(['status' => 'error', 'message' => 'Authentication required']);
    exit;
}

// Route requests
switch($action) {
    case 'get_dashboard_stats':
        getDashboardStats($conn, $org_id);
        break;
    case 'get_products':
        getProducts($conn, $org_id);
        break;
    case 'get_billing_history':
        getBillingHistory($conn, $org_id);
        break;
    case 'save_settings':
        saveSettings($conn, $org_id, $user_id);
        break;
    case 'get_settings':
        getSettings($conn, $org_id);
        break;
    case 'run_auto_billing':
        runAutoBilling($conn, $org_id, $user_id);
        break;
    case 'create_manual_invoice':
        createManualInvoice($conn, $org_id, $user_id);
        break;
    case 'get_client_billing_preferences':
        getClientBillingPreferences($conn, $org_id);
        break;
    case 'get_recent_activity':
        getRecentActivity($conn, $org_id);
        break;
    default:
        echo json_encode(['status' => 'error', 'message' => 'Invalid action: ' . $action]);
        break;
        // Add to your existing switch statement in billing_system_api.php
case 'run_auto_billing':
    runAutoBilling($conn, $org_id, $user_id);
    break;
case 'get_billing_runs':
    getBillingRuns($conn, $org_id);
    break;
}

// ==================== API FUNCTIONS ====================

/**
 * Get dashboard statistics
 */
function getDashboardStats($conn, $org_id) {
    try {
        // Total Revenue (sum of all paid amounts)
        $stmt = $conn->prepare("
            SELECT COALESCE(SUM(paid), 0) as total_revenue 
            FROM billing_subscription 
            WHERE org_id = ? AND status = 'Active'
        ");
        $stmt->execute([$org_id]);
        $total_revenue = $stmt->fetch(PDO::FETCH_ASSOC)['total_revenue'];

        // Pending Invoices
        $stmt = $conn->prepare("
            SELECT COUNT(*) as pending_invoices 
            FROM billing_subscription 
            WHERE org_id = ? AND payment_status = 'Pending' AND status = 'Active'
        ");
        $stmt->execute([$org_id]);
        $pending_invoices = $stmt->fetch(PDO::FETCH_ASSOC)['pending_invoices'];

        // Active Subscriptions
        $stmt = $conn->prepare("
            SELECT COUNT(DISTINCT client_id) as active_subscriptions 
            FROM billing_subscription 
            WHERE org_id = ? AND status = 'Active'
        ");
        $stmt->execute([$org_id]);
        $active_subscriptions = $stmt->fetch(PDO::FETCH_ASSOC)['active_subscriptions'];

        // Today's Collections
        $today = date('Y-m-d');
        $stmt = $conn->prepare("
            SELECT COALESCE(SUM(transaction_amount), 0) as today_collections 
            FROM transaction_details 
            WHERE org_id = ? AND transaction_date = ? AND transaction_status = 'active'
        ");
        $stmt->execute([$org_id, $today]);
        $today_collections = $stmt->fetch(PDO::FETCH_ASSOC)['today_collections'];

        echo json_encode([
            'status' => 'success',
            'total_revenue' => number_format($total_revenue, 2),
            'pending_invoices' => $pending_invoices,
            'active_subscriptions' => $active_subscriptions,
            'today_collections' => number_format($today_collections, 2)
        ]);

    } catch(PDOException $e) {
        echo json_encode(['status' => 'error', 'message' => 'Failed to fetch dashboard stats: ' . $e->getMessage()]);
    }
}

/**
 * Get all products for billing
 */
function getProducts($conn, $org_id) {
    try {
        $stmt = $conn->prepare("
            SELECT p.id, p.name, p.product_code, p.price, p.charge_rate, 
                   p.category, p.status, p.description,
                   bs.billing_interval, bs.auto_billing_enabled,
                   pbr.grace_period_days
            FROM tblproduct p
            LEFT JOIN billing_schedules bs ON p.id = bs.product_id AND bs.org_id = ?
            LEFT JOIN product_billing_rules pbr ON p.id = pbr.product_id AND pbr.org_id = ?
            WHERE p.status = 'active'
            ORDER BY p.name
        ");
        $stmt->execute([$org_id, $org_id]);
        $products = $stmt->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'status' => 'success',
            'products' => $products
        ]);

    } catch(PDOException $e) {
        echo json_encode(['status' => 'error', 'message' => 'Failed to fetch products: ' . $e->getMessage()]);
    }
}

/**
 * Get billing history with pagination and search
 */
function getBillingHistory($conn, $org_id) {
    try {
        $search = $_GET['search'] ?? '';
        $page = $_GET['page'] ?? 1;
        $limit = $_GET['limit'] ?? 20;
        $offset = ($page - 1) * $limit;

        $query = "
            SELECT bs.b_id, bs.invoice_no, bs.client_name, bs.product_name, 
                   bs.total_charge, bs.paid, bs.balance, bs.payment_status,
                   bs.due_date, bs.transaction_date, bs.b_status,
                   u.phone as client_phone, u.email as client_email
            FROM billing_subscription bs
            LEFT JOIN users u ON bs.client_id = u.id
            WHERE bs.org_id = ?
        ";

        $params = [$org_id];

        if (!empty($search)) {
            $query .= " AND (bs.client_name LIKE ? OR bs.product_name LIKE ? OR bs.invoice_no LIKE ?)";
            $searchTerm = "%$search%";
            $params[] = $searchTerm;
            $params[] = $searchTerm;
            $params[] = $searchTerm;
        }

        $query .= " ORDER BY bs.transaction_date DESC, bs.invoice_no DESC LIMIT ? OFFSET ?";
        $params[] = (int)$limit;
        $params[] = (int)$offset;

        $stmt = $conn->prepare($query);
        $stmt->execute($params);
        $history = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Get total count for pagination
        $countQuery = "
            SELECT COUNT(*) as total 
            FROM billing_subscription bs 
            WHERE bs.org_id = ?
        ";
        $countParams = [$org_id];

        if (!empty($search)) {
            $countQuery .= " AND (bs.client_name LIKE ? OR bs.product_name LIKE ? OR bs.invoice_no LIKE ?)";
            $countParams[] = $searchTerm;
            $countParams[] = $searchTerm;
            $countParams[] = $searchTerm;
        }

        $stmt = $conn->prepare($countQuery);
        $stmt->execute($countParams);
        $total = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

        echo json_encode([
            'status' => 'success',
            'history' => $history,
            'pagination' => [
                'page' => (int)$page,
                'limit' => (int)$limit,
                'total' => (int)$total,
                'pages' => ceil($total / $limit)
            ]
        ]);

    } catch(PDOException $e) {
        echo json_encode(['status' => 'error', 'message' => 'Failed to fetch billing history: ' . $e->getMessage()]);
    }
}

/**
 * Get billing settings
 */
function getSettings($conn, $org_id) {
    try {
        $stmt = $conn->prepare("
            SELECT setting_name, setting_value, description 
            FROM billing_settings 
            WHERE org_id = ?
        ");
        $stmt->execute([$org_id]);
        $settings = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Convert to key-value pairs for easier frontend use
        $settingsMap = [];
        foreach ($settings as $setting) {
            $settingsMap[$setting['setting_name']] = $setting['setting_value'];
        }

        echo json_encode([
            'status' => 'success',
            'settings' => $settingsMap
        ]);

    } catch(PDOException $e) {
        echo json_encode(['status' => 'error', 'message' => 'Failed to fetch settings: ' . $e->getMessage()]);
    }
}

/**
 * Save billing settings
 */
function saveSettings($conn, $org_id, $user_id) {
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        foreach ($input as $setting_name => $setting_value) {
            $stmt = $conn->prepare("
                INSERT INTO billing_settings (setting_name, setting_value, org_id, updated_by) 
                VALUES (?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE 
                setting_value = VALUES(setting_value), 
                updated_by = VALUES(updated_by),
                updated_at = CURRENT_TIMESTAMP
            ");
            $stmt->execute([$setting_name, $setting_value, $org_id, $user_id]);
        }

        echo json_encode(['status' => 'success', 'message' => 'Settings saved successfully']);

    } catch(PDOException $e) {
        echo json_encode(['status' => 'error', 'message' => 'Failed to save settings: ' . $e->getMessage()]);
    }
}

/**
 * Get recent activity for dashboard
 */
function getRecentActivity($conn, $org_id) {
    try {
        $stmt = $conn->prepare("
            SELECT td.id, td.cfname as client_name, td.product_name, td.transaction_amount, 
                   td.transaction_date, td.transaction_time, td.transaction_status,
                   'payment' as type
            FROM transaction_details td
            WHERE td.org_id = ?
            UNION
            SELECT bs.b_id as id, bs.client_name, bs.product_name, bs.total_charge as transaction_amount,
                   bs.transaction_date, bs.transaction_time, bs.payment_status as transaction_status,
                   'invoice' as type
            FROM billing_subscription bs
            WHERE bs.org_id = ?
            ORDER BY transaction_date DESC, transaction_time DESC 
            LIMIT 10
        ");
        $stmt->execute([$org_id, $org_id]);
        $recent_activity = $stmt->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'status' => 'success',
            'recent_activity' => $recent_activity
        ]);

    } catch(PDOException $e) {
        echo json_encode(['status' => 'error', 'message' => 'Failed to fetch recent activity: ' . $e->getMessage()]);
    }
}

// Placeholder functions for other endpoints
 /**
 * Run auto billing
 */
 
/**
 * Run auto billing - COMPLETE VERSION
 */
function runAutoBilling($conn, $org_id, $user_id) {
    $run_id = null;
    
    try {
        // Start transaction
        $conn->beginTransaction();
        
        // Get billing settings
        $stmt = $conn->prepare("
            SELECT setting_name, setting_value 
            FROM billing_settings 
            WHERE org_id = ?
        ");
        $stmt->execute([$org_id]);
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        $auto_billing_enabled = $settings['auto_billing_enabled'] ?? '0';
        $billing_day = $settings['billing_day_of_month'] ?? '1';
        $grace_period = $settings['grace_period_days'] ?? '7';
        $late_fee_percentage = $settings['late_fee_percentage'] ?? '5';
        
       // In your runAutoBilling function, temporarily comment out this check:
// if ($auto_billing_enabled != '1') {
//     throw new Exception("Auto billing is not enabled in system settings");
// }
        
        // Check if it's billing day (if monthly billing)
        $current_day = date('j');
        if ($billing_day != $current_day) {
            throw new Exception("Today is not the scheduled billing day (Day $billing_day). Current day: $current_day");
        }
        
        // Create billing run record
        $stmt = $conn->prepare("
            INSERT INTO billing_runs (run_type, run_date, run_time, status, run_by_user_id, org_id, org_name)
            VALUES ('auto', CURDATE(), CURTIME(), 'running', ?, ?, ?)
        ");
        $stmt->execute([$user_id, $org_id, 'Lukulu Town Council']);
        $run_id = $conn->lastInsertId();
        
        // Get all active billing schedules due today
        $stmt = $conn->prepare("
            SELECT bs.id, bs.product_id, bs.product_name, bs.product_code, bs.billing_interval,
                   p.price, p.category, p.charge_rate
            FROM billing_schedules bs
            INNER JOIN tblproduct p ON bs.product_id = p.id
            WHERE bs.org_id = ? 
            AND bs.auto_billing_enabled = 1 
            AND bs.status = 'active'
            AND bs.next_billing_date <= CURDATE()
            AND p.status = 'active'
        ");
        $stmt->execute([$org_id]);
        $due_schedules = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $total_invoices = 0;
        $total_amount = 0;
        $successful_invoices = 0;
        $failed_invoices = 0;
        $billing_summary = [];
        
        foreach ($due_schedules as $schedule) {
            try {
                // Get all active clients for this product
                $stmt = $conn->prepare("
                    SELECT DISTINCT u.id as client_id, u.names as client_name, 
                           u.email, u.phone, u.address, u.nrc,
                           cbp.auto_billing_enabled as client_auto_billing
                    FROM users u
                    LEFT JOIN client_billing_preferences cbp ON u.id = cbp.client_id AND cbp.org_id = ?
                    WHERE u.org_id = ? 
                    AND u.status = 'active'
                    AND (cbp.auto_billing_enabled IS NULL OR cbp.auto_billing_enabled = 1)
                    AND (cbp.suspend_billing IS NULL OR cbp.suspend_billing = 0)
                    AND NOT EXISTS (
                        SELECT 1 FROM billing_subscription bs2 
                        WHERE bs2.client_id = u.id 
                        AND bs2.product_id = ?
                        AND bs2.org_id = ?
                        AND bs2.payment_status = 'Pending'
                        AND bs2.status = 'Active'
                    )
                ");
                $stmt->execute([$org_id, $org_id, $schedule['product_id'], $org_id]);
                $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                foreach ($clients as $client) {
                    // Skip if client has auto-billing disabled
                    if ($client['client_auto_billing'] === '0') {
                        continue;
                    }
                    
                    // Calculate due date based on grace period
                    $due_date = date('Y-m-d', strtotime("+$grace_period days"));
                    
                    // Generate invoice number
                    $stmt = $conn->prepare("
                        SELECT COALESCE(MAX(CAST(invoice_no AS UNSIGNED)), 0) + 1 as next_invoice 
                        FROM billing_subscription 
                        WHERE org_id = ?
                    ");
                    $stmt->execute([$org_id]);
                    $invoice_no = $stmt->fetch(PDO::FETCH_ASSOC)['next_invoice'];
                    
                    // Insert the invoice
                    $stmt = $conn->prepare("
                        INSERT INTO billing_subscription (
                            client_name, client_id, product_name, product_id, product_code,
                            units, unit_price, total_charge, balance, balance_cf, paid,
                            due_date, payment_status, invoice_no, document_type,
                            phone_contact, email_contact, billing_interval,
                            org_id, org_name, user_id, billing_schedule_id,
                            transaction_date, transaction_time, status
                        ) VALUES (?, ?, ?, ?, ?, 1, ?, ?, ?, ?, 0, ?, 'Pending', ?, 'INVOICE', 
                                 ?, ?, ?, ?, ?, ?, CURDATE(), CURTIME(), 'Active')
                    ");
                    
                    $stmt->execute([
                        $client['client_name'], $client['client_id'], $schedule['product_name'], 
                        $schedule['product_id'], $schedule['product_code'], $schedule['price'],
                        $schedule['price'], $schedule['price'], $schedule['price'], $due_date,
                        $invoice_no, $client['phone'], $client['email'], $schedule['billing_interval'],
                        $org_id, 'Lukulu Town Council', $user_id, $schedule['id']
                    ]);
                    
                    $invoice_id = $conn->lastInsertId();
                    $total_invoices++;
                    $total_amount += $schedule['price'];
                    $successful_invoices++;
                    
                    // Record in summary
                    $billing_summary[] = [
                        'client' => $client['client_name'],
                        'product' => $schedule['product_name'],
                        'amount' => $schedule['price'],
                        'invoice_no' => $invoice_no
                    ];
                    
                    // Send notification (placeholder - implement your notification system)
                    $this->sendBillingNotification($conn, $client, $schedule, $invoice_no, $due_date, $org_id);
                }
                
                // Update next billing date for this schedule
                $next_date = $this->calculateNextBillingDate($schedule['billing_interval']);
                $stmt = $conn->prepare("
                    UPDATE billing_schedules 
                    SET last_billing_date = CURDATE(), next_billing_date = ?
                    WHERE id = ?
                ");
                $stmt->execute([$next_date, $schedule['id']]);
                
            } catch (Exception $e) {
                $failed_invoices++;
                // Log exception
                $this->logBillingException($conn, $run_id, $client['client_id'] ?? null, 
                                         $client['client_name'] ?? 'Unknown', 
                                         $schedule['product_id'], $schedule['product_name'],
                                         'CLIENT_BILLING_ERROR', $e->getMessage(), $org_id);
            }
        }
        
        // Update billing run as completed
        $summary_details = json_encode([
            'total_processed' => $total_invoices,
            'successful' => $successful_invoices,
            'failed' => $failed_invoices,
            'total_amount' => $total_amount,
            'billing_summary' => $billing_summary
        ]);
        
        $stmt = $conn->prepare("
            UPDATE billing_runs 
            SET status = 'completed', completed_at = NOW(), 
                total_clients_billed = ?, total_invoices_generated = ?,
                total_amount_billed = ?, successful_invoices = ?,
                failed_invoices = ?, summary_details = ?
            WHERE id = ?
        ");
        $stmt->execute([count($clients ?? []), $total_invoices, $total_amount, 
                       $successful_invoices, $failed_invoices, $summary_details, $run_id]);
        
        // Commit transaction
        $conn->commit();
        
        echo json_encode([
            'status' => 'success',
            'message' => "Auto billing completed successfully",
            'run_id' => $run_id,
            'summary' => [
                'total_invoices' => $total_invoices,
                'total_amount' => number_format($total_amount, 2),
                'successful' => $successful_invoices,
                'failed' => $failed_invoices
            ]
        ]);
        
    } catch (Exception $e) {
        // Rollback transaction on error
        if ($conn->inTransaction()) {
            $conn->rollBack();
        }
        
        // Update billing run as failed
        if ($run_id) {
            $stmt = $conn->prepare("
                UPDATE billing_runs 
                SET status = 'failed', completed_at = NOW(), error_message = ?
                WHERE id = ?
            ");
            $stmt->execute([$e->getMessage(), $run_id]);
        }
        
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}
/**
 * Get billing runs history
 */
function getBillingRuns($conn, $org_id) {
    try {
        $page = $_GET['page'] ?? 1;
        $limit = $_GET['limit'] ?? 10;
        $offset = ($page - 1) * $limit;
        
        $stmt = $conn->prepare("
            SELECT id, run_type, run_date, run_time, total_invoices_generated, 
                   total_amount_billed, status, started_at, completed_at,
                   run_by_user_name, error_message
            FROM billing_runs 
            WHERE org_id = ?
            ORDER BY run_date DESC, run_time DESC 
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$org_id, (int)$limit, (int)$offset]);
        $runs = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get total count
        $stmt = $conn->prepare("SELECT COUNT(*) as total FROM billing_runs WHERE org_id = ?");
        $stmt->execute([$org_id]);
        $total = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        echo json_encode([
            'status' => 'success',
            'runs' => $runs,
            'pagination' => [
                'page' => (int)$page,
                'limit' => (int)$limit,
                'total' => (int)$total
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}
/**
 * Calculate next billing date based on interval
 */
function calculateNextBillingDate($interval) {
    switch ($interval) {
        case 'daily':
            return date('Y-m-d', strtotime('+1 day'));
        case 'weekly':
            return date('Y-m-d', strtotime('+1 week'));
        case 'monthly':
            return date('Y-m-d', strtotime('+1 month'));
        case 'quarterly':
            return date('Y-m-d', strtotime('+3 months'));
        case 'yearly':
            return date('Y-m-d', strtotime('+1 year'));
        default:
            return date('Y-m-d', strtotime('+1 month'));
    }
}

/**
 * Log billing exceptions
 */
function logBillingException($conn, $run_id, $client_id, $client_name, $product_id, $product_name, $error_type, $error_message, $org_id) {
    try {
        $stmt = $conn->prepare("
            INSERT INTO billing_exceptions (billing_run_id, client_id, client_name, product_id, product_name, error_type, error_message, org_id)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([$run_id, $client_id, $client_name, $product_id, $product_name, $error_type, $error_message, $org_id]);
    } catch (Exception $e) {
        // Silently fail exception logging to avoid breaking main process
        error_log("Failed to log billing exception: " . $e->getMessage());
    }
}

/**
 * Send billing notification (placeholder - implement your notification system)
 */
function sendBillingNotification($conn, $client, $product, $invoice_no, $due_date, $org_id) {
    try {
        // Insert into notifications table
        $stmt = $conn->prepare("
            INSERT INTO billing_notifications (client_id, notification_type, notification_method, subject, message_content, org_id)
            VALUES (?, 'invoice_created', 'email', ?, ?, ?)
        ");
        
        $subject = "New Invoice Generated - #$invoice_no";
        $message = "Dear {$client['client_name']},\n\nA new invoice for {$product['product_name']} has been generated.\nAmount: {$product['price']}\nDue Date: $due_date\n\nThank you.";
        
        $stmt->execute([$client['client_id'], $subject, $message, $org_id]);
        
    } catch (Exception $e) {
        error_log("Failed to send notification: " . $e->getMessage());
    }
}
 /**
 * Create manual invoice
 */
function createManualInvoice($conn, $org_id, $user_id) {
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        // Validate required fields
        $required = ['client_id', 'product_id', 'quantity', 'due_date'];
        foreach ($required as $field) {
            if (empty($input[$field])) {
                throw new Exception("Missing required field: $field");
            }
        }
        
        $client_id = $input['client_id'];
        $product_id = $input['product_id'];
        $quantity = $input['quantity'];
        $due_date = $input['due_date'];
        $notes = $input['notes'] ?? '';
        
        // Get product details
        $stmt = $conn->prepare("
            SELECT name, price, product_code, category, charge_rate 
            FROM tblproduct 
            WHERE id = ? AND status = 'active'
        ");
        $stmt->execute([$product_id]);
        $product = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$product) {
            throw new Exception("Product not found or inactive");
        }
        
        // Get client details
        $stmt = $conn->prepare("
            SELECT names, phone, email, address, nrc 
            FROM users 
            WHERE id = ? AND status = 'active'
        ");
        $stmt->execute([$client_id]);
        $client = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$client) {
            throw new Exception("Client not found or inactive");
        }
        
        // Calculate amounts
        $unit_price = $product['price'];
        $total_charge = $unit_price * $quantity;
        
        // Generate invoice number
        $stmt = $conn->prepare("
            SELECT COALESCE(MAX(CAST(invoice_no AS UNSIGNED)), 0) + 1 as next_invoice 
            FROM billing_subscription 
            WHERE org_id = ?
        ");
        $stmt->execute([$org_id]);
        $invoice_no = $stmt->fetch(PDO::FETCH_ASSOC)['next_invoice'];
        
        // Insert invoice
        $stmt = $conn->prepare("
            INSERT INTO billing_subscription (
                client_name, client_id, product_name, product_id, product_code,
                units, unit_price, total_charge, balance, balance_cf,
                due_date, payment_status, invoice_no, document_type,
                phone_contact, email_contact, org_id, org_name, user_id
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Pending', ?, 'INVOICE', ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $client['names'], $client_id, $product['name'], $product_id, $product['product_code'],
            $quantity, $unit_price, $total_charge, $total_charge, $total_charge,
            $due_date, $invoice_no, $client['phone'], $client['email'], $org_id, 'Lukulu Town Council', $user_id
        ]);
        
        $invoice_id = $conn->lastInsertId();
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Invoice created successfully',
            'invoice_id' => $invoice_id,
            'invoice_no' => $invoice_no
        ]);
        
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

function getClientBillingPreferences($conn, $org_id) {
    echo json_encode(['status' => 'info', 'message' => 'Client preferences coming soon']);
}

/**
 * Run auto billing
 */
 

?>