<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PATCH, PUT, DELETE, OPTIONS');

session_start();
require_once '../../../../xd-assets/backend/config/dbconfig.php';

// Print the raw POST data for debugging
error_log("Raw POST data: " . print_r($_POST, true));

$startDateInput = filter_input(INPUT_POST, "startDate", FILTER_DEFAULT);
$endDateInput = filter_input(INPUT_POST, "endDate", FILTER_DEFAULT);
$bankname = filter_input(INPUT_POST, "searchKey", FILTER_DEFAULT);

// Debugging: print received inputs
error_log("Received startDate: " . $startDateInput);
error_log("Received endDate: " . $endDateInput);

// Ensure both dates and bank name are provided and not empty
if (empty($startDateInput) || empty($endDateInput) || empty($bankname)) {
    echo json_encode(["error" => "startDate, endDate, and searchKey are required"]);
    exit;
}

// Validate date format (YYYY-MM-DD)
if (!DateTime::createFromFormat('Y-m-d', $startDateInput) || !DateTime::createFromFormat('Y-m-d', $endDateInput)) {
    echo json_encode(["error" => "Invalid date format. Use YYYY-MM-DD"]);
    exit;
}

try {
    // First query to calculate total daily sales
    $getServiceRequestQuery = "
        SELECT SUM(transaction_amount) AS totaldailysales 
        FROM transaction_details 
        WHERE bank_account_name != '' 
        AND transaction_status = 'approved' 
        AND deposit_status = 'Approved' 
        AND transaction_date BETWEEN :startDate AND :endDate 
        AND bank_account_name = :bankname";

    $getServiceRequest = $conn->prepare($getServiceRequestQuery);
    $getServiceRequest->bindParam(':startDate', $startDateInput);
    $getServiceRequest->bindParam(':endDate', $endDateInput);
    $getServiceRequest->bindParam(':bankname', $bankname);
    $getServiceRequest->execute();

    $getServiceRequest->setFetchMode(PDO::FETCH_ASSOC);
    $row = $getServiceRequest->fetch();

    if (!$row) {
        echo json_encode(['error' => 'No records found for the given date range and bank name.']);
        exit;
    }

    $totaldailysales = $row['totaldailysales'];

    // Second query to fetch transaction details grouped by product
    $getDriversQuery = "
        SELECT id, SUM(transaction_amount) AS bank_balance, product_name, product_id, transaction_amount, transaction_date, bank_account_name, bank_account_number, deposit_slip_date 
        FROM transaction_details 
        WHERE bank_account_name != '' 
        AND transaction_status = 'approved' 
        AND deposit_status = 'Approved' 
        AND transaction_date BETWEEN :startDate AND :endDate 
        AND bank_account_name = :bankname
        GROUP BY product_name, product_id
        ORDER BY product_name ASC";

    $getDrivers = $conn->prepare($getDriversQuery);
    $getDrivers->bindParam(':startDate', $startDateInput);
    $getDrivers->bindParam(':endDate', $endDateInput);
    $getDrivers->bindParam(':bankname', $bankname);
    $getDrivers->execute();

    $getDrivers->setFetchMode(PDO::FETCH_ASSOC);
    $details = []; 

    while ($rowDs = $getDrivers->fetch()) { 
        $details[] = array(
            'id' => $rowDs['id'],
            'product_name' => $rowDs['product_name'],
            'product_id' => $rowDs['product_id'],
            'bank_balance' => $rowDs['bank_balance'],
            'bank_account_name' => $rowDs['bank_account_name'],
            'transaction_date' => $rowDs['transaction_date'],
            'deposit_slip_date' => $rowDs['deposit_slip_date']
        );
    }

    // Third query to fetch deposit slip and deposit ID
    $getDepositQuery = "
        SELECT bank_deposit_slip, bank_deposit_id
        FROM transaction_details 
        WHERE bank_account_name != '' 
        AND transaction_status = 'approved' 
        AND deposit_status = 'Approved' 
        AND deposit_slip_date BETWEEN :startDate AND :endDate 
        AND bank_account_name = :bankname
        GROUP BY bank_deposit_id";

    $getDeposit = $conn->prepare($getDepositQuery);
    $getDeposit->bindParam(':startDate', $startDateInput);
    $getDeposit->bindParam(':endDate', $endDateInput);
    $getDeposit->bindParam(':bankname', $bankname);
    $getDeposit->execute();

    $getDeposit->setFetchMode(PDO::FETCH_ASSOC);
    $depositDetails = [];

    while ($rowDeposit = $getDeposit->fetch()) {
        $depositDetails[] = array(
            'bank_deposit_slip' => $rowDeposit['bank_deposit_slip'],
            'bank_deposit_id' => $rowDeposit['bank_deposit_id']
        );
    }

    // Prepare JSON response
    if (empty($details) && empty($depositDetails)) {
        $json_data = array("details" => 'No Data');
    } else {
        $json_data = array(
            "details" => $details,
            "deposits" => $depositDetails,
            'numofrows' => count($details),
            'total' => $totaldailysales
        );
    }

    echo json_encode($json_data);
    exit;

} catch (Exception $e) {
    // Error handling
    error_log("Error: " . $e->getMessage());
    echo json_encode(["error" => "An error occurred while processing your request."]);
    exit;
}
?>
