<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PATCH, PUT, DELETE, OPTIONS');
session_start();

require_once '../../../xd-assets/backend/config/dbconfig.php';

try {
    $conn = new PDO(
        "mysql:host=$servername;dbname=$database",
        $username, $password,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
} catch (PDOException $e) {
    echo json_encode(['status'=>'error','message'=>'Connection failed: '.$e->getMessage()]);
    exit;
}

// 1) Identify user/org
$userid = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
$orgid  = $_SESSION['orgid'] ?? null;
if (!$userid || !$orgid) {
    echo json_encode(['status'=>'error','message'=>'Missing user or organization']);
    exit;
}

// 2) Optional date filters
$startDate = filter_input(INPUT_POST, 'startDate', FILTER_UNSAFE_RAW);
$endDate   = filter_input(INPUT_POST, 'endDate',   FILTER_UNSAFE_RAW);

$invDateCond = '';
$payDateCond = '';
$params      = [
    ':userid' => $userid,
    ':orgid'  => $orgid
];

if ($startDate) {
    $invDateCond .= " AND STR_TO_DATE(b.due_date,'%Y-%m-%d') >= :start_date";
    $payDateCond .= " AND STR_TO_DATE(td.transaction_date,'%Y-%m-%d') >= :start_date";
    $params[':start_date'] = date('Y-m-d', strtotime($startDate));
}
if ($endDate) {
    $invDateCond .= " AND STR_TO_DATE(b.due_date,'%Y-%m-%d') <= :end_date";
    $payDateCond .= " AND STR_TO_DATE(td.transaction_date,'%Y-%m-%d') <= :end_date";
    $params[':end_date'] = date('Y-m-d', strtotime($endDate));
}

// 3) Initialize running balance
$conn->exec("SET @running_balance := 0");

// 4) Build & execute the UNION ALL with aggregation
 // … earlier code …

$sql = "
SELECT
  t.date,
  t.type,
  t.ref,
  FORMAT(t.debit,  2) AS debit,
  FORMAT(t.credit, 2) AS credit,
  FORMAT((@running_balance := @running_balance + t.debit - t.credit), 2) AS running_balance
FROM (
  /* —— Aggregate Invoices by invoice_no —— */
  SELECT
    DATE(MIN(STR_TO_DATE(b.due_date, '%Y-%m-%d')))   AS date,
    'Invoice'                                        AS type,
    b.invoice_no                                     AS ref,
    SUM(CAST(b.total_charge AS DECIMAL(12,2)))       AS debit,
    0.00                                             AS credit
  FROM billing_subscription AS b                      -- alias here
  WHERE b.client_id     = :userid
    AND b.org_id        = :orgid
    AND b.document_type = 'INVOICE'
    {$invDateCond}
  GROUP BY b.invoice_no

  UNION ALL

  /* —— Aggregate Payments by invoiceNumber, only for included invoices —— */
  SELECT
    DATE(MIN(STR_TO_DATE(td.transaction_date, '%Y-%m-%d'))) AS date,
    'Payment'                                               AS type,
    td.invoiceNumber                                        AS ref,
    0.00                                                    AS debit,
    SUM(CAST(td.transaction_amount AS DECIMAL(12,2)))       AS credit
  FROM transaction_details td
    INNER JOIN (
      /* build exactly the same invoice list, with alias b */
      SELECT b.invoice_no
      FROM billing_subscription AS b
      WHERE b.client_id     = :userid
        AND b.org_id        = :orgid
        AND b.document_type = 'INVOICE'
        {$invDateCond}      -- now b.due_date resolves
      GROUP BY b.invoice_no
    ) AS invs
      ON invs.invoice_no = td.invoiceNumber
  WHERE td.user_id         = :userid
    AND td.org_id          = :orgid
    AND td.debit_or_credit = 'Credit'
    AND td.trans_type      = 'RECEIPT'
    {$payDateCond}
  GROUP BY td.invoiceNumber
) AS t
ORDER BY t.date ASC
LIMIT 1000
";


try {
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Convert running_balance to float and add interpretation
    foreach ($transactions as &$row) {
        $balance = floatval(str_replace(',', '', $row['running_balance']));
        if ($balance > 0) {
            $row['balance_status'] = 'Amount Owed';
        } elseif ($balance < 0) {
            $row['balance_status'] = 'Credit Balance';
        } else {
            $row['balance_status'] = 'Fully Paid';
        }
        $row['raw_balance'] = $balance;
    }

    if (empty($transactions)) {
        echo json_encode(['status'=>'empty','message'=>'No transactions found']);
        exit;
    }

    echo json_encode([
        'status'       => 'success',
        'transactions' => $transactions
    ], JSON_PRETTY_PRINT);

} catch (PDOException $e) {
    echo json_encode(['status'=>'error','message'=>'Query failed: '.$e->getMessage()]);
    exit;
}
