// ========================
// products.js (Offline Product Manager with Image Blobs)
// ========================

// Open IndexedDB database for products
function openDB() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('productsDB', 4);

        request.onupgradeneeded = function (e) {
            const db = e.target.result;
            if (!db.objectStoreNames.contains('products')) {
                db.createObjectStore('products', { keyPath: 'id' });
            }
        };

        request.onsuccess = function (e) {
            resolve(e.target.result);
        };

        request.onerror = function (e) {
            reject(e.target.error);
        };
    });
}

// =============================
// Save products to IndexedDB (with image Blob support) - CRITICAL FIX
// =============================
async function saveProductsToIndexedDB(products) {
    console.log(`💾 💾 💾 Starting to save ${products.length} products to IndexedDB...`);
    
    // ✅ CRITICAL FIX: Process ALL images FIRST (outside of transaction)
    // Then open transaction and save all at once
    const processedProducts = [];
    
    for (let i = 0; i < products.length; i++) {
        const product = products[i];
        console.log(`📦 Processing product ${i + 1}/${products.length}: ${product.name}`);
        
        try {
            if (product.image && product.image.trim() !== "") {
                try {
                    const response = await fetch(product.image);
                    if (response.ok) {
                        const blob = await response.blob();
                        product.imageBlob = blob;
                        console.log(`✅ Image loaded for: ${product.name}`);
                    } else {
                        console.warn(`⚠️ Image not found for ${product.name}, using default.`);
                        const defaultResponse = await fetch('xd-assets/img/10101.jpg');
                        const defaultBlob = await defaultResponse.blob();
                        product.imageBlob = defaultBlob;
                    }
                } catch (imgError) {
                    console.error(`❌ Error fetching image for ${product.name}:`, imgError);
                    try {
                        const defaultResponse = await fetch('xd-assets/img/10101.jpg');
                        const defaultBlob = await defaultResponse.blob();
                        product.imageBlob = defaultBlob;
                    } catch (defaultError) {
                        console.error(`❌ Error fetching default image:`, defaultError);
                    }
                }
            } else {
                try {
                    const defaultResponse = await fetch('xd-assets/img/10101.jpg');
                    const defaultBlob = await defaultResponse.blob();
                    product.imageBlob = defaultBlob;
                } catch (defaultError) {
                    console.error(`❌ Error fetching default image:`, defaultError);
                }
            }
            
            processedProducts.push(product);
        } catch (error) {
            console.error(`❌ Critical error processing ${product.name}:`, error);
            processedProducts.push(product);
        }
    }

    console.log(`✅ All images processed. Now saving ${processedProducts.length} products to IndexedDB...`);

    // ✅ CRITICAL: Now open transaction and save all products synchronously
    return new Promise(async (resolve, reject) => {
        try {
            const db = await openDB();
            console.log('✅ Database opened successfully');
            
            const transaction = db.transaction(['products'], 'readwrite');
            const store = transaction.objectStore('products');
            
            console.log('✅ Transaction started, saving products...');
            
            // Save all products synchronously in the transaction
            let savedCount = 0;
            for (const product of processedProducts) {
                try {
                    store.put(product);
                    savedCount++;
                    console.log(`✅ [${savedCount}/${processedProducts.length}] Saved: ${product.name}`);
                } catch (putError) {
                    console.error(`❌ Error saving ${product.name}:`, putError);
                }
            }
            
            transaction.oncomplete = () => {
                console.log('✅ ✅ ✅ TRANSACTION COMPLETE - ALL PRODUCTS SAVED ✅ ✅ ✅');
                console.log(`📊 Total products saved: ${savedCount}/${processedProducts.length}`);
                
                // Verify immediately
                getProductsFromIndexedDB().then(savedProducts => {
                    console.log(`🔍 VERIFICATION: ${savedProducts.length} products now in IndexedDB`);
                    if (savedProducts.length > 0) {
                        console.log('✅ ✅ ✅ SAVE SUCCESSFUL - PRODUCTS CONFIRMED IN DATABASE ✅ ✅ ✅');
                    } else {
                        console.error('❌ ❌ ❌ SAVE FAILED - NO PRODUCTS IN DATABASE ❌ ❌ ❌');
                    }
                    resolve(savedProducts);
                }).catch(verifyError => {
                    console.error('❌ Verification failed:', verifyError);
                    resolve([]);
                });
            };
            
            transaction.onerror = (e) => {
                console.error('❌ ❌ ❌ TRANSACTION ERROR:', e.target.error);
                reject(e.target.error);
            };
            
            transaction.onabort = (e) => {
                console.error('❌ ❌ ❌ TRANSACTION ABORTED:', e.target.error);
                reject(e.target.error);
            };
            
        } catch (error) {
            console.error('❌ ❌ ❌ ERROR OPENING DATABASE:', error);
            reject(error);
        }
    });
}

// =============================
// Retrieve all products
// =============================
function getProductsFromIndexedDB() {
    return new Promise((resolve, reject) => {
        openDB().then((db) => {
            const transaction = db.transaction(['products'], 'readonly');
            const store = transaction.objectStore('products');
            const request = store.getAll();
            request.onsuccess = () => {
                console.log(`📦 Retrieved ${request.result.length} products from IndexedDB`);
                resolve(request.result);
            };
            request.onerror = (e) => {
                console.error('❌ Error retrieving products from IndexedDB:', e.target.error);
                reject(e.target.error);
            };
        }).catch(reject);
    });
}

// =============================
// Clear all products
// =============================
function clearProductsFromIndexedDB() {
    return new Promise((resolve, reject) => {
        openDB().then((db) => {
            const transaction = db.transaction(['products'], 'readwrite');
            const store = transaction.objectStore('products');
            const clearRequest = store.clear();
            clearRequest.onsuccess = () => {
                console.log("🧹 Cleared products from IndexedDB");
                resolve();
            };
            clearRequest.onerror = (e) => reject(e.target.error);
        }).catch(reject);
    });
}

// =============================
// Fetch products from server and save to IndexedDB
// =============================
async function fetchProductsFromServerAndPopulate() {
    console.log('🌐 🌐 🌐 fetchProductsFromServerAndPopulate CALLED 🌐 🌐 🌐');
    
    if (!navigator.onLine) {
        console.warn('⚠️ OFFLINE: Cannot fetch products from server');
        $('#invoice_modal_div_2').html('<p style="color:orange;">You are offline. Please connect to the internet to load products for the first time.</p>');
        return;
    }

    $('#invoice_modal_div_2').empty().append('<center><div class="body-loader"></div><br/>Loading products from server...</center>');
    console.log("🌐 Fetching products from server...");
    
    try {
        const response = await fetch('xd-modules/revenue-accountant/backend/getProductsToAddToReceipt.php');
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const data = await response.json();
        console.log("📥 Server response received");
        console.log("📊 Server data:", data);
        
        if (data.details && Array.isArray(data.details) && data.details.length > 0) {
            console.log(`📦 Server returned ${data.details.length} products`);
            console.log("💾 Now calling saveProductsToIndexedDB...");
            
            // ✅ CRITICAL: Wait for save to complete
            const savedProducts = await saveProductsToIndexedDB(data.details);
            
            console.log(`✅ Save operation completed. Saved products count: ${savedProducts.length}`);
            
            // ✅ Double verify by reading from IndexedDB again
            const verifyProducts = await getProductsFromIndexedDB();
            console.log(`🔍 Double verification: ${verifyProducts.length} products in IndexedDB`);
            
            if (verifyProducts.length > 0) {
                console.log('✅ SUCCESS: Products confirmed in IndexedDB, now displaying...');
                appendProductsToDOM(verifyProducts);
            } else {
                console.error('❌ CRITICAL: Products were not saved to IndexedDB!');
                $('#invoice_modal_div_2').html('<p style="color:red;">Error: Products could not be saved. Please refresh and try again.</p>');
            }
        } else {
            console.warn('⚠️ No products in server response');
            $('#invoice_modal_div_2').html('<p>No products available from server.</p>');
        }
    } catch (error) {
        console.error("❌ Error fetching products from server:", error);
        $('#invoice_modal_div_2').html('<p style="color:red;">Unable to fetch products. Please check your internet connection.</p>');
    }
}

// =============================
// Populate products if empty or belongs to another user
// =============================
async function populateProductsIfEmpty() {
    console.log('🔍 🔍 🔍 populateProductsIfEmpty CALLED 🔍 🔍 🔍');
    
    try {
        const currentUserId = await getUserIDFromIndexedDB();
        console.log(`👤 Current user ID: ${currentUserId}`);
        
        const localProducts = await getProductsFromIndexedDB();
        console.log(`📦 Found ${localProducts.length} products in IndexedDB`);
        
        if (localProducts && localProducts.length > 0) {
            const belongs = localProducts.every(p => p.user_id == currentUserId);
            
            if (!belongs) {
                console.log("⚠️ Products belong to another user. Clearing and fetching new ones...");
                await clearProductsFromIndexedDB();
                await fetchProductsFromServerAndPopulate();
            } else {
                console.log("✅ ✅ ✅ Local DB has products for this user. NO FETCH NEEDED. ✅ ✅ ✅");
            }
        } else {
            console.log("📭 IndexedDB is empty. Fetching from server...");
            await fetchProductsFromServerAndPopulate();
        }
    } catch (error) {
        console.error("❌ Error in populateProductsIfEmpty:", error);
        
        if (!navigator.onLine) {
            console.warn('⚠️ Offline mode: Cannot populate products');
        }
    }
}

// Update products on the server using local IndexedDB products
async function updateProductsDatabase(products) {
    if (!navigator.onLine) {
        console.log("⚠️ Offline: Skipping product update");
        return;
    }

    try {
        const response = await fetch('xd-modules/revenue-accountant/backend/getProductsToAddToReceipt.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ products }),
        });
        if (!response.ok) {
            throw new Error('Failed to update products');
        }
        console.log('✅ Products updated successfully on server');
    } catch (error) {
        console.error('❌ Error updating products:', error);
    }
}

// Start periodic updates (30-second interval)
let updateIntervalId;
let periodicUpdatesStarted = false;
function startPeriodicUpdates(interval = 30 * 1000) {
    if (periodicUpdatesStarted) return;
    periodicUpdatesStarted = true;

    updateIntervalId = setInterval(async () => {
        if (!navigator.onLine) {
            console.log("⚠️ User is offline. Skipping update.");
            return;
        }
        console.log("🔄 Periodic update triggered at", new Date().toLocaleTimeString());
        const products = await getProductsFromIndexedDB();
        if (products && products.length > 0) {
            await updateProductsDatabase(products);
            console.log("✅ Products updated on server at", new Date().toLocaleTimeString());
        } else {
            console.log("📭 No products found in IndexedDB for update.");
        }
    }, interval);
}

// =============================
// User DB (for current user ID)
// =============================
function openUserDB() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('userDB', 3);
        request.onupgradeneeded = function (event) {
            const db = event.target.result;
            if (!db.objectStoreNames.contains('user')) {
                db.createObjectStore('user', { keyPath: 'id' });
            }
        };
        request.onsuccess = (e) => resolve(e.target.result);
        request.onerror = (e) => reject(e.target.error);
    });
}

function getUserIDFromIndexedDB() {
    return new Promise((resolve, reject) => {
        openUserDB().then(db => {
            const tx = db.transaction(['user'], 'readonly');
            const store = tx.objectStore('user');
            const req = store.getAll();
            req.onsuccess = function () {
                if (req.result && req.result.length > 0) {
                    resolve(req.result[0].id);
                } else {
                    console.error('❌ No user data found in IndexedDB');
                    reject("No user data found");
                }
            };
            req.onerror = (e) => reject(e.target.error);
        }).catch(reject);
    });
}

// =============================
// Fetch and append products - ALWAYS USE INDEXEDDB FIRST
// =============================
function fetchAndAppendProducts() {
    console.log('🔄 🔄 🔄 fetchAndAppendProducts CALLED 🔄 🔄 🔄');
    $('#invoice_modal_div_2').empty().append('<center><div class="body-loader"></div><br/>Loading products...</center>');
    
    getUserIDFromIndexedDB().then(userid => {
        console.log(`👤 Fetching products for user ID: ${userid}`);
        
        getProductsFromIndexedDB().then(products => {
            console.log(`📦 Got ${products.length} products from IndexedDB`);
            
            const filtered = products.filter(p => p.user_id === userid);
            console.log(`✅ ${filtered.length} products match user ID ${userid}`);
            
            if (filtered && filtered.length > 0) {
                // ✅ ALWAYS use IndexedDB products if available (online or offline)
                console.log('✅ ✅ ✅ Using products from IndexedDB cache ✅ ✅ ✅');
                appendProductsToDOM(filtered);
            } else {
                // Only fetch from server if IndexedDB is truly empty
                console.log('📭 No products for this user in IndexedDB');
                if (navigator.onLine) {
                    console.log('🌐 Online: Fetching from server...');
                    fetchProductsFromServerAndPopulate();
                } else {
                    console.log('⚠️ Offline: Cannot fetch products');
                    $('#invoice_modal_div_2').html('<p style="color:orange;">No products available offline. Please connect to the internet to load products.</p>');
                }
            }
        }).catch(error => {
            console.error("❌ Error getting products from IndexedDB:", error);
            
            if (navigator.onLine) {
                console.log('🌐 Online: Attempting to fetch from server as fallback...');
                fetchProductsFromServerAndPopulate();
            } else {
                console.log('⚠️ Offline: Cannot fetch products');
                $('#invoice_modal_div_2').html('<p style="color:red;">Error loading products offline.</p>');
            }
        });
    }).catch(error => {
        console.error("❌ Error getting user ID:", error);
        $('#invoice_modal_div_2').html('<p style="color:red;">Error: User not found. Please log in again.</p>');
    });
}

// =============================
// Event Handlers
// =============================
$(document).on('click', '#addProductToReceipt101', function () {
    console.log('🛒 Add product to receipt button clicked');
    addProductToReceipt101();
});

// Cart counter update + animation
function updateCartCount(change) {
    const $cartCount = $('#cart-count0');
    let currentCount = parseInt($cartCount.text()) || 0;
    let newCount = Math.max(0, currentCount + change);
    $cartCount.text(newCount);

    $('#cart-btn .fx-shopping-cart').css({
        transform: 'scale(1.3)',
        transition: 'transform 0.2s ease'
    });

    setTimeout(() => {
        $('#cart-btn .fx-shopping-cart').css('transform', 'scale(1)');
    }, 200);
}

// Enhanced Add Product function
function addProductToReceipt101() {
    console.log('📝 Setting up product selection UI...');
    $('#invoice_modal_div_0').fadeOut('slow');
    $('.invoiceDivActions101').fadeOut('slow');
    $('#invoice_modal_div_1').empty().fadeIn('slow');

    $('#invoice_modal_div_1').append(`
        <div class="w3-col l12 m12 s12 w3-card-2 w3-top" style="background-color:#fff;height:40px;line-height:40px;">
            <div class="close_receipt_modal_div_1 w3-col l1 m1 s1" style="padding-top:0px;cursor:pointer;">
                <span class="fontx fx-arrow-left" style="color:var(--primary-color);font-size:2.5em;float:left"></span>
            </div>
            <div class="w3-col l9 m9 s9" style="color:var(--primary-color);font-weight:600;">
                <div class="w3-col l12 m12 s12 w3-center" style="margin-top:0px;">
                    <div class="w3-col l1 m1 s1 w3-center" style="padding-top:5px;">
                        <span class="fontx fx-search" style="color:rgba(0,0,0,0.3);font-size:1.5em"></span>
                    </div>
                    <div class="w3-col l10 m10 s10 w3-center">
                        <input class="searchProductsToAddToReceipt77" type="text" placeholder="Search & add items to cart..." 
                            style="font-weight:400;outline:none;color:gray;border:none;width:100%;height:30px;background-color:transparent;padding-left:7px;"/>
                    </div>
                </div>
            </div>
            <div class="w3-col l2 m2 s2" style="padding-top:0px;position:relative;">
                <a href="#" id="cart-btn" style="text-decoration:none;outline:none;background-color:transparent;
                    color:var(--primary-color);font-size:1.6em;display:inline-block;position:relative;">
                    <span class="fontx fx-shopping-cart" 
                        style="position:relative;display:inline-block;">
                        <span id="cart-count0" 
                            style="position:absolute;top:-6px;right:-8px;
                            color:white;height:16px;width:16px;
                            background-color:orange;border-radius:50%;
                            font-size:0.55em;line-height:16px;text-align:center;">
                            0
                        </span>
                    </span>
                </a>
            </div>
        </div>
        <div id="invoice_modal_div_2" class="w3-col l12 m12 s12" style="margin-top:50px;padding-bottom:100px;"></div>

        <div class="close_receipt_modal_div_1 proceedToReceiptOnMobile01 w3-bottom w3-hide-large" style="display:none;position:fixed;height:40px;width:100%;color:#fff;background-color:var(--primary-color);line-height:40px;cursor: pointer;font-weight:600;">
        PROCEED 
        <span class="fontx fx-arrow-right"></span>
        </div>
    `);

    fetchAndAppendProducts();
}

// Event listener for searching products in IndexedDB
$(document).on('change keyup', '.searchProductsToAddToReceipt77', function (event) {
    event.preventDefault();
    const searchKey = $(this).val().trim();
    console.log(`🔍 Searching for: "${searchKey}"`);
    
    $('#invoice_modal_div_2').empty().append('<center><div class="body-loader"></div><br/>Searching...</center>');
    
    getUserIDFromIndexedDB().then(userid => {
        if (searchKey === '') {
            // Empty search - show all products
            getProductsFromIndexedDB().then(allProducts => {
                const filtered = allProducts.filter(p => p.user_id === userid);
                console.log(`📦 Showing all ${filtered.length} products`);
                if (filtered.length > 0) {
                    appendProductsToDOM(filtered);
                } else {
                    $('#invoice_modal_div_2').html('<p>No products found.</p>');
                }
            });
        } else {
            // Search with keyword
            searchProductsInIndexedDB(userid, searchKey).then(products => {
                console.log(`🔍 Found ${products.length} products matching "${searchKey}"`);
                if (products.length > 0) {
                    appendProductsToDOM(products);
                } else {
                    $('#invoice_modal_div_2').html(`<p>No products match "${searchKey}"</p>`);
                }
            }).catch(error => {
                console.error("❌ Search error:", error);
                $('#invoice_modal_div_2').html('<p style="color:red;">Error searching products.</p>');
            });
        }
    }).catch(error => {
        console.error("❌ Error getting user ID:", error);
        $('#invoice_modal_div_2').html('<p style="color:red;">Error: User not found.</p>');
    });
});

// =============================
// Append products (with image blobs)
// =============================
function appendProductsToDOM(products) {
    console.log(`📊 📊 📊 Appending ${products.length} products to DOM 📊 📊 📊`);
    $('#invoice_modal_div_2').empty();

    if (!products || products.length === 0) {
        $('#invoice_modal_div_2').html('<p>No products to display.</p>');
        return;
    }

    products.forEach(product => {
        let imgURL = product.image;
        if (product.imageBlob) {
            imgURL = URL.createObjectURL(product.imageBlob);
        }

        $('#invoice_modal_div_2').append(`
            <div class="inner-data-div w3-col l4 m6 s12"
                id="addProductToReceipt102"
                data-id="${product.id}"
                data-name="${product.name}"
                data-price="${product.price}"
                data-chargerate="${product.charge_rate}"
                data-description="${product.description}"
                data-productcode="${product.product_code}"
                data-category="${product.category}"
                data-categorycode="${product.category_code}"
                data-depot="${product.depot}"
                data-depotid="${product.depot_id}"
                data-depotcode="${product.depot_code}"
                style="cursor:pointer;padding-left:7px;margin-top:3px;">
                <div class="w3-col l12 m12 s12">
                    <img src="${imgURL}" alt="${product.name}"
                         style="width:60px;height:60px;border-radius:5px;object-fit:cover;margin-right:10px;float:left;" />
                    <div class="w3-rest">
                        <b style="color:black;font-size:1em;">${product.name}</b>
                        <input type="checkbox" id="addProductToInvoiceCheckbox102" 
                        data-id="${product.id}" 
                        data-name="${product.name}" 
                        data-price="${product.price}" 
                        data-chargerate="${product.charge_rate}" 
                        data-description="${product.description}" 
                        data-productcode="${product.product_code}" 
                        data-category="${product.category}" 
                        data-categorycode="${product.category_code}" 
                        data-depot="${product.depot}" 
                        data-depotid="${product.depot_id}" 
                        data-depotcode="${product.depot_code}" 
                        style="display:none;background-color:var(--primary-color);color:#fff;text-decoration:none;float:right;margin-right:10px;padding-right:7px;padding-left:7px;border-radius:4px;height:20px;cursor:pointer;margin-top:15px;border:none;" />
                        <br/>
                        <span style="font-weight:600;color:var(--primary-color);font-size:1em;">K${product.price}</span>
                    </div>
                </div>
            </div>
        `);
    });
    
    console.log('✅ ✅ ✅ Products appended to DOM successfully ✅ ✅ ✅');
}

// Clean up Blob URLs to avoid memory leaks
window.addEventListener('beforeunload', () => {
    $('img').each(function() {
        const src = $(this).attr('src');
        if (src && src.startsWith('blob:')) URL.revokeObjectURL(src);
    });
});

// =============================
// Search & Fetch Integration
// =============================
function searchProductsInIndexedDB(userid, searchKey) {
    return new Promise((resolve, reject) => {
        openDB().then(db => {
            const tx = db.transaction(['products'], 'readonly');
            const store = tx.objectStore('products');
            const products = [];
            const req = store.openCursor();
            req.onsuccess = e => {
                const cursor = e.target.result;
                if (cursor) {
                    if (cursor.value.user_id === userid &&
                        (cursor.value.name.toLowerCase().includes(searchKey.toLowerCase()) ||
                         cursor.value.product_code.toLowerCase().includes(searchKey.toLowerCase()))) {
                        products.push(cursor.value);
                    }
                    cursor.continue();
                } else resolve(products);
            };
            req.onerror = e => reject(e.target.error);
        }).catch(reject);
    });
}

// Add animation when product is added or removed
$(document).on('click', '#addProductToReceipt102', function () {
    const checkbox = $(this).find('input[type="checkbox"]');
    const isChecked = !checkbox.prop('checked');
    checkbox.prop('checked', isChecked);

    $('.proceedToReceiptOnMobile01').fadeIn('slow');
    
    $(this).css({
        transition: 'all 0.3s ease',
        transform: 'scale(1.03)',
        backgroundColor: isChecked ? 'var(--primary-color-transparent1)' : '#fff'
    });
    setTimeout(() => $(this).css('transform', 'scale(1)'), 200);

    if (isChecked) {
        checkbox.css({
            display: 'inline-block',
            opacity: '0',
            transform: 'scale(0.5)',
            transition: 'all 0.25s ease-in-out'
        });
        setTimeout(() => {
            checkbox.css({
                opacity: '1',
                transform: 'scale(1.1)'
            });
        }, 50);
        setTimeout(() => {
            checkbox.css('transform', 'scale(1)');
        }, 200);
    } else {
        checkbox.css({
            opacity: '1',
            transform: 'scale(1)',
            transition: 'all 0.25s ease-in-out'
        });
        setTimeout(() => {
            checkbox.css({
                opacity: '0',
                transform: 'scale(0.5)'
            });
        }, 50);
        setTimeout(() => {
            checkbox.css('display', 'none');
        }, 300);
    }

    const name = $(this).data('name');
    const price = parseFloat($(this).data('price')) || 0;
    const charge_rate = $(this).data('chargerate');
    const id = $(this).data('id');
    const category = $(this).data('category');
    const categorycode = $(this).data('categorycode');
    const productcode = $(this).data('productcode');
    const depot = $(this).data('depot');
    const depotid = $(this).data('depotid');
    const depotcode = $(this).data('depotcode');
    const balancebroughtforward = parseFloat($(this).data('balancebroughtforward')) || 0;
    const balanceCF = '00';

    if (isChecked) {
        shoppingCart.addItemToCart(name, price, charge_rate, 1, id, productcode, category, categorycode, depot, depotid, depotcode, price, balancebroughtforward, price, balanceCF);
        updateCartCount(+1);
    } else {
        shoppingCart.removeItemFromCartAll(name);
        updateCartCount(-1);
    }

    displayCart();
});

// =============================
// INITIALIZATION - Wait for jQuery and DOM
// =============================
$(document).ready(async function() {
    console.log('🚀 🚀 🚀 PRODUCTS.JS INITIALIZING 🚀 🚀 🚀');
    console.log('⏰ Time:', new Date().toLocaleTimeString());
    
    try {
        await populateProductsIfEmpty();
        console.log("✅ ✅ ✅ Product population complete. Starting periodic updates... ✅ ✅ ✅");
        startPeriodicUpdates();
    } catch (error) {
        console.error("❌ ❌ ❌ Error initializing products:", error);
    }
});