-- ============================================================
-- COMPREHENSIVE INVENTORY MANAGEMENT SYSTEM FOR CYBERDESK SAAS
-- ============================================================

-- 1. WAREHOUSE/LOCATION MANAGEMENT
-- ============================================================

CREATE TABLE `warehouse` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `warehouse_code` VARCHAR(50) UNIQUE NOT NULL,
  `warehouse_name` VARCHAR(255) NOT NULL,
  `warehouse_type` ENUM('main', 'branch', 'transit', 'virtual') DEFAULT 'branch',
  `address` TEXT,
  `city` VARCHAR(100),
  `state` VARCHAR(100),
  `country` VARCHAR(100),
  `postal_code` VARCHAR(20),
  `lat` FLOAT(10,6),
  `lng` FLOAT(10,6),
  `phone` VARCHAR(50),
  `email` VARCHAR(100),
  `manager_id` INT,
  `capacity` DECIMAL(15,2) COMMENT 'Storage capacity in cubic meters',
  `current_utilization` DECIMAL(5,2) DEFAULT 0.00 COMMENT 'Percentage',
  `is_active` BOOLEAN DEFAULT TRUE,
  `org_id` VARCHAR(255),
  `created_by` INT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  INDEX `idx_org_warehouse` (`org_id`, `warehouse_code`),
  INDEX `idx_warehouse_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `warehouse_zone` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `warehouse_id` INT NOT NULL,
  `zone_code` VARCHAR(50) NOT NULL,
  `zone_name` VARCHAR(255) NOT NULL,
  `zone_type` ENUM('storage', 'picking', 'packing', 'receiving', 'shipping', 'quarantine') DEFAULT 'storage',
  `aisle` VARCHAR(50),
  `rack` VARCHAR(50),
  `shelf` VARCHAR(50),
  `bin` VARCHAR(50),
  `temperature_controlled` BOOLEAN DEFAULT FALSE,
  `temperature_range` VARCHAR(50),
  `capacity` DECIMAL(15,2),
  `is_active` BOOLEAN DEFAULT TRUE,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_zone` (`warehouse_id`, `zone_code`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`) ON DELETE CASCADE,
  INDEX `idx_zone_type` (`zone_type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 2. STOCK/INVENTORY TRACKING
-- ============================================================

CREATE TABLE `stock_movement` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `movement_code` VARCHAR(50) UNIQUE NOT NULL,
  `product_id` INT NOT NULL,
  `movement_type` ENUM('inbound', 'outbound', 'transfer', 'adjustment', 'return', 'damage', 'expired') NOT NULL,
  `reference_type` ENUM('purchase_order', 'sales_order', 'transfer', 'adjustment', 'production', 'return') NOT NULL,
  `reference_id` INT COMMENT 'ID of related document (PO, SO, etc)',
  `from_warehouse_id` INT,
  `from_zone_id` INT,
  `to_warehouse_id` INT,
  `to_zone_id` INT,
  `quantity` DECIMAL(15,3) NOT NULL,
  `unit_cost` DECIMAL(15,2),
  `total_cost` DECIMAL(15,2),
  `batch_number` VARCHAR(100),
  `serial_number` VARCHAR(100),
  `expiry_date` DATE,
  `movement_date` DATETIME NOT NULL,
  `reason` TEXT,
  `notes` TEXT,
  `approved_by` INT,
  `processed_by` INT,
  `status` ENUM('pending', 'approved', 'completed', 'cancelled') DEFAULT 'pending',
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  FOREIGN KEY (`from_warehouse_id`) REFERENCES `warehouse`(`id`),
  FOREIGN KEY (`to_warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_movement_product` (`product_id`, `movement_date`),
  INDEX `idx_movement_type` (`movement_type`, `status`),
  INDEX `idx_reference` (`reference_type`, `reference_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `stock_balance` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `product_id` INT NOT NULL,
  `warehouse_id` INT NOT NULL,
  `zone_id` INT,
  `quantity_on_hand` DECIMAL(15,3) DEFAULT 0,
  `quantity_allocated` DECIMAL(15,3) DEFAULT 0 COMMENT 'Reserved for orders',
  `quantity_available` DECIMAL(15,3) DEFAULT 0 COMMENT 'on_hand - allocated',
  `quantity_in_transit` DECIMAL(15,3) DEFAULT 0,
  `reorder_point` DECIMAL(15,3),
  `max_stock_level` DECIMAL(15,3),
  `average_cost` DECIMAL(15,2),
  `last_movement_date` DATETIME,
  `last_count_date` DATE,
  `org_id` VARCHAR(255),
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_stock` (`product_id`, `warehouse_id`, `zone_id`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_stock_available` (`quantity_available`),
  INDEX `idx_reorder_check` (`quantity_available`, `reorder_point`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `stock_batch` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `product_id` INT NOT NULL,
  `batch_number` VARCHAR(100) NOT NULL,
  `warehouse_id` INT NOT NULL,
  `quantity` DECIMAL(15,3) NOT NULL,
  `unit_cost` DECIMAL(15,2),
  `manufacture_date` DATE,
  `expiry_date` DATE,
  `supplier_id` INT,
  `received_date` DATE,
  `quality_status` ENUM('passed', 'failed', 'pending', 'quarantine') DEFAULT 'pending',
  `is_active` BOOLEAN DEFAULT TRUE,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_batch` (`product_id`, `batch_number`, `warehouse_id`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_expiry` (`expiry_date`, `is_active`),
  INDEX `idx_batch_product` (`product_id`, `is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `serial_number` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `product_id` INT NOT NULL,
  `serial_number` VARCHAR(100) UNIQUE NOT NULL,
  `batch_id` INT,
  `warehouse_id` INT,
  `zone_id` INT,
  `status` ENUM('in_stock', 'sold', 'reserved', 'damaged', 'returned') DEFAULT 'in_stock',
  `purchase_order_id` INT,
  `sales_order_id` INT,
  `warranty_start_date` DATE,
  `warranty_end_date` DATE,
  `customer_id` INT,
  `notes` TEXT,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_serial_status` (`status`),
  INDEX `idx_serial_product` (`product_id`, `status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 3. PROCUREMENT MANAGEMENT
-- ============================================================

CREATE TABLE `supplier` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `supplier_code` VARCHAR(50) UNIQUE NOT NULL,
  `supplier_name` VARCHAR(255) NOT NULL,
  `company_name` VARCHAR(255),
  `email` VARCHAR(100),
  `phone` VARCHAR(50),
  `mobile` VARCHAR(50),
  `website` VARCHAR(255),
  `tax_id` VARCHAR(100),
  `payment_terms` VARCHAR(100) COMMENT 'e.g., Net 30, Net 60',
  `credit_limit` DECIMAL(15,2),
  `currency` VARCHAR(10) DEFAULT 'USD',
  `address` TEXT,
  `city` VARCHAR(100),
  `state` VARCHAR(100),
  `country` VARCHAR(100),
  `postal_code` VARCHAR(20),
  `contact_person` VARCHAR(255),
  `contact_phone` VARCHAR(50),
  `contact_email` VARCHAR(100),
  `rating` DECIMAL(3,2) COMMENT 'Supplier rating 1-5',
  `lead_time_days` INT DEFAULT 0,
  `is_active` BOOLEAN DEFAULT TRUE,
  `notes` TEXT,
  `org_id` VARCHAR(255),
  `created_by` INT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  INDEX `idx_supplier_org` (`org_id`, `is_active`),
  INDEX `idx_supplier_code` (`supplier_code`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `supplier_product` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `supplier_id` INT NOT NULL,
  `product_id` INT NOT NULL,
  `supplier_product_code` VARCHAR(100),
  `supplier_product_name` VARCHAR(255),
  `unit_price` DECIMAL(15,2),
  `currency` VARCHAR(10) DEFAULT 'USD',
  `minimum_order_quantity` DECIMAL(15,3) DEFAULT 1,
  `lead_time_days` INT DEFAULT 0,
  `is_preferred` BOOLEAN DEFAULT FALSE,
  `last_purchase_price` DECIMAL(15,2),
  `last_purchase_date` DATE,
  `is_active` BOOLEAN DEFAULT TRUE,
  `notes` TEXT,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_supplier_product` (`supplier_id`, `product_id`),
  FOREIGN KEY (`supplier_id`) REFERENCES `supplier`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`) ON DELETE CASCADE,
  INDEX `idx_preferred_supplier` (`product_id`, `is_preferred`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `purchase_requisition` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `requisition_number` VARCHAR(50) UNIQUE NOT NULL,
  `requisition_date` DATE NOT NULL,
  `requested_by` INT NOT NULL,
  `department` VARCHAR(100),
  `priority` ENUM('low', 'medium', 'high', 'urgent') DEFAULT 'medium',
  `required_by_date` DATE,
  `justification` TEXT,
  `total_estimated_cost` DECIMAL(15,2),
  `status` ENUM('draft', 'pending', 'approved', 'rejected', 'partially_ordered', 'completed', 'cancelled') DEFAULT 'draft',
  `approved_by` INT,
  `approved_date` DATETIME,
  `rejected_reason` TEXT,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  INDEX `idx_requisition_status` (`status`, `requisition_date`),
  INDEX `idx_requisition_org` (`org_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `purchase_requisition_item` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `requisition_id` INT NOT NULL,
  `product_id` INT NOT NULL,
  `quantity_requested` DECIMAL(15,3) NOT NULL,
  `quantity_ordered` DECIMAL(15,3) DEFAULT 0,
  `estimated_unit_price` DECIMAL(15,2),
  `estimated_total` DECIMAL(15,2),
  `notes` TEXT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`requisition_id`) REFERENCES `purchase_requisition`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  INDEX `idx_requisition_items` (`requisition_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `purchase_order` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `po_number` VARCHAR(50) UNIQUE NOT NULL,
  `po_date` DATE NOT NULL,
  `supplier_id` INT NOT NULL,
  `requisition_id` INT,
  `warehouse_id` INT NOT NULL COMMENT 'Delivery warehouse',
  `expected_delivery_date` DATE,
  `payment_terms` VARCHAR(100),
  `shipping_method` VARCHAR(100),
  `shipping_cost` DECIMAL(15,2) DEFAULT 0,
  `tax_amount` DECIMAL(15,2) DEFAULT 0,
  `discount_amount` DECIMAL(15,2) DEFAULT 0,
  `subtotal` DECIMAL(15,2) NOT NULL,
  `total_amount` DECIMAL(15,2) NOT NULL,
  `currency` VARCHAR(10) DEFAULT 'USD',
  `notes` TEXT,
  `terms_conditions` TEXT,
  `status` ENUM('draft', 'sent', 'confirmed', 'partially_received', 'received', 'closed', 'cancelled') DEFAULT 'draft',
  `created_by` INT NOT NULL,
  `approved_by` INT,
  `approved_date` DATETIME,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`supplier_id`) REFERENCES `supplier`(`id`),
  FOREIGN KEY (`requisition_id`) REFERENCES `purchase_requisition`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_po_status` (`status`, `po_date`),
  INDEX `idx_po_supplier` (`supplier_id`, `po_date`),
  INDEX `idx_po_org` (`org_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `purchase_order_item` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `po_id` INT NOT NULL,
  `product_id` INT NOT NULL,
  `quantity_ordered` DECIMAL(15,3) NOT NULL,
  `quantity_received` DECIMAL(15,3) DEFAULT 0,
  `quantity_rejected` DECIMAL(15,3) DEFAULT 0,
  `unit_price` DECIMAL(15,2) NOT NULL,
  `tax_rate` DECIMAL(5,2) DEFAULT 0,
  `discount_rate` DECIMAL(5,2) DEFAULT 0,
  `line_total` DECIMAL(15,2) NOT NULL,
  `expected_delivery_date` DATE,
  `notes` TEXT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`po_id`) REFERENCES `purchase_order`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  INDEX `idx_po_items` (`po_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `goods_receipt` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `receipt_number` VARCHAR(50) UNIQUE NOT NULL,
  `receipt_date` DATETIME NOT NULL,
  `po_id` INT NOT NULL,
  `warehouse_id` INT NOT NULL,
  `received_by` INT NOT NULL,
  `carrier_name` VARCHAR(255),
  `tracking_number` VARCHAR(100),
  `delivery_note_number` VARCHAR(100),
  `quality_check_status` ENUM('pending', 'passed', 'failed', 'partial') DEFAULT 'pending',
  `quality_checked_by` INT,
  `quality_check_date` DATETIME,
  `notes` TEXT,
  `status` ENUM('draft', 'pending_qc', 'completed', 'cancelled') DEFAULT 'draft',
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`po_id`) REFERENCES `purchase_order`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_receipt_po` (`po_id`),
  INDEX `idx_receipt_date` (`receipt_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `goods_receipt_item` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `receipt_id` INT NOT NULL,
  `po_item_id` INT NOT NULL,
  `product_id` INT NOT NULL,
  `quantity_received` DECIMAL(15,3) NOT NULL,
  `quantity_accepted` DECIMAL(15,3) DEFAULT 0,
  `quantity_rejected` DECIMAL(15,3) DEFAULT 0,
  `batch_number` VARCHAR(100),
  `serial_numbers` TEXT COMMENT 'JSON array of serial numbers',
  `expiry_date` DATE,
  `zone_id` INT,
  `rejection_reason` TEXT,
  `notes` TEXT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`receipt_id`) REFERENCES `goods_receipt`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`po_item_id`) REFERENCES `purchase_order_item`(`id`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  INDEX `idx_receipt_items` (`receipt_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 4. STOCK ADJUSTMENTS & CYCLE COUNTING
-- ============================================================

CREATE TABLE `stock_adjustment` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `adjustment_number` VARCHAR(50) UNIQUE NOT NULL,
  `adjustment_date` DATE NOT NULL,
  `adjustment_type` ENUM('physical_count', 'damage', 'theft', 'expiry', 'quality', 'correction', 'write_off', 'found') NOT NULL,
  `warehouse_id` INT NOT NULL,
  `reason` TEXT NOT NULL,
  `notes` TEXT,
  `total_value_impact` DECIMAL(15,2) COMMENT 'Financial impact of adjustment',
  `status` ENUM('draft', 'pending_approval', 'approved', 'rejected', 'completed') DEFAULT 'draft',
  `created_by` INT NOT NULL,
  `approved_by` INT,
  `approved_date` DATETIME,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_adjustment_status` (`status`, `adjustment_date`),
  INDEX `idx_adjustment_type` (`adjustment_type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `stock_adjustment_item` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `adjustment_id` INT NOT NULL,
  `product_id` INT NOT NULL,
  `zone_id` INT,
  `batch_number` VARCHAR(100),
  `system_quantity` DECIMAL(15,3) NOT NULL COMMENT 'Quantity in system',
  `physical_quantity` DECIMAL(15,3) NOT NULL COMMENT 'Actual counted quantity',
  `adjustment_quantity` DECIMAL(15,3) NOT NULL COMMENT 'Difference (physical - system)',
  `unit_cost` DECIMAL(15,2),
  `total_value_impact` DECIMAL(15,2),
  `notes` TEXT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`adjustment_id`) REFERENCES `stock_adjustment`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  INDEX `idx_adjustment_items` (`adjustment_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `cycle_count` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `count_number` VARCHAR(50) UNIQUE NOT NULL,
  `count_date` DATE NOT NULL,
  `warehouse_id` INT NOT NULL,
  `zone_id` INT,
  `count_type` ENUM('full', 'partial', 'abc', 'random') DEFAULT 'partial',
  `scheduled_date` DATE,
  `frequency` ENUM('daily', 'weekly', 'monthly', 'quarterly', 'annual', 'ad_hoc') DEFAULT 'ad_hoc',
  `assigned_to` INT,
  `counted_by` INT,
  `reviewed_by` INT,
  `total_items_to_count` INT DEFAULT 0,
  `total_items_counted` INT DEFAULT 0,
  `total_discrepancies` INT DEFAULT 0,
  `accuracy_percentage` DECIMAL(5,2),
  `status` ENUM('scheduled', 'in_progress', 'completed', 'reviewed', 'cancelled') DEFAULT 'scheduled',
  `notes` TEXT,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_count_status` (`status`, `count_date`),
  INDEX `idx_count_warehouse` (`warehouse_id`, `count_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `cycle_count_item` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `cycle_count_id` INT NOT NULL,
  `product_id` INT NOT NULL,
  `zone_id` INT,
  `batch_number` VARCHAR(100),
  `system_quantity` DECIMAL(15,3),
  `counted_quantity` DECIMAL(15,3),
  `variance` DECIMAL(15,3) COMMENT 'counted - system',
  `variance_percentage` DECIMAL(5,2),
  `count_status` ENUM('pending', 'counted', 'verified', 'discrepancy') DEFAULT 'pending',
  `counted_at` DATETIME,
  `notes` TEXT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`cycle_count_id`) REFERENCES `cycle_count`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  INDEX `idx_cycle_items` (`cycle_count_id`),
  INDEX `idx_discrepancy` (`count_status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 5. INVENTORY TRANSFERS
-- ============================================================

CREATE TABLE `stock_transfer` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `transfer_number` VARCHAR(50) UNIQUE NOT NULL,
  `transfer_date` DATE NOT NULL,
  `from_warehouse_id` INT NOT NULL,
  `to_warehouse_id` INT NOT NULL,
  `expected_delivery_date` DATE,
  `actual_delivery_date` DATE,
  `transfer_type` ENUM('inter_warehouse', 'inter_zone', 'return', 'relocation') DEFAULT 'inter_warehouse',
  `shipping_method` VARCHAR(100),
  `tracking_number` VARCHAR(100),
  `shipping_cost` DECIMAL(15,2) DEFAULT 0,
  `reason` TEXT,
  `notes` TEXT,
  `status` ENUM('draft', 'pending', 'in_transit', 'received', 'partially_received', 'cancelled') DEFAULT 'draft',
  `created_by` INT NOT NULL,
  `approved_by` INT,
  `received_by` INT,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`from_warehouse_id`) REFERENCES `warehouse`(`id`),
  FOREIGN KEY (`to_warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_transfer_status` (`status`, `transfer_date`),
  INDEX `idx_transfer_warehouses` (`from_warehouse_id`, `to_warehouse_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `stock_transfer_item` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `transfer_id` INT NOT NULL,
  `product_id` INT NOT NULL,
  `from_zone_id` INT,
  `to_zone_id` INT,
  `batch_number` VARCHAR(100),
  `quantity_sent` DECIMAL(15,3) NOT NULL,
  `quantity_received` DECIMAL(15,3) DEFAULT 0,
  `quantity_damaged` DECIMAL(15,3) DEFAULT 0,
  `unit_cost` DECIMAL(15,2),
  `notes` TEXT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`transfer_id`) REFERENCES `stock_transfer`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  INDEX `idx_transfer_items` (`transfer_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 6. ALERTS & NOTIFICATIONS
-- ============================================================

CREATE TABLE `inventory_alert` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `alert_type` ENUM('low_stock', 'out_of_stock', 'overstock', 'expiry_warning', 'expired', 'reorder_point', 'slow_moving', 'fast_moving', 'quality_issue') NOT NULL,
  `product_id` INT NOT NULL,
  `warehouse_id` INT,
  `batch_id` INT,
  `severity` ENUM('low', 'medium', 'high', 'critical') DEFAULT 'medium',
  `message` TEXT NOT NULL,
  `current_quantity` DECIMAL(15,3),
  `threshold_quantity` DECIMAL(15,3),
  `expiry_date` DATE,
  `is_read` BOOLEAN DEFAULT FALSE,
  `is_resolved` BOOLEAN DEFAULT FALSE,
  `resolved_by` INT,
  `resolved_date` DATETIME,
  `resolution_notes` TEXT,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_alert_unresolved` (`is_resolved`, `severity`),
  INDEX `idx_alert_type` (`alert_type`, `created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 7. REPORTING & ANALYTICS VIEWS
-- ============================================================

CREATE TABLE `inventory_valuation` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `product_id` INT NOT NULL,
  `warehouse_id` INT NOT NULL,
  `valuation_date` DATE NOT NULL,
  `quantity` DECIMAL(15,3) NOT NULL,
  `average_cost` DECIMAL(15,2) NOT NULL,
  `total_value` DECIMAL(15,2) NOT NULL,
  `valuation_method` ENUM('FIFO', 'LIFO', 'WAC', 'Standard') DEFAULT 'WAC',
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_valuation_date` (`valuation_date`),
  INDEX `idx_valuation_product` (`product_id`, `valuation_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `inventory_turnover` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `product_id` INT NOT NULL,
  `warehouse_id` INT,
  `period_start` DATE NOT NULL,
  `period_end` DATE NOT NULL,
  `opening_stock` DECIMAL(15,3),
  `closing_stock` DECIMAL(15,3),
  `total_sold` DECIMAL(15,3),
  `total_purchased` DECIMAL(15,3),
  `average_stock` DECIMAL(15,3),
  `turnover_ratio` DECIMAL(10,2) COMMENT 'Sales / Average Stock',
  `days_to_sell` DECIMAL(10,2) COMMENT '365 / Turnover Ratio',
  `stock_status` ENUM('fast_moving', 'normal', 'slow_moving', 'obsolete') DEFAULT 'normal',
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_turnover_period` (`period_start`, `period_end`),
  INDEX `idx_turnover_product` (`product_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 8. DEMAND FORECASTING & PLANNING
-- ============================================================

CREATE TABLE `demand_forecast` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `product_id` INT NOT NULL,
  `warehouse_id` INT,
  `forecast_period` DATE NOT NULL COMMENT 'Month/Week start date',
  `forecast_type` ENUM('monthly', 'weekly', 'daily', 'quarterly') DEFAULT 'monthly',
  `forecasted_quantity` DECIMAL(15,3) NOT NULL,
  `actual_quantity` DECIMAL(15,3),
  `forecast_accuracy` DECIMAL(5,2) COMMENT 'Percentage accuracy',
  `forecast_method` VARCHAR(100) COMMENT 'e.g., Moving Average, Linear Regression',
  `confidence_level` DECIMAL(5,2),
  `seasonal_factor` DECIMAL(5,2),
  `trend_factor` DECIMAL(5,2),
  `notes` TEXT,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_forecast` (`product_id`, `warehouse_id`, `forecast_period`, `forecast_type`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_forecast_period` (`forecast_period`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `reorder_recommendation` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `product_id` INT NOT NULL,
  `warehouse_id` INT NOT NULL,
  `recommendation_date` DATE NOT NULL,
  `current_stock` DECIMAL(15,3),
  `reorder_point` DECIMAL(15,3),
  `economic_order_quantity` DECIMAL(15,3) COMMENT 'EOQ',
  `recommended_quantity` DECIMAL(15,3),
  `forecasted_demand` DECIMAL(15,3),
  `lead_time_days` INT,
  `safety_stock` DECIMAL(15,3),
  `estimated_stockout_date` DATE,
  `priority` ENUM('low', 'medium', 'high', 'critical') DEFAULT 'medium',
  `status` ENUM('pending', 'ordered', 'ignored', 'cancelled') DEFAULT 'pending',
  `po_id` INT COMMENT 'Reference to created PO',
  `notes` TEXT,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  FOREIGN KEY (`po_id`) REFERENCES `purchase_order`(`id`),
  INDEX `idx_recommendation_status` (`status`, `priority`),
  INDEX `idx_recommendation_date` (`recommendation_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 9. QUALITY CONTROL
-- ============================================================

CREATE TABLE `quality_check` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `check_number` VARCHAR(50) UNIQUE NOT NULL,
  `check_date` DATETIME NOT NULL,
  `check_type` ENUM('incoming', 'in_process', 'outgoing', 'periodic', 'complaint') NOT NULL,
  `product_id` INT NOT NULL,
  `batch_number` VARCHAR(100),
  `warehouse_id` INT,
  `reference_type` VARCHAR(50) COMMENT 'e.g., GRN, Transfer, Production',
  `reference_id` INT,
  `quantity_checked` DECIMAL(15,3),
  `quantity_passed` DECIMAL(15,3),
  `quantity_failed` DECIMAL(15,3),
  `inspector_id` INT NOT NULL,
  `inspection_criteria` TEXT COMMENT 'JSON of criteria checked',
  `defects_found` TEXT COMMENT 'JSON array of defects',
  `result` ENUM('passed', 'failed', 'conditional', 'pending') DEFAULT 'pending',
  `action_taken` ENUM('accept', 'reject', 'rework', 'quarantine', 'return_supplier') DEFAULT 'accept',
  `notes` TEXT,
  `attachments` TEXT COMMENT 'JSON array of file paths',
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_qc_result` (`result`, `check_date`),
  INDEX `idx_qc_product` (`product_id`, `check_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 10. BARCODE & LABELING
-- ============================================================

CREATE TABLE `barcode_generation` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `entity_type` ENUM('product', 'batch', 'serial', 'location', 'pallet', 'package') NOT NULL,
  `entity_id` INT NOT NULL,
  `barcode` VARCHAR(255) UNIQUE NOT NULL,
  `barcode_type` VARCHAR(50) DEFAULT 'CODE128' COMMENT 'EAN13, CODE128, QR, etc',
  `barcode_image` TEXT COMMENT 'Base64 or file path',
  `print_count` INT DEFAULT 0,
  `last_printed_date` DATETIME,
  `is_active` BOOLEAN DEFAULT TRUE,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  INDEX `idx_barcode` (`barcode`),
  INDEX `idx_entity` (`entity_type`, `entity_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `label_template` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `template_name` VARCHAR(255) NOT NULL,
  `template_type` ENUM('product', 'shelf', 'pallet', 'package', 'shipping') NOT NULL,
  `paper_size` VARCHAR(50) COMMENT 'e.g., 4x6, A4, Letter',
  `template_content` TEXT NOT NULL COMMENT 'HTML/JSON template',
  `is_default` BOOLEAN DEFAULT FALSE,
  `is_active` BOOLEAN DEFAULT TRUE,
  `org_id` VARCHAR(255),
  `created_by` INT,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  INDEX `idx_template_type` (`template_type`, `is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 11. KIT/BUNDLE MANAGEMENT
-- ============================================================

CREATE TABLE `product_kit` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `kit_product_id` INT NOT NULL COMMENT 'The kit/bundle product ID',
  `component_product_id` INT NOT NULL COMMENT 'Component product ID',
  `quantity_required` DECIMAL(15,3) NOT NULL DEFAULT 1,
  `is_optional` BOOLEAN DEFAULT FALSE,
  `sort_order` INT DEFAULT 0,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_kit_component` (`kit_product_id`, `component_product_id`),
  FOREIGN KEY (`kit_product_id`) REFERENCES `tblproduct`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`component_product_id`) REFERENCES `tblproduct`(`id`) ON DELETE CASCADE,
  INDEX `idx_kit_components` (`kit_product_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 12. INTEGRATIONS & AUDIT
-- ============================================================

CREATE TABLE `inventory_audit_log` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `entity_type` VARCHAR(100) NOT NULL COMMENT 'Table name',
  `entity_id` INT NOT NULL,
  `action` ENUM('create', 'update', 'delete', 'approve', 'reject', 'cancel') NOT NULL,
  `field_name` VARCHAR(100),
  `old_value` TEXT,
  `new_value` TEXT,
  `user_id` INT NOT NULL,
  `ip_address` VARCHAR(45),
  `user_agent` TEXT,
  `reason` TEXT,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  INDEX `idx_audit_entity` (`entity_type`, `entity_id`),
  INDEX `idx_audit_user` (`user_id`, `created_at`),
  INDEX `idx_audit_date` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE `integration_sync` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `integration_name` VARCHAR(100) NOT NULL COMMENT 'e.g., Shopify, QuickBooks, SAP',
  `entity_type` VARCHAR(100) NOT NULL,
  `entity_id` INT NOT NULL,
  `external_id` VARCHAR(255) COMMENT 'ID in external system',
  `sync_type` ENUM('import', 'export', 'bidirectional') NOT NULL,
  `sync_status` ENUM('pending', 'success', 'failed', 'partial') NOT NULL,
  `last_sync_date` DATETIME,
  `error_message` TEXT,
  `retry_count` INT DEFAULT 0,
  `sync_data` TEXT COMMENT 'JSON of synced data',
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  INDEX `idx_sync_entity` (`entity_type`, `entity_id`),
  INDEX `idx_sync_status` (`sync_status`, `last_sync_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 13. ANALYTICS SUMMARY TABLES (for performance)
-- ============================================================

CREATE TABLE `inventory_snapshot` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `snapshot_date` DATE NOT NULL,
  `product_id` INT NOT NULL,
  `warehouse_id` INT NOT NULL,
  `quantity` DECIMAL(15,3) NOT NULL,
  `value` DECIMAL(15,2) NOT NULL,
  `turnover_ratio` DECIMAL(10,2),
  `days_on_hand` INT,
  `org_id` VARCHAR(255),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_snapshot` (`snapshot_date`, `product_id`, `warehouse_id`),
  FOREIGN KEY (`product_id`) REFERENCES `tblproduct`(`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse`(`id`),
  INDEX `idx_snapshot_date` (`snapshot_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================================
-- USEFUL VIEWS FOR REPORTING
-- ============================================================

-- View: Current Stock Levels across all warehouses
CREATE OR REPLACE VIEW vw_current_stock_levels AS
SELECT 
    p.id as product_id,
    p.product_code,
    p.name as product_name,
    p.category,
    w.id as warehouse_id,
    w.warehouse_name,
    sb.quantity_on_hand,
    sb.quantity_allocated,
    sb.quantity_available,
    sb.quantity_in_transit,
    sb.reorder_point,
    CASE 
        WHEN sb.quantity_available <= 0 THEN 'Out of Stock'
        WHEN sb.quantity_available <= sb.reorder_point THEN 'Low Stock'
        WHEN sb.quantity_available > sb.max_stock_level THEN 'Overstock'
        ELSE 'Normal'
    END as stock_status,
    sb.average_cost,
    (sb.quantity_on_hand * sb.average_cost) as total_value,
    sb.last_movement_date,
    p.org_id
FROM tblproduct p
JOIN stock_balance sb ON p.id = sb.product_id
JOIN warehouse w ON sb.warehouse_id = w.id
WHERE p.status = 'active' AND w.is_active = TRUE;

-- View: Products needing reorder
CREATE OR REPLACE VIEW vw_reorder_alerts AS
SELECT 
    p.id as product_id,
    p.product_code,
    p.name as product_name,
    w.warehouse_name,
    sb.quantity_available,
    sb.reorder_point,
    (sb.reorder_point - sb.quantity_available) as quantity_to_order,
    sp.supplier_name,
    sp.lead_time_days,
    spr.unit_price,
    spr.minimum_order_quantity,
    p.org_id
FROM tblproduct p
JOIN stock_balance sb ON p.id = sb.product_id
JOIN warehouse w ON sb.warehouse_id = w.id
LEFT JOIN supplier_product spr ON p.id = spr.product_id AND spr.is_preferred = TRUE
LEFT JOIN supplier sp ON spr.supplier_id = sp.id
WHERE sb.quantity_available <= sb.reorder_point
    AND p.status = 'active'
    AND w.is_active = TRUE;

-- View: Expiring Stock
CREATE OR REPLACE VIEW vw_expiring_stock AS
SELECT 
    p.id as product_id,
    p.product_code,
    p.name as product_name,
    w.warehouse_name,
    sb.batch_number,
    sb.quantity,
    sb.expiry_date,
    DATEDIFF(sb.expiry_date, CURDATE()) as days_to_expiry,
    CASE 
        WHEN DATEDIFF(sb.expiry_date, CURDATE()) < 0 THEN 'Expired'
        WHEN DATEDIFF(sb.expiry_date, CURDATE()) <= 30 THEN 'Critical'
        WHEN DATEDIFF(sb.expiry_date, CURDATE()) <= 90 THEN 'Warning'
        ELSE 'Normal'
    END as expiry_status,
    (sb.quantity * sb.unit_cost) as total_value_at_risk,
    p.org_id
FROM tblproduct p
JOIN stock_batch sb ON p.id = sb.product_id
JOIN warehouse w ON sb.warehouse_id = w.id
WHERE sb.expiry_date IS NOT NULL
    AND sb.is_active = TRUE
    AND DATEDIFF(sb.expiry_date, CURDATE()) <= 90;

-- View: Purchase Order Status Summary
CREATE OR REPLACE VIEW vw_purchase_order_summary AS
SELECT 
    po.id,
    po.po_number,
    po.po_date,
    s.supplier_name,
    po.total_amount,
    po.status,
    po.expected_delivery_date,
    COUNT(poi.id) as total_items,
    SUM(poi.quantity_ordered) as total_quantity_ordered,
    SUM(poi.quantity_received) as total_quantity_received,
    CASE 
        WHEN SUM(poi.quantity_received) = 0 THEN 0
        ELSE (SUM(poi.quantity_received) / SUM(poi.quantity_ordered)) * 100
    END as fulfillment_percentage,
    po.org_id
FROM purchase_order po
JOIN supplier s ON po.supplier_id = s.id
LEFT JOIN purchase_order_item poi ON po.id = poi.po_id
GROUP BY po.id;

-- View: Inventory Valuation by Category
CREATE OR REPLACE VIEW vw_inventory_valuation_by_category AS
SELECT 
    p.category,
    COUNT(DISTINCT p.id) as total_products,
    SUM(sb.quantity_on_hand) as total_quantity,
    SUM(sb.quantity_on_hand * sb.average_cost) as total_value,
    AVG(sb.average_cost) as avg_unit_cost,
    p.org_id
FROM tblproduct p
JOIN stock_balance sb ON p.id = sb.product_id
WHERE p.status = 'active'
GROUP BY p.category, p.org_id;

-- View: Warehouse Capacity Utilization
CREATE OR REPLACE VIEW vw_warehouse_utilization AS
SELECT 
    w.id,
    w.warehouse_name,
    w.capacity,
    COUNT(DISTINCT sb.product_id) as total_products,
    SUM(sb.quantity_on_hand) as total_quantity,
    SUM(sb.quantity_on_hand * sb.average_cost) as total_inventory_value,
    w.current_utilization,
    CASE 
        WHEN w.current_utilization >= 90 THEN 'Critical'
        WHEN w.current_utilization >= 75 THEN 'High'
        WHEN w.current_utilization >= 50 THEN 'Medium'
        ELSE 'Low'
    END as utilization_status,
    w.org_id
FROM warehouse w
LEFT JOIN stock_balance sb ON w.id = sb.warehouse_id
WHERE w.is_active = TRUE
GROUP BY w.id;

-- ============================================================
-- STORED PROCEDURES FOR COMMON OPERATIONS
-- ============================================================

DELIMITER //

-- Procedure: Create Stock Movement and Update Balance
CREATE PROCEDURE sp_create_stock_movement(
    IN p_product_id INT,
    IN p_movement_type VARCHAR(50),
    IN p_from_warehouse_id INT,
    IN p_to_warehouse_id INT,
    IN p_quantity DECIMAL(15,3),
    IN p_unit_cost DECIMAL(15,2),
    IN p_reference_type VARCHAR(50),
    IN p_reference_id INT,
    IN p_batch_number VARCHAR(100),
    IN p_user_id INT,
    IN p_org_id VARCHAR(255),
    OUT p_movement_id INT
)
BEGIN
    DECLARE v_movement_code VARCHAR(50);
    
    -- Generate movement code
    SET v_movement_code = CONCAT('MOV-', DATE_FORMAT(NOW(), '%Y%m%d'), '-', LPAD(FLOOR(RAND() * 10000), 4, '0'));
    
    -- Insert movement record
    INSERT INTO stock_movement (
        movement_code, product_id, movement_type, reference_type, reference_id,
        from_warehouse_id, to_warehouse_id, quantity, unit_cost, 
        total_cost, batch_number, movement_date, processed_by, 
        status, org_id
    ) VALUES (
        v_movement_code, p_product_id, p_movement_type, p_reference_type, p_reference_id,
        p_from_warehouse_id, p_to_warehouse_id, p_quantity, p_unit_cost,
        (p_quantity * p_unit_cost), p_batch_number, NOW(), p_user_id,
        'completed', p_org_id
    );
    
    SET p_movement_id = LAST_INSERT_ID();
    
    -- Update stock balance (deduct from source)
    IF p_from_warehouse_id IS NOT NULL THEN
        UPDATE stock_balance 
        SET quantity_on_hand = quantity_on_hand - p_quantity,
            quantity_available = quantity_available - p_quantity,
            last_movement_date = NOW()
        WHERE product_id = p_product_id 
        AND warehouse_id = p_from_warehouse_id;
    END IF;
    
    -- Update stock balance (add to destination)
    IF p_to_warehouse_id IS NOT NULL THEN
        INSERT INTO stock_balance (
            product_id, warehouse_id, quantity_on_hand, quantity_available, 
            average_cost, last_movement_date, org_id
        ) VALUES (
            p_product_id, p_to_warehouse_id, p_quantity, p_quantity,
            p_unit_cost, NOW(), p_org_id
        )
        ON DUPLICATE KEY UPDATE
            quantity_on_hand = quantity_on_hand + p_quantity,
            quantity_available = quantity_available + p_quantity,
            average_cost = ((average_cost * quantity_on_hand) + (p_unit_cost * p_quantity)) / (quantity_on_hand + p_quantity),
            last_movement_date = NOW();
    END IF;
END//

-- Procedure: Calculate Reorder Recommendations
CREATE PROCEDURE sp_calculate_reorder_recommendations(
    IN p_org_id VARCHAR(255)
)
BEGIN
    INSERT INTO reorder_recommendation (
        product_id, warehouse_id, recommendation_date, current_stock,
        reorder_point, recommended_quantity, lead_time_days, priority, org_id
    )
    SELECT 
        sb.product_id,
        sb.warehouse_id,
        CURDATE(),
        sb.quantity_available,
        sb.reorder_point,
        (sb.reorder_point * 2) - sb.quantity_available as recommended_qty,
        COALESCE(sp.lead_time_days, p.lead_time, 0),
        CASE 
            WHEN sb.quantity_available <= 0 THEN 'critical'
            WHEN sb.quantity_available <= (sb.reorder_point * 0.5) THEN 'high'
            ELSE 'medium'
        END as priority,
        p.org_id
    FROM stock_balance sb
    JOIN tblproduct p ON sb.product_id = p.id
    LEFT JOIN supplier_product sp ON p.id = sp.product_id AND sp.is_preferred = TRUE
    WHERE sb.quantity_available <= sb.reorder_point
    AND p.status = 'active'
    AND p.org_id = p_org_id
    AND NOT EXISTS (
        SELECT 1 FROM reorder_recommendation rr
        WHERE rr.product_id = sb.product_id
        AND rr.warehouse_id = sb.warehouse_id
        AND rr.status = 'pending'
        AND rr.recommendation_date >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
    );
END//

DELIMITER ;

-- ============================================================
-- TRIGGERS FOR AUTOMATIC ALERTS
-- ============================================================

DELIMITER //

CREATE TRIGGER trg_check_low_stock_after_movement
AFTER INSERT ON stock_movement
FOR EACH ROW
BEGIN
    DECLARE v_current_qty DECIMAL(15,3);
    DECLARE v_reorder_point DECIMAL(15,3);
    
    -- Get current stock and reorder point
    SELECT quantity_available, reorder_point 
    INTO v_current_qty, v_reorder_point
    FROM stock_balance
    WHERE product_id = NEW.product_id 
    AND warehouse_id = COALESCE(NEW.to_warehouse_id, NEW.from_warehouse_id)
    LIMIT 1;
    
    -- Create alert if below reorder point
    IF v_current_qty <= v_reorder_point THEN
        INSERT INTO inventory_alert (
            alert_type, product_id, warehouse_id, severity, message,
            current_quantity, threshold_quantity, org_id
        ) VALUES (
            'low_stock', NEW.product_id, COALESCE(NEW.to_warehouse_id, NEW.from_warehouse_id),
            CASE WHEN v_current_qty <= 0 THEN 'critical' ELSE 'high' END,
            CONCAT('Product stock is below reorder point'),
            v_current_qty, v_reorder_point, NEW.org_id
        );
    END IF;
END//

DELIMITER ;