 
CREATE TABLE `cycle_count` (
  `id` int NOT NULL,
  `count_number` varchar(50) NOT NULL,
  `count_date` date NOT NULL,
  `warehouse_id` int NOT NULL,
  `zone_id` int DEFAULT NULL,
  `count_type` enum('full','partial','abc','random') DEFAULT 'partial',
  `scheduled_date` date DEFAULT NULL,
  `frequency` enum('daily','weekly','monthly','quarterly','annual','ad_hoc') DEFAULT 'ad_hoc',
  `assigned_to` int DEFAULT NULL,
  `counted_by` int DEFAULT NULL,
  `reviewed_by` int DEFAULT NULL,
  `total_items_to_count` int DEFAULT '0',
  `total_items_counted` int DEFAULT '0',
  `total_discrepancies` int DEFAULT '0',
  `accuracy_percentage` decimal(5,2) DEFAULT NULL,
  `status` enum('scheduled','in_progress','completed','reviewed','cancelled') DEFAULT 'scheduled',
  `notes` text,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `cycle_count_item`
--

CREATE TABLE `cycle_count_item` (
  `id` int NOT NULL,
  `cycle_count_id` int NOT NULL,
  `product_id` int NOT NULL,
  `zone_id` int DEFAULT NULL,
  `batch_number` varchar(100) DEFAULT NULL,
  `system_quantity` decimal(15,3) DEFAULT NULL,
  `counted_quantity` decimal(15,3) DEFAULT NULL,
  `variance` decimal(15,3) DEFAULT NULL COMMENT 'counted - system',
  `variance_percentage` decimal(5,2) DEFAULT NULL,
  `count_status` enum('pending','counted','verified','discrepancy') DEFAULT 'pending',
  `counted_at` datetime DEFAULT NULL,
  `notes` text,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `demand_forecast`
--

CREATE TABLE `demand_forecast` (
  `id` int NOT NULL,
  `product_id` int NOT NULL,
  `warehouse_id` int DEFAULT NULL,
  `forecast_period` date NOT NULL COMMENT 'Month/Week start date',
  `forecast_type` enum('monthly','weekly','daily','quarterly') DEFAULT 'monthly',
  `forecasted_quantity` decimal(15,3) NOT NULL,
  `actual_quantity` decimal(15,3) DEFAULT NULL,
  `forecast_accuracy` decimal(5,2) DEFAULT NULL COMMENT 'Percentage accuracy',
  `forecast_method` varchar(100) DEFAULT NULL COMMENT 'e.g., Moving Average, Linear Regression',
  `confidence_level` decimal(5,2) DEFAULT NULL,
  `seasonal_factor` decimal(5,2) DEFAULT NULL,
  `trend_factor` decimal(5,2) DEFAULT NULL,
  `notes` text,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `goods_receipt`
--

CREATE TABLE `goods_receipt` (
  `id` int NOT NULL,
  `receipt_number` varchar(50) NOT NULL,
  `receipt_date` datetime NOT NULL,
  `po_id` int NOT NULL,
  `warehouse_id` int NOT NULL,
  `received_by` int NOT NULL,
  `carrier_name` varchar(255) DEFAULT NULL,
  `tracking_number` varchar(100) DEFAULT NULL,
  `delivery_note_number` varchar(100) DEFAULT NULL,
  `quality_check_status` enum('pending','passed','failed','partial') DEFAULT 'pending',
  `quality_checked_by` int DEFAULT NULL,
  `quality_check_date` datetime DEFAULT NULL,
  `notes` text,
  `status` enum('draft','pending_qc','completed','cancelled') DEFAULT 'draft',
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `goods_receipt_item`
--

CREATE TABLE `goods_receipt_item` (
  `id` int NOT NULL,
  `receipt_id` int NOT NULL,
  `po_item_id` int NOT NULL,
  `product_id` int NOT NULL,
  `quantity_received` decimal(15,3) NOT NULL,
  `quantity_accepted` decimal(15,3) DEFAULT '0.000',
  `quantity_rejected` decimal(15,3) DEFAULT '0.000',
  `batch_number` varchar(100) DEFAULT NULL,
  `serial_numbers` text COMMENT 'JSON array of serial numbers',
  `expiry_date` date DEFAULT NULL,
  `zone_id` int DEFAULT NULL,
  `rejection_reason` text,
  `notes` text,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `inventory_alert`
--

CREATE TABLE `inventory_alert` (
  `id` int NOT NULL,
  `alert_type` enum('low_stock','out_of_stock','overstock','expiry_warning','expired','reorder_point','slow_moving','fast_moving','quality_issue') NOT NULL,
  `product_id` int NOT NULL,
  `warehouse_id` int DEFAULT NULL,
  `batch_id` int DEFAULT NULL,
  `severity` enum('low','medium','high','critical') DEFAULT 'medium',
  `message` text NOT NULL,
  `current_quantity` decimal(15,3) DEFAULT NULL,
  `threshold_quantity` decimal(15,3) DEFAULT NULL,
  `expiry_date` date DEFAULT NULL,
  `is_read` tinyint(1) DEFAULT '0',
  `is_resolved` tinyint(1) DEFAULT '0',
  `resolved_by` int DEFAULT NULL,
  `resolved_date` datetime DEFAULT NULL,
  `resolution_notes` text,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `inventory_audit_log`
--

CREATE TABLE `inventory_audit_log` (
  `id` int NOT NULL,
  `entity_type` varchar(100) NOT NULL COMMENT 'Table name',
  `entity_id` int NOT NULL,
  `action` enum('create','update','delete','approve','reject','cancel') NOT NULL,
  `field_name` varchar(100) DEFAULT NULL,
  `old_value` text,
  `new_value` text,
  `user_id` int NOT NULL,
  `ip_address` varchar(45) DEFAULT NULL,
  `user_agent` text,
  `reason` text,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `inventory_snapshot`
--

CREATE TABLE `inventory_snapshot` (
  `id` int NOT NULL,
  `snapshot_date` date NOT NULL,
  `product_id` int NOT NULL,
  `warehouse_id` int NOT NULL,
  `quantity` decimal(15,3) NOT NULL,
  `value` decimal(15,2) NOT NULL,
  `turnover_ratio` decimal(10,2) DEFAULT NULL,
  `days_on_hand` int DEFAULT NULL,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `inventory_turnover`
--

CREATE TABLE `inventory_turnover` (
  `id` int NOT NULL,
  `product_id` int NOT NULL,
  `warehouse_id` int DEFAULT NULL,
  `period_start` date NOT NULL,
  `period_end` date NOT NULL,
  `opening_stock` decimal(15,3) DEFAULT NULL,
  `closing_stock` decimal(15,3) DEFAULT NULL,
  `total_sold` decimal(15,3) DEFAULT NULL,
  `total_purchased` decimal(15,3) DEFAULT NULL,
  `average_stock` decimal(15,3) DEFAULT NULL,
  `turnover_ratio` decimal(10,2) DEFAULT NULL COMMENT 'Sales / Average Stock',
  `days_to_sell` decimal(10,2) DEFAULT NULL COMMENT '365 / Turnover Ratio',
  `stock_status` enum('fast_moving','normal','slow_moving','obsolete') DEFAULT 'normal',
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `inventory_valuation`
--

CREATE TABLE `inventory_valuation` (
  `id` int NOT NULL,
  `product_id` int NOT NULL,
  `warehouse_id` int NOT NULL,
  `valuation_date` date NOT NULL,
  `quantity` decimal(15,3) NOT NULL,
  `average_cost` decimal(15,2) NOT NULL,
  `total_value` decimal(15,2) NOT NULL,
  `valuation_method` enum('FIFO','LIFO','WAC','Standard') DEFAULT 'WAC',
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `product_kit`
--

CREATE TABLE `product_kit` (
  `id` int NOT NULL,
  `kit_product_id` int NOT NULL COMMENT 'The kit/bundle product ID',
  `component_product_id` int NOT NULL COMMENT 'Component product ID',
  `quantity_required` decimal(15,3) NOT NULL DEFAULT '1.000',
  `is_optional` tinyint(1) DEFAULT '0',
  `sort_order` int DEFAULT '0',
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `purchase_order`
--

CREATE TABLE `purchase_order` (
  `id` int NOT NULL,
  `po_number` varchar(50) NOT NULL,
  `po_date` date NOT NULL,
  `supplier_id` int NOT NULL,
  `requisition_id` int DEFAULT NULL,
  `warehouse_id` int DEFAULT NULL COMMENT 'Delivery warehouse',
  `department_id` int NOT NULL,
  `expected_delivery_date` date DEFAULT NULL,
  `payment_terms` varchar(100) DEFAULT NULL,
  `shipping_method` varchar(100) DEFAULT NULL,
  `shipping_cost` decimal(15,2) DEFAULT '0.00',
  `tax_amount` decimal(15,2) DEFAULT '0.00',
  `discount_amount` decimal(15,2) DEFAULT '0.00',
  `subtotal` decimal(15,2) NOT NULL,
  `total_amount` decimal(15,2) NOT NULL,
  `currency` varchar(10) DEFAULT 'USD',
  `notes` text,
  `terms_conditions` text,
  `status` enum('draft','sent','confirmed','partially_received','received','closed','cancelled') DEFAULT 'draft',
  `priority` varchar(255) NOT NULL,
  `created_by` int NOT NULL,
  `approved_by` int DEFAULT NULL,
  `approved_date` datetime DEFAULT NULL,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `purchase_order_item`
--

CREATE TABLE `purchase_order_item` (
  `id` int NOT NULL,
  `po_id` int NOT NULL,
  `product_id` int NOT NULL,
  `quantity_ordered` decimal(15,3) NOT NULL,
  `quantity_received` decimal(15,3) DEFAULT '0.000',
  `quantity_rejected` decimal(15,3) DEFAULT '0.000',
  `unit_price` decimal(15,2) NOT NULL,
  `tax_rate` decimal(5,2) DEFAULT '0.00',
  `discount_rate` decimal(5,2) DEFAULT '0.00',
  `line_total` decimal(15,2) NOT NULL,
  `expected_delivery_date` date DEFAULT NULL,
  `notes` text,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `purchase_requisition`
--

CREATE TABLE `purchase_requisition` (
  `id` int NOT NULL,
  `requisition_number` varchar(50) NOT NULL,
  `requisition_date` date NOT NULL,
  `requested_by` int NOT NULL,
  `department` varchar(100) DEFAULT NULL,
  `priority` enum('low','medium','high','urgent') DEFAULT 'medium',
  `required_by_date` date DEFAULT NULL,
  `justification` text,
  `total_estimated_cost` decimal(15,2) DEFAULT NULL,
  `status` enum('draft','pending','approved','rejected','partially_ordered','completed','cancelled') DEFAULT 'draft',
  `approved_by` int DEFAULT NULL,
  `approved_date` datetime DEFAULT NULL,
  `rejected_reason` text,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `purchase_requisition_item`
--

CREATE TABLE `purchase_requisition_item` (
  `id` int NOT NULL,
  `requisition_id` int NOT NULL,
  `product_id` int NOT NULL,
  `quantity_requested` decimal(15,3) NOT NULL,
  `quantity_ordered` decimal(15,3) DEFAULT '0.000',
  `estimated_unit_price` decimal(15,2) DEFAULT NULL,
  `estimated_total` decimal(15,2) DEFAULT NULL,
  `notes` text,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `quality_check`
--

CREATE TABLE `quality_check` (
  `id` int NOT NULL,
  `check_number` varchar(50) NOT NULL,
  `check_date` datetime NOT NULL,
  `check_type` enum('incoming','in_process','outgoing','periodic','complaint') NOT NULL,
  `product_id` int NOT NULL,
  `batch_number` varchar(100) DEFAULT NULL,
  `warehouse_id` int DEFAULT NULL,
  `reference_type` varchar(50) DEFAULT NULL COMMENT 'e.g., GRN, Transfer, Production',
  `reference_id` int DEFAULT NULL,
  `quantity_checked` decimal(15,3) DEFAULT NULL,
  `quantity_passed` decimal(15,3) DEFAULT NULL,
  `quantity_failed` decimal(15,3) DEFAULT NULL,
  `inspector_id` int NOT NULL,
  `inspection_criteria` text COMMENT 'JSON of criteria checked',
  `defects_found` text COMMENT 'JSON array of defects',
  `result` enum('passed','failed','conditional','pending') DEFAULT 'pending',
  `action_taken` enum('accept','reject','rework','quarantine','return_supplier') DEFAULT 'accept',
  `notes` text,
  `attachments` text COMMENT 'JSON array of file paths',
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `reorder_recommendation`
--

CREATE TABLE `reorder_recommendation` (
  `id` int NOT NULL,
  `product_id` int NOT NULL,
  `warehouse_id` int NOT NULL,
  `recommendation_date` date NOT NULL,
  `current_stock` decimal(15,3) DEFAULT NULL,
  `reorder_point` decimal(15,3) DEFAULT NULL,
  `economic_order_quantity` decimal(15,3) DEFAULT NULL COMMENT 'EOQ',
  `recommended_quantity` decimal(15,3) DEFAULT NULL,
  `forecasted_demand` decimal(15,3) DEFAULT NULL,
  `lead_time_days` int DEFAULT NULL,
  `safety_stock` decimal(15,3) DEFAULT NULL,
  `estimated_stockout_date` date DEFAULT NULL,
  `priority` enum('low','medium','high','critical') DEFAULT 'medium',
  `status` enum('pending','ordered','ignored','cancelled') DEFAULT 'pending',
  `po_id` int DEFAULT NULL COMMENT 'Reference to created PO',
  `notes` text,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `serial_number`
--

CREATE TABLE `serial_number` (
  `id` int NOT NULL,
  `product_id` int NOT NULL,
  `serial_number` varchar(100) NOT NULL,
  `batch_id` int DEFAULT NULL,
  `warehouse_id` int DEFAULT NULL,
  `zone_id` int DEFAULT NULL,
  `status` enum('in_stock','sold','reserved','damaged','returned') DEFAULT 'in_stock',
  `purchase_order_id` int DEFAULT NULL,
  `sales_order_id` int DEFAULT NULL,
  `warranty_start_date` date DEFAULT NULL,
  `warranty_end_date` date DEFAULT NULL,
  `customer_id` int DEFAULT NULL,
  `notes` text,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `stock_adjustment`
--

CREATE TABLE `stock_adjustment` (
  `id` int NOT NULL,
  `adjustment_number` varchar(50) NOT NULL,
  `adjustment_date` date NOT NULL,
  `adjustment_type` enum('physical_count','damage','theft','expiry','quality','correction','write_off','found') NOT NULL,
  `warehouse_id` int NOT NULL,
  `reason` text NOT NULL,
  `notes` text,
  `total_value_impact` decimal(15,2) DEFAULT NULL COMMENT 'Financial impact of adjustment',
  `status` enum('draft','pending_approval','approved','rejected','completed') DEFAULT 'draft',
  `created_by` int NOT NULL,
  `approved_by` int DEFAULT NULL,
  `approved_date` datetime DEFAULT NULL,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `stock_adjustment_item`
--

CREATE TABLE `stock_adjustment_item` (
  `id` int NOT NULL,
  `adjustment_id` int NOT NULL,
  `product_id` int NOT NULL,
  `zone_id` int DEFAULT NULL,
  `batch_number` varchar(100) DEFAULT NULL,
  `system_quantity` decimal(15,3) NOT NULL COMMENT 'Quantity in system',
  `physical_quantity` decimal(15,3) NOT NULL COMMENT 'Actual counted quantity',
  `adjustment_quantity` decimal(15,3) NOT NULL COMMENT 'Difference (physical - system)',
  `unit_cost` decimal(15,2) DEFAULT NULL,
  `total_value_impact` decimal(15,2) DEFAULT NULL,
  `notes` text,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `stock_balance`
--

CREATE TABLE `stock_balance` (
  `id` int NOT NULL,
  `product_id` int NOT NULL,
  `warehouse_id` int NOT NULL,
  `zone_id` int DEFAULT NULL,
  `quantity_on_hand` decimal(15,3) DEFAULT '0.000',
  `quantity_allocated` decimal(15,3) DEFAULT '0.000' COMMENT 'Reserved for orders',
  `quantity_available` decimal(15,3) DEFAULT '0.000' COMMENT 'on_hand - allocated',
  `reorder_level` decimal(10,2) DEFAULT '0.00',
  `quantity_in_transit` decimal(15,3) DEFAULT '0.000',
  `reorder_point` decimal(15,3) DEFAULT NULL,
  `max_stock_level` decimal(15,3) DEFAULT NULL,
  `average_cost` decimal(15,2) DEFAULT NULL,
  `last_movement_date` datetime DEFAULT NULL,
  `last_count_date` date DEFAULT NULL,
  `org_id` varchar(255) DEFAULT NULL,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `stock_batch`
--

CREATE TABLE `stock_batch` (
  `id` int NOT NULL,
  `product_id` int NOT NULL,
  `batch_number` varchar(100) NOT NULL,
  `warehouse_id` int NOT NULL,
  `quantity` decimal(15,3) NOT NULL,
  `unit_cost` decimal(15,2) DEFAULT NULL,
  `manufacture_date` date DEFAULT NULL,
  `expiry_date` date DEFAULT NULL,
  `supplier_id` int DEFAULT NULL,
  `received_date` date DEFAULT NULL,
  `quality_status` enum('passed','failed','pending','quarantine') DEFAULT 'pending',
  `is_active` tinyint(1) DEFAULT '1',
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `stock_movement`
--

CREATE TABLE `stock_movement` (
  `id` int NOT NULL,
  `movement_code` varchar(50) NOT NULL,
  `product_id` int NOT NULL,
  `movement_type` enum('inbound','outbound','transfer','adjustment','return','damage','expired') NOT NULL,
  `reference_type` enum('purchase_order','sales_order','transfer','adjustment','production','return') NOT NULL,
  `reference_id` int DEFAULT NULL COMMENT 'ID of related document (PO, SO, etc)',
  `from_warehouse_id` int DEFAULT NULL,
  `from_zone_id` int DEFAULT NULL,
  `to_warehouse_id` int DEFAULT NULL,
  `to_zone_id` int DEFAULT NULL,
  `quantity` decimal(15,3) NOT NULL,
  `unit_cost` decimal(15,2) DEFAULT NULL,
  `total_cost` decimal(15,2) DEFAULT NULL,
  `batch_number` varchar(100) DEFAULT NULL,
  `serial_number` varchar(100) DEFAULT NULL,
  `expiry_date` date DEFAULT NULL,
  `movement_date` datetime NOT NULL,
  `reason` text,
  `notes` text,
  `approved_by` int DEFAULT NULL,
  `processed_by` int DEFAULT NULL,
  `status` enum('pending','approved','completed','cancelled') DEFAULT 'pending',
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

--
-- Triggers `stock_movement`
--
DELIMITER $$
CREATE TRIGGER `trg_check_low_stock_after_movement` AFTER INSERT ON `stock_movement` FOR EACH ROW BEGIN
    DECLARE v_current_qty DECIMAL(15,3);
    DECLARE v_reorder_point DECIMAL(15,3);
    
    -- Get current stock and reorder point
    SELECT quantity_available, reorder_point 
    INTO v_current_qty, v_reorder_point
    FROM stock_balance
    WHERE product_id = NEW.product_id 
    AND warehouse_id = COALESCE(NEW.to_warehouse_id, NEW.from_warehouse_id)
    LIMIT 1;
    
    -- Create alert if below reorder point
    IF v_current_qty <= v_reorder_point THEN
        INSERT INTO inventory_alert (
            alert_type, product_id, warehouse_id, severity, message,
            current_quantity, threshold_quantity, org_id
        ) VALUES (
            'low_stock', NEW.product_id, COALESCE(NEW.to_warehouse_id, NEW.from_warehouse_id),
            CASE WHEN v_current_qty <= 0 THEN 'critical' ELSE 'high' END,
            CONCAT('Product stock is below reorder point'),
            v_current_qty, v_reorder_point, NEW.org_id
        );
    END IF;
END
$$
DELIMITER ;

-- --------------------------------------------------------

--
-- Table structure for table `stock_transfer`
--

CREATE TABLE `stock_transfer` (
  `id` int NOT NULL,
  `transfer_number` varchar(50) NOT NULL,
  `transfer_date` date NOT NULL,
  `from_warehouse_id` int NOT NULL,
  `to_warehouse_id` int NOT NULL,
  `expected_delivery_date` date DEFAULT NULL,
  `actual_delivery_date` date DEFAULT NULL,
  `transfer_type` enum('inter_warehouse','inter_zone','return','relocation') DEFAULT 'inter_warehouse',
  `shipping_method` varchar(100) DEFAULT NULL,
  `tracking_number` varchar(100) DEFAULT NULL,
  `shipping_cost` decimal(15,2) DEFAULT '0.00',
  `reason` text,
  `notes` text,
  `status` enum('draft','pending','in_transit','received','partially_received','cancelled') DEFAULT 'draft',
  `created_by` int NOT NULL,
  `approved_by` int DEFAULT NULL,
  `received_by` int DEFAULT NULL,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `stock_transfer_item`
--

CREATE TABLE `stock_transfer_item` (
  `id` int NOT NULL,
  `transfer_id` int NOT NULL,
  `product_id` int NOT NULL,
  `from_zone_id` int DEFAULT NULL,
  `to_zone_id` int DEFAULT NULL,
  `batch_number` varchar(100) DEFAULT NULL,
  `quantity_sent` decimal(15,3) NOT NULL,
  `quantity_received` decimal(15,3) DEFAULT '0.000',
  `quantity_damaged` decimal(15,3) DEFAULT '0.000',
  `unit_cost` decimal(15,2) DEFAULT NULL,
  `notes` text,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `supplier`
--

CREATE TABLE `supplier` (
  `id` int NOT NULL,
  `user_id` int NOT NULL,
  `supplier_code` varchar(50) NOT NULL,
  `supplier_name` varchar(255) NOT NULL,
  `company_name` varchar(255) DEFAULT NULL,
  `email` varchar(100) DEFAULT NULL,
  `phone` varchar(50) DEFAULT NULL,
  `mobile` varchar(50) DEFAULT NULL,
  `website` varchar(255) DEFAULT NULL,
  `tax_id` varchar(100) DEFAULT NULL,
  `payment_terms` varchar(100) DEFAULT NULL COMMENT 'e.g., Net 30, Net 60',
  `credit_limit` decimal(15,2) DEFAULT NULL,
  `currency` varchar(10) DEFAULT 'USD',
  `address` text,
  `city` varchar(100) DEFAULT NULL,
  `state` varchar(100) DEFAULT NULL,
  `country` varchar(100) DEFAULT NULL,
  `postal_code` varchar(20) DEFAULT NULL,
  `contact_person` varchar(255) DEFAULT NULL,
  `contact_phone` varchar(50) DEFAULT NULL,
  `contact_email` varchar(100) DEFAULT NULL,
  `rating` decimal(3,2) DEFAULT NULL COMMENT 'Supplier rating 1-5',
  `lead_time_days` int DEFAULT '0',
  `is_active` tinyint(1) DEFAULT '1',
  `notes` text,
  `role_id` int NOT NULL,
  `role_name` varchar(255) NOT NULL,
  `status` varchar(255) NOT NULL,
  `department_id` int NOT NULL,
  `org_id` varchar(255) DEFAULT NULL,
  `created_by` int DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `supplier_product`
--

CREATE TABLE `supplier_product` (
  `id` int NOT NULL,
  `supplier_id` int NOT NULL,
  `product_id` int NOT NULL,
  `supplier_product_code` varchar(100) DEFAULT NULL,
  `supplier_product_name` varchar(255) DEFAULT NULL,
  `unit_price` decimal(15,2) DEFAULT NULL,
  `currency` varchar(10) DEFAULT 'USD',
  `minimum_order_quantity` decimal(15,3) DEFAULT '1.000',
  `lead_time_days` int DEFAULT '0',
  `is_preferred` tinyint(1) DEFAULT '0',
  `last_purchase_price` decimal(15,2) DEFAULT NULL,
  `last_purchase_date` date DEFAULT NULL,
  `is_active` tinyint(1) DEFAULT '1',
  `notes` text,
  `org_id` varchar(255) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `tblproduct`
--

CREATE TABLE `tblproduct` (
  `id` int NOT NULL,
  `name` varchar(255) DEFAULT NULL,
  `product_code` varchar(255) DEFAULT NULL,
  `image` varchar(255) DEFAULT 'xd-assets/img/10101.jpg',
  `price` float DEFAULT NULL,
  `charge_rate` varchar(255) NOT NULL DEFAULT 'Once off',
  `quantity` int DEFAULT NULL,
  `reorder_level` int DEFAULT NULL,
  `cost` int NOT NULL DEFAULT '0',
  `max_stock_level` int DEFAULT NULL,
  `warranty` varchar(255) NOT NULL,
  `description` varchar(255) DEFAULT NULL,
  `color` varchar(255) DEFAULT NULL,
  `weight` varchar(255) DEFAULT NULL,
  `dimensions` varchar(255) DEFAULT NULL,
  `images` varchar(255) CHARACTER SET utf8mb4 COLLATE utf8mb4_bin DEFAULT NULL,
  `videos` varchar(255) CHARACTER SET utf8mb4 COLLATE utf8mb4_bin DEFAULT NULL,
  `category` varchar(255) DEFAULT NULL,
  `category_code` varchar(255) DEFAULT NULL,
  `sub_category` varchar(255) DEFAULT NULL,
  `coa_type` varchar(255) DEFAULT NULL,
  `units` int DEFAULT NULL,
  `product_type` varchar(255) NOT NULL DEFAULT 'intangible',
  `tracking_type` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `product_brand` varchar(255) DEFAULT NULL,
  `barcode` varchar(255) DEFAULT NULL,
  `lead_time` varchar(255) DEFAULT NULL,
  `tax_rate` int NOT NULL DEFAULT '0',
  `date_of_manufacture` varchar(255) DEFAULT NULL,
  `expiry_date` varchar(255) DEFAULT NULL,
  `product_location` varchar(255) DEFAULT NULL,
  `depot_id` int DEFAULT NULL,
  `penalty_code` varchar(255) DEFAULT NULL,
  `workAddress` varchar(255) DEFAULT NULL,
  `lat_area` float(10,6) DEFAULT NULL,
  `lng_area` float(10,6) DEFAULT NULL,
  `radius` varchar(255) NOT NULL DEFAULT '3',
  `path` varchar(255) DEFAULT NULL,
  `user_id` int DEFAULT NULL,
  `department` varchar(255) DEFAULT NULL,
  `status` varchar(255) NOT NULL DEFAULT 'active',
  `due_date` date DEFAULT NULL,
  `due_time` time DEFAULT NULL,
  `supplier_name` varchar(255) DEFAULT NULL,
  `supplier_id` int DEFAULT NULL,
  `datemodified` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `org_name` varchar(255) DEFAULT NULL,
  `org_id` varchar(255) DEFAULT NULL,
  `sessionUserId` int DEFAULT NULL
) ENGINE=InnoDB DEFAULT CHARSET=latin1;

-- --------------------------------------------------------

--
-- Table structure for table `temproal_password`
--

CREATE TABLE `temproal_password` (
  `id` int NOT NULL,
  `user_id` int DEFAULT NULL,
  `temppass` varchar(255) COLLATE utf8mb3_unicode_ci DEFAULT NULL,
  `timestamp` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `org_id` int NOT NULL DEFAULT '0'
) ENGINE=MyISAM DEFAULT CHARSET=utf8mb3 COLLATE=utf8mb3_unicode_ci;

-- --------------------------------------------------------

--
-- Table structure for table `users`
--

CREATE TABLE `users` (
  `id` int NOT NULL,
  `email` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `names` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `gender` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci NOT NULL,
  `phone` varchar(20) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `password` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `address` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `nrc` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `avatar` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT 'xd-assets/img/profile_picture.png',
  `role` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci NOT NULL DEFAULT 'user',
  `role_id` int DEFAULT NULL,
  `department` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `department_id` int DEFAULT NULL,
  `org_name` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `company_registration_number` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `tpin` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `area` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `area_id` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `area_code` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `plot_number` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `org_id` int DEFAULT NULL,
  `business_category` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `user_type` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `status` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT 'active',
  `signup` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `lastlogin` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci DEFAULT NULL,
  `activated` enum('0','1') NOT NULL DEFAULT '1',
  `lat` float(10,6) DEFAULT NULL,
  `lng` float(10,6) DEFAULT NULL,
  `gps_address` varchar(255) DEFAULT NULL,
  `gps_timestamp` varchar(255) DEFAULT NULL,
  `sessionUserId` varchar(255) DEFAULT NULL,
  `user_id` int NOT NULL DEFAULT '0',
  `user_name` varchar(255) CHARACTER SET latin1 COLLATE latin1_swedish_ci NOT NULL DEFAULT '0',
  `timestamp` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=latin1;

-- --------------------------------------------------------

--
-- Stand-in structure for view `vw_current_stock_levels`
-- (See below for the actual view)
--
CREATE TABLE `vw_current_stock_levels` (
`average_cost` decimal(15,2)
,`category` varchar(255)
,`last_movement_date` datetime
,`org_id` varchar(255)
,`product_code` varchar(255)
,`product_id` int
,`product_name` varchar(255)
,`quantity_allocated` decimal(15,3)
,`quantity_available` decimal(15,3)
,`quantity_in_transit` decimal(15,3)
,`quantity_on_hand` decimal(15,3)
,`reorder_point` decimal(15,3)
,`stock_status` varchar(12)
,`total_value` decimal(30,5)
,`warehouse_id` int
,`warehouse_name` varchar(255)
);

-- --------------------------------------------------------

--
-- Stand-in structure for view `vw_expiring_stock`
-- (See below for the actual view)
--
CREATE TABLE `vw_expiring_stock` (
`batch_number` varchar(100)
,`days_to_expiry` int
,`expiry_date` date
,`expiry_status` varchar(8)
,`org_id` varchar(255)
,`product_code` varchar(255)
,`product_id` int
,`product_name` varchar(255)
,`quantity` decimal(15,3)
,`total_value_at_risk` decimal(30,5)
,`warehouse_name` varchar(255)
);

-- --------------------------------------------------------

--
-- Stand-in structure for view `vw_inventory_valuation_by_category`
-- (See below for the actual view)
--
CREATE TABLE `vw_inventory_valuation_by_category` (
`avg_unit_cost` decimal(19,6)
,`category` varchar(255)
,`org_id` varchar(255)
,`total_products` bigint
,`total_quantity` decimal(37,3)
,`total_value` decimal(52,5)
);

-- --------------------------------------------------------

--
-- Stand-in structure for view `vw_purchase_order_summary`
-- (See below for the actual view)
--
CREATE TABLE `vw_purchase_order_summary` (
`expected_delivery_date` date
,`fulfillment_percentage` decimal(47,7)
,`id` int
,`org_id` varchar(255)
,`po_date` date
,`po_number` varchar(50)
,`status` enum('draft','sent','confirmed','partially_received','received','closed','cancelled')
,`supplier_name` varchar(255)
,`total_amount` decimal(15,2)
,`total_items` bigint
,`total_quantity_ordered` decimal(37,3)
,`total_quantity_received` decimal(37,3)
);

-- --------------------------------------------------------

--
-- Stand-in structure for view `vw_reorder_alerts`
-- (See below for the actual view)
--
CREATE TABLE `vw_reorder_alerts` (
`lead_time_days` int
,`minimum_order_quantity` decimal(15,3)
,`org_id` varchar(255)
,`product_code` varchar(255)
,`product_id` int
,`product_name` varchar(255)
,`quantity_available` decimal(15,3)
,`quantity_to_order` decimal(16,3)
,`reorder_point` decimal(15,3)
,`supplier_name` varchar(255)
,`unit_price` decimal(15,2)
,`warehouse_name` varchar(255)
);

-- --------------------------------------------------------

--
-- Stand-in structure for view `vw_warehouse_utilization`
-- (See below for the actual view)
--
CREATE TABLE `vw_warehouse_utilization` (
`capacity` decimal(15,2)
,`current_utilization` decimal(5,2)
,`id` int
,`org_id` varchar(255)
,`total_inventory_value` decimal(52,5)
,`total_products` bigint
,`total_quantity` decimal(37,3)
,`utilization_status` varchar(8)
,`warehouse_name` varchar(255)
);

-- --------------------------------------------------------

--
-- Table structure for table `warehouse`
--

CREATE TABLE `warehouse` (
  `id` int NOT NULL,
  `warehouse_code` varchar(50) NOT NULL,
  `warehouse_name` varchar(255) NOT NULL,
  `warehouse_type` enum('main','branch','transit','virtual') DEFAULT 'branch',
  `address` text,
  `city` varchar(100) DEFAULT NULL,
  `state` varchar(100) DEFAULT NULL,
  `country` varchar(100) DEFAULT NULL,
  `postal_code` varchar(20) DEFAULT NULL,
  `lat` float(10,6) DEFAULT NULL,
  `lng` float(10,6) DEFAULT NULL,
  `phone` varchar(50) DEFAULT NULL,
  `email` varchar(100) DEFAULT NULL,
  `manager_id` int DEFAULT NULL,
  `capacity` decimal(15,2) DEFAULT NULL COMMENT 'Storage capacity in cubic meters',
  `current_utilization` decimal(5,2) DEFAULT '0.00' COMMENT 'Percentage',
  `is_active` tinyint(1) DEFAULT '1',
  `org_id` varchar(255) DEFAULT NULL,
  `created_by` int DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `warehouse_zone`
--

CREATE TABLE `warehouse_zone` (
  `id` int NOT NULL,
  `warehouse_id` int NOT NULL,
  `zone_code` varchar(50) NOT NULL,
  `zone_name` varchar(255) NOT NULL,
  `zone_type` enum('storage','picking','packing','receiving','shipping','quarantine') DEFAULT 'storage',
  `aisle` varchar(50) DEFAULT NULL,
  `rack` varchar(50) DEFAULT NULL,
  `shelf` varchar(50) DEFAULT NULL,
  `bin` varchar(50) DEFAULT NULL,
  `temperature_controlled` tinyint(1) DEFAULT '0',
  `temperature_range` varchar(50) DEFAULT NULL,
  `capacity` decimal(15,2) DEFAULT NULL,
  `is_active` tinyint(1) DEFAULT '1',
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci;

-- --------------------------------------------------------

--
-- Table structure for table `workflows`
--

CREATE TABLE `workflows` (
  `id` int NOT NULL,
  `workflow_name` varchar(255) COLLATE utf8mb4_general_ci NOT NULL,
  `org_name` varchar(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `org_id` int NOT NULL,
  `is_active` tinyint(1) DEFAULT '1',
  `workflow_type` varchar(255) COLLATE utf8mb4_general_ci NOT NULL,
  `sessionUserId` int NOT NULL,
  `timestamp` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- --------------------------------------------------------

--
-- Table structure for table `workflow_approvals`
--

CREATE TABLE `workflow_approvals` (
  `id` int NOT NULL,
  `workflow_id` int NOT NULL,
  `step_id` int NOT NULL,
  `approved_by` int DEFAULT NULL,
  `approval_status` enum('Pending','Approved','Declined') COLLATE utf8mb4_general_ci DEFAULT 'Pending',
  `approved_at` datetime DEFAULT NULL,
  `remarks` text COLLATE utf8mb4_general_ci,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- --------------------------------------------------------

--
-- Table structure for table `workflow_logs`
--

CREATE TABLE `workflow_logs` (
  `log_id` int NOT NULL,
  `workflow_request_id` int NOT NULL,
  `receipt_number` int NOT NULL,
  `previous_status` enum('Pending','In Review','Approved','Declined') COLLATE utf8mb4_general_ci DEFAULT NULL,
  `new_status` enum('Pending','In Review','Approved','Declined') COLLATE utf8mb4_general_ci DEFAULT NULL,
  `previous_approved_by` int DEFAULT NULL,
  `new_approved_by` int DEFAULT NULL,
  `previous_current_step` int DEFAULT NULL,
  `new_current_step` int DEFAULT NULL,
  `previous_role_in_progress` varchar(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `new_role_in_progress` varchar(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `change_date` datetime DEFAULT CURRENT_TIMESTAMP,
  `changed_by` int DEFAULT NULL,
  `log_type` varchar(255) COLLATE utf8mb4_general_ci NOT NULL DEFAULT 'RECEIPT',
  `timestamp` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- --------------------------------------------------------

--
-- Table structure for table `workflow_privileges`
--

CREATE TABLE `workflow_privileges` (
  `id` int NOT NULL,
  `workflow_id` int NOT NULL,
  `step_order` int NOT NULL,
  `role` varchar(255) COLLATE utf8mb4_general_ci NOT NULL,
  `action` varchar(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `next_step` int DEFAULT NULL,
  `is_final` tinyint(1) DEFAULT '0',
  `sessionUserId` int NOT NULL,
  `timestamp` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- --------------------------------------------------------

--
-- Table structure for table `workflow_requests`
--

CREATE TABLE `workflow_requests` (
  `id` int NOT NULL,
  `receipt_number` int NOT NULL,
  `requested_by` int DEFAULT NULL,
  `requester_name` varchar(255) COLLATE utf8mb4_general_ci NOT NULL,
  `request_date` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `comment` longtext COLLATE utf8mb4_general_ci NOT NULL,
  `response_comment` varchar(255) COLLATE utf8mb4_general_ci NOT NULL,
  `status` enum('Pending','In Review','Approved','Declined') COLLATE utf8mb4_general_ci DEFAULT 'Pending',
  `approved_by` int DEFAULT NULL,
  `approval_date` timestamp NULL DEFAULT NULL,
  `workflow_type` varchar(255) COLLATE utf8mb4_general_ci DEFAULT 'ZERO',
  `current_step` int DEFAULT NULL,
  `workflow_id` int DEFAULT NULL,
  `org_name` varchar(255) COLLATE utf8mb4_general_ci NOT NULL,
  `org_id` varchar(255) COLLATE utf8mb4_general_ci NOT NULL,
  `role_in_progress` varchar(255) COLLATE utf8mb4_general_ci DEFAULT 'Revenue Accountant'
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

--
-- Triggers `workflow_requests`
--
DELIMITER $$
CREATE TRIGGER `log_workflow_changes` AFTER UPDATE ON `workflow_requests` FOR EACH ROW BEGIN
    -- Insert a new record into workflow_logs only if relevant columns are updated
    IF (OLD.status <> NEW.status OR
        OLD.approved_by <> NEW.approved_by OR
        OLD.current_step <> NEW.current_step OR
        OLD.role_in_progress <> NEW.role_in_progress) THEN
        
        INSERT INTO workflow_logs (
            workflow_request_id,
            receipt_number,
            previous_status, new_status,
            previous_approved_by, new_approved_by,
            previous_current_step, new_current_step,
            previous_role_in_progress, new_role_in_progress,
            change_date,
            changed_by
        )
        VALUES (
            NEW.id,
            NEW.receipt_number,
            OLD.status, NEW.status,
            OLD.approved_by, NEW.approved_by,
            OLD.current_step, NEW.current_step,
            OLD.role_in_progress, NEW.role_in_progress,
            NOW(),
            NEW.requested_by  -- Assuming the user who made the change is stored in requested_by
        );
    END IF;
END
$$
DELIMITER ;
DELIMITER $$
CREATE TRIGGER `update_transaction_status_on_approval` AFTER UPDATE ON `workflow_requests` FOR EACH ROW BEGIN
    -- Check if the status has been updated to 'Approved'
    IF NEW.status = 'Approved' THEN
        -- Update the transaction_status in transaction_details to 'Reversed'
        UPDATE transaction_details
        SET transaction_status = 'Reversed'
        WHERE receipt_no = NEW.receipt_number;
    END IF;
END
$$
DELIMITER ;

-- --------------------------------------------------------

--
-- Table structure for table `workflow_steps`
--

CREATE TABLE `workflow_steps` (
  `id` int NOT NULL,
  `workflow_id` int NOT NULL,
  `step_order` int NOT NULL,
  `role` varchar(255) COLLATE utf8mb4_general_ci NOT NULL,
  `role_id` int NOT NULL,
  `action` varchar(255) COLLATE utf8mb4_general_ci DEFAULT NULL,
  `next_step` int DEFAULT NULL,
  `is_final` tinyint(1) DEFAULT '0',
  `sessionUserId` int NOT NULL,
  `org_id` int NOT NULL,
  `timestamp` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- --------------------------------------------------------

--
-- Structure for view `vw_current_stock_levels`
--
DROP TABLE IF EXISTS `vw_current_stock_levels`;

CREATE ALGORITHM=UNDEFINED DEFINER=`root`@`localhost` SQL SECURITY DEFINER VIEW `vw_current_stock_levels`  AS SELECT `p`.`id` AS `product_id`, `p`.`product_code` AS `product_code`, `p`.`name` AS `product_name`, `p`.`category` AS `category`, `w`.`id` AS `warehouse_id`, `w`.`warehouse_name` AS `warehouse_name`, `sb`.`quantity_on_hand` AS `quantity_on_hand`, `sb`.`quantity_allocated` AS `quantity_allocated`, `sb`.`quantity_available` AS `quantity_available`, `sb`.`quantity_in_transit` AS `quantity_in_transit`, `sb`.`reorder_point` AS `reorder_point`, (case when (`sb`.`quantity_available` <= 0) then 'Out of Stock' when (`sb`.`quantity_available` <= `sb`.`reorder_point`) then 'Low Stock' when (`sb`.`quantity_available` > `sb`.`max_stock_level`) then 'Overstock' else 'Normal' end) AS `stock_status`, `sb`.`average_cost` AS `average_cost`, (`sb`.`quantity_on_hand` * `sb`.`average_cost`) AS `total_value`, `sb`.`last_movement_date` AS `last_movement_date`, `p`.`org_id` AS `org_id` FROM ((`tblproduct` `p` join `stock_balance` `sb` on((`p`.`id` = `sb`.`product_id`))) join `warehouse` `w` on((`sb`.`warehouse_id` = `w`.`id`))) WHERE ((`p`.`status` = 'active') AND (`w`.`is_active` = true)) ;

-- --------------------------------------------------------

--
-- Structure for view `vw_expiring_stock`
--
DROP TABLE IF EXISTS `vw_expiring_stock`;

CREATE ALGORITHM=UNDEFINED DEFINER=`root`@`localhost` SQL SECURITY DEFINER VIEW `vw_expiring_stock`  AS SELECT `p`.`id` AS `product_id`, `p`.`product_code` AS `product_code`, `p`.`name` AS `product_name`, `w`.`warehouse_name` AS `warehouse_name`, `sb`.`batch_number` AS `batch_number`, `sb`.`quantity` AS `quantity`, `sb`.`expiry_date` AS `expiry_date`, (to_days(`sb`.`expiry_date`) - to_days(curdate())) AS `days_to_expiry`, (case when ((to_days(`sb`.`expiry_date`) - to_days(curdate())) < 0) then 'Expired' when ((to_days(`sb`.`expiry_date`) - to_days(curdate())) <= 30) then 'Critical' when ((to_days(`sb`.`expiry_date`) - to_days(curdate())) <= 90) then 'Warning' else 'Normal' end) AS `expiry_status`, (`sb`.`quantity` * `sb`.`unit_cost`) AS `total_value_at_risk`, `p`.`org_id` AS `org_id` FROM ((`tblproduct` `p` join `stock_batch` `sb` on((`p`.`id` = `sb`.`product_id`))) join `warehouse` `w` on((`sb`.`warehouse_id` = `w`.`id`))) WHERE ((`sb`.`expiry_date` is not null) AND (`sb`.`is_active` = true) AND ((to_days(`sb`.`expiry_date`) - to_days(curdate())) <= 90)) ;

-- --------------------------------------------------------

--
-- Structure for view `vw_inventory_valuation_by_category`
--
DROP TABLE IF EXISTS `vw_inventory_valuation_by_category`;

CREATE ALGORITHM=UNDEFINED DEFINER=`root`@`localhost` SQL SECURITY DEFINER VIEW `vw_inventory_valuation_by_category`  AS SELECT `p`.`category` AS `category`, count(distinct `p`.`id`) AS `total_products`, sum(`sb`.`quantity_on_hand`) AS `total_quantity`, sum((`sb`.`quantity_on_hand` * `sb`.`average_cost`)) AS `total_value`, avg(`sb`.`average_cost`) AS `avg_unit_cost`, `p`.`org_id` AS `org_id` FROM (`tblproduct` `p` join `stock_balance` `sb` on((`p`.`id` = `sb`.`product_id`))) WHERE (`p`.`status` = 'active') GROUP BY `p`.`category`, `p`.`org_id` ;

-- --------------------------------------------------------

--
-- Structure for view `vw_purchase_order_summary`
--
DROP TABLE IF EXISTS `vw_purchase_order_summary`;

CREATE ALGORITHM=UNDEFINED DEFINER=`root`@`localhost` SQL SECURITY DEFINER VIEW `vw_purchase_order_summary`  AS SELECT `po`.`id` AS `id`, `po`.`po_number` AS `po_number`, `po`.`po_date` AS `po_date`, `s`.`supplier_name` AS `supplier_name`, `po`.`total_amount` AS `total_amount`, `po`.`status` AS `status`, `po`.`expected_delivery_date` AS `expected_delivery_date`, count(`poi`.`id`) AS `total_items`, sum(`poi`.`quantity_ordered`) AS `total_quantity_ordered`, sum(`poi`.`quantity_received`) AS `total_quantity_received`, (case when (sum(`poi`.`quantity_received`) = 0) then 0 else ((sum(`poi`.`quantity_received`) / sum(`poi`.`quantity_ordered`)) * 100) end) AS `fulfillment_percentage`, `po`.`org_id` AS `org_id` FROM ((`purchase_order` `po` join `supplier` `s` on((`po`.`supplier_id` = `s`.`id`))) left join `purchase_order_item` `poi` on((`po`.`id` = `poi`.`po_id`))) GROUP BY `po`.`id` ;

-- --------------------------------------------------------

--
-- Structure for view `vw_reorder_alerts`
--
DROP TABLE IF EXISTS `vw_reorder_alerts`;

CREATE ALGORITHM=UNDEFINED DEFINER=`root`@`localhost` SQL SECURITY DEFINER VIEW `vw_reorder_alerts`  AS SELECT `p`.`id` AS `product_id`, `p`.`product_code` AS `product_code`, `p`.`name` AS `product_name`, `w`.`warehouse_name` AS `warehouse_name`, `sb`.`quantity_available` AS `quantity_available`, `sb`.`reorder_point` AS `reorder_point`, (`sb`.`reorder_point` - `sb`.`quantity_available`) AS `quantity_to_order`, `sp`.`supplier_name` AS `supplier_name`, `sp`.`lead_time_days` AS `lead_time_days`, `spr`.`unit_price` AS `unit_price`, `spr`.`minimum_order_quantity` AS `minimum_order_quantity`, `p`.`org_id` AS `org_id` FROM ((((`tblproduct` `p` join `stock_balance` `sb` on((`p`.`id` = `sb`.`product_id`))) join `warehouse` `w` on((`sb`.`warehouse_id` = `w`.`id`))) left join `supplier_product` `spr` on(((`p`.`id` = `spr`.`product_id`) and (`spr`.`is_preferred` = true)))) left join `supplier` `sp` on((`spr`.`supplier_id` = `sp`.`id`))) WHERE ((`sb`.`quantity_available` <= `sb`.`reorder_point`) AND (`p`.`status` = 'active') AND (`w`.`is_active` = true)) ;

-- --------------------------------------------------------

--
-- Structure for view `vw_warehouse_utilization`
--
DROP TABLE IF EXISTS `vw_warehouse_utilization`;

CREATE ALGORITHM=UNDEFINED DEFINER=`root`@`localhost` SQL SECURITY DEFINER VIEW `vw_warehouse_utilization`  AS SELECT `w`.`id` AS `id`, `w`.`warehouse_name` AS `warehouse_name`, `w`.`capacity` AS `capacity`, count(distinct `sb`.`product_id`) AS `total_products`, sum(`sb`.`quantity_on_hand`) AS `total_quantity`, sum((`sb`.`quantity_on_hand` * `sb`.`average_cost`)) AS `total_inventory_value`, `w`.`current_utilization` AS `current_utilization`, (case when (`w`.`current_utilization` >= 90) then 'Critical' when (`w`.`current_utilization` >= 75) then 'High' when (`w`.`current_utilization` >= 50) then 'Medium' else 'Low' end) AS `utilization_status`, `w`.`org_id` AS `org_id` FROM (`warehouse` `w` left join `stock_balance` `sb` on((`w`.`id` = `sb`.`warehouse_id`))) WHERE (`w`.`is_active` = true) GROUP BY `w`.`id` ;

--
-- Indexes for dumped tables
--

--
-- Indexes for table `cycle_count`
--
ALTER TABLE `cycle_count`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `count_number` (`count_number`),
  ADD KEY `idx_count_status` (`status`,`count_date`),
  ADD KEY `idx_count_warehouse` (`warehouse_id`,`count_date`);

--
-- Indexes for table `cycle_count_item`
--
ALTER TABLE `cycle_count_item`
  ADD PRIMARY KEY (`id`),
  ADD KEY `product_id` (`product_id`),
  ADD KEY `idx_cycle_items` (`cycle_count_id`),
  ADD KEY `idx_discrepancy` (`count_status`);

--
-- Indexes for table `demand_forecast`
--
ALTER TABLE `demand_forecast`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `unique_forecast` (`product_id`,`warehouse_id`,`forecast_period`,`forecast_type`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_forecast_period` (`forecast_period`);

--
-- Indexes for table `goods_receipt`
--
ALTER TABLE `goods_receipt`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `receipt_number` (`receipt_number`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_receipt_po` (`po_id`),
  ADD KEY `idx_receipt_date` (`receipt_date`);

--
-- Indexes for table `goods_receipt_item`
--
ALTER TABLE `goods_receipt_item`
  ADD PRIMARY KEY (`id`),
  ADD KEY `po_item_id` (`po_item_id`),
  ADD KEY `product_id` (`product_id`),
  ADD KEY `idx_receipt_items` (`receipt_id`);

--
-- Indexes for table `inventory_alert`
--
ALTER TABLE `inventory_alert`
  ADD PRIMARY KEY (`id`),
  ADD KEY `product_id` (`product_id`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_alert_unresolved` (`is_resolved`,`severity`),
  ADD KEY `idx_alert_type` (`alert_type`,`created_at`);

--
-- Indexes for table `inventory_audit_log`
--
ALTER TABLE `inventory_audit_log`
  ADD PRIMARY KEY (`id`),
  ADD KEY `idx_audit_entity` (`entity_type`,`entity_id`),
  ADD KEY `idx_audit_user` (`user_id`,`created_at`),
  ADD KEY `idx_audit_date` (`created_at`);

--
-- Indexes for table `inventory_snapshot`
--
ALTER TABLE `inventory_snapshot`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `unique_snapshot` (`snapshot_date`,`product_id`,`warehouse_id`),
  ADD KEY `product_id` (`product_id`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_snapshot_date` (`snapshot_date`);

--
-- Indexes for table `inventory_turnover`
--
ALTER TABLE `inventory_turnover`
  ADD PRIMARY KEY (`id`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_turnover_period` (`period_start`,`period_end`),
  ADD KEY `idx_turnover_product` (`product_id`);

--
-- Indexes for table `inventory_valuation`
--
ALTER TABLE `inventory_valuation`
  ADD PRIMARY KEY (`id`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_valuation_date` (`valuation_date`),
  ADD KEY `idx_valuation_product` (`product_id`,`valuation_date`);

--
-- Indexes for table `product_kit`
--
ALTER TABLE `product_kit`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `unique_kit_component` (`kit_product_id`,`component_product_id`),
  ADD KEY `component_product_id` (`component_product_id`),
  ADD KEY `idx_kit_components` (`kit_product_id`);

--
-- Indexes for table `purchase_order`
--
ALTER TABLE `purchase_order`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `po_number` (`po_number`),
  ADD KEY `requisition_id` (`requisition_id`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_po_status` (`status`,`po_date`),
  ADD KEY `idx_po_supplier` (`supplier_id`,`po_date`),
  ADD KEY `idx_po_org` (`org_id`);

--
-- Indexes for table `purchase_order_item`
--
ALTER TABLE `purchase_order_item`
  ADD PRIMARY KEY (`id`),
  ADD KEY `product_id` (`product_id`),
  ADD KEY `idx_po_items` (`po_id`);

--
-- Indexes for table `purchase_requisition`
--
ALTER TABLE `purchase_requisition`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `requisition_number` (`requisition_number`),
  ADD KEY `idx_requisition_status` (`status`,`requisition_date`),
  ADD KEY `idx_requisition_org` (`org_id`);

--
-- Indexes for table `purchase_requisition_item`
--
ALTER TABLE `purchase_requisition_item`
  ADD PRIMARY KEY (`id`),
  ADD KEY `product_id` (`product_id`),
  ADD KEY `idx_requisition_items` (`requisition_id`);

--
-- Indexes for table `quality_check`
--
ALTER TABLE `quality_check`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `check_number` (`check_number`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_qc_result` (`result`,`check_date`),
  ADD KEY `idx_qc_product` (`product_id`,`check_date`);

--
-- Indexes for table `reorder_recommendation`
--
ALTER TABLE `reorder_recommendation`
  ADD PRIMARY KEY (`id`),
  ADD KEY `product_id` (`product_id`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `po_id` (`po_id`),
  ADD KEY `idx_recommendation_status` (`status`,`priority`),
  ADD KEY `idx_recommendation_date` (`recommendation_date`);

--
-- Indexes for table `serial_number`
--
ALTER TABLE `serial_number`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `serial_number` (`serial_number`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_serial_status` (`status`),
  ADD KEY `idx_serial_product` (`product_id`,`status`);

--
-- Indexes for table `stock_adjustment`
--
ALTER TABLE `stock_adjustment`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `adjustment_number` (`adjustment_number`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_adjustment_status` (`status`,`adjustment_date`),
  ADD KEY `idx_adjustment_type` (`adjustment_type`);

--
-- Indexes for table `stock_adjustment_item`
--
ALTER TABLE `stock_adjustment_item`
  ADD PRIMARY KEY (`id`),
  ADD KEY `product_id` (`product_id`),
  ADD KEY `idx_adjustment_items` (`adjustment_id`);

--
-- Indexes for table `stock_balance`
--
ALTER TABLE `stock_balance`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `unique_stock` (`product_id`,`warehouse_id`,`zone_id`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_stock_available` (`quantity_available`),
  ADD KEY `idx_reorder_check` (`quantity_available`,`reorder_point`);

--
-- Indexes for table `stock_batch`
--
ALTER TABLE `stock_batch`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `unique_batch` (`product_id`,`batch_number`,`warehouse_id`),
  ADD KEY `warehouse_id` (`warehouse_id`),
  ADD KEY `idx_expiry` (`expiry_date`,`is_active`),
  ADD KEY `idx_batch_product` (`product_id`,`is_active`);

--
-- Indexes for table `stock_movement`
--
ALTER TABLE `stock_movement`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `movement_code` (`movement_code`),
  ADD KEY `from_warehouse_id` (`from_warehouse_id`),
  ADD KEY `to_warehouse_id` (`to_warehouse_id`),
  ADD KEY `idx_movement_product` (`product_id`,`movement_date`),
  ADD KEY `idx_movement_type` (`movement_type`,`status`),
  ADD KEY `idx_reference` (`reference_type`,`reference_id`);

--
-- Indexes for table `stock_transfer`
--
ALTER TABLE `stock_transfer`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `transfer_number` (`transfer_number`),
  ADD KEY `to_warehouse_id` (`to_warehouse_id`),
  ADD KEY `idx_transfer_status` (`status`,`transfer_date`),
  ADD KEY `idx_transfer_warehouses` (`from_warehouse_id`,`to_warehouse_id`);

--
-- Indexes for table `stock_transfer_item`
--
ALTER TABLE `stock_transfer_item`
  ADD PRIMARY KEY (`id`),
  ADD KEY `product_id` (`product_id`),
  ADD KEY `idx_transfer_items` (`transfer_id`);

--
-- Indexes for table `supplier`
--
ALTER TABLE `supplier`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `supplier_code` (`supplier_code`),
  ADD KEY `idx_supplier_org` (`org_id`,`is_active`),
  ADD KEY `idx_supplier_code` (`supplier_code`);

--
-- Indexes for table `supplier_product`
--
ALTER TABLE `supplier_product`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `unique_supplier_product` (`supplier_id`,`product_id`),
  ADD KEY `idx_preferred_supplier` (`product_id`,`is_preferred`);

--
-- Indexes for table `tblproduct`
--
ALTER TABLE `tblproduct`
  ADD PRIMARY KEY (`id`);

--
-- Indexes for table `temproal_password`
--
ALTER TABLE `temproal_password`
  ADD PRIMARY KEY (`id`);

--
-- Indexes for table `users`
--
ALTER TABLE `users`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `username` (`email`),
  ADD UNIQUE KEY `phone` (`phone`);

--
-- Indexes for table `warehouse`
--
ALTER TABLE `warehouse`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `warehouse_code` (`warehouse_code`),
  ADD KEY `idx_org_warehouse` (`org_id`,`warehouse_code`),
  ADD KEY `idx_warehouse_active` (`is_active`);

--
-- Indexes for table `warehouse_zone`
--
ALTER TABLE `warehouse_zone`
  ADD PRIMARY KEY (`id`),
  ADD UNIQUE KEY `unique_zone` (`warehouse_id`,`zone_code`),
  ADD KEY `idx_zone_type` (`zone_type`);

--
-- Indexes for table `workflows`
--
ALTER TABLE `workflows`
  ADD PRIMARY KEY (`id`);

--
-- Indexes for table `workflow_approvals`
--
ALTER TABLE `workflow_approvals`
  ADD PRIMARY KEY (`id`),
  ADD KEY `workflow_id` (`workflow_id`),
  ADD KEY `step_id` (`step_id`),
  ADD KEY `fk_approval_user` (`approved_by`);

--
-- Indexes for table `workflow_logs`
--
ALTER TABLE `workflow_logs`
  ADD PRIMARY KEY (`log_id`),
  ADD KEY `fk_workflow_request_id` (`workflow_request_id`);

--
-- Indexes for table `workflow_privileges`
--
ALTER TABLE `workflow_privileges`
  ADD PRIMARY KEY (`id`),
  ADD KEY `workflow_id` (`workflow_id`);

--
-- Indexes for table `workflow_requests`
--
ALTER TABLE `workflow_requests`
  ADD PRIMARY KEY (`id`),
  ADD KEY `fk_requested_by` (`requested_by`),
  ADD KEY `fk_approved_by` (`approved_by`),
  ADD KEY `fk_workflow_id` (`workflow_id`),
  ADD KEY `fk_current_step` (`current_step`);

--
-- Indexes for table `workflow_steps`
--
ALTER TABLE `workflow_steps`
  ADD PRIMARY KEY (`id`),
  ADD KEY `workflow_id` (`workflow_id`);

--
-- AUTO_INCREMENT for dumped tables
--

--
-- AUTO_INCREMENT for table `cycle_count`
--
ALTER TABLE `cycle_count`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `cycle_count_item`
--
ALTER TABLE `cycle_count_item`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `demand_forecast`
--
ALTER TABLE `demand_forecast`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `goods_receipt`
--
ALTER TABLE `goods_receipt`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `goods_receipt_item`
--
ALTER TABLE `goods_receipt_item`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `inventory_alert`
--
ALTER TABLE `inventory_alert`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `inventory_audit_log`
--
ALTER TABLE `inventory_audit_log`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `inventory_snapshot`
--
ALTER TABLE `inventory_snapshot`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `inventory_turnover`
--
ALTER TABLE `inventory_turnover`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `inventory_valuation`
--
ALTER TABLE `inventory_valuation`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `product_kit`
--
ALTER TABLE `product_kit`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `purchase_order`
--
ALTER TABLE `purchase_order`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `purchase_order_item`
--
ALTER TABLE `purchase_order_item`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `purchase_requisition`
--
ALTER TABLE `purchase_requisition`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `purchase_requisition_item`
--
ALTER TABLE `purchase_requisition_item`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `quality_check`
--
ALTER TABLE `quality_check`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `reorder_recommendation`
--
ALTER TABLE `reorder_recommendation`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `serial_number`
--
ALTER TABLE `serial_number`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `stock_adjustment`
--
ALTER TABLE `stock_adjustment`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `stock_adjustment_item`
--
ALTER TABLE `stock_adjustment_item`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `stock_balance`
--
ALTER TABLE `stock_balance`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `stock_batch`
--
ALTER TABLE `stock_batch`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `stock_movement`
--
ALTER TABLE `stock_movement`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `stock_transfer`
--
ALTER TABLE `stock_transfer`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `stock_transfer_item`
--
ALTER TABLE `stock_transfer_item`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `supplier`
--
ALTER TABLE `supplier`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `supplier_product`
--
ALTER TABLE `supplier_product`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `tblproduct`
--
ALTER TABLE `tblproduct`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `temproal_password`
--
ALTER TABLE `temproal_password`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `users`
--
ALTER TABLE `users`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `warehouse`
--
ALTER TABLE `warehouse`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `warehouse_zone`
--
ALTER TABLE `warehouse_zone`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `workflows`
--
ALTER TABLE `workflows`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `workflow_approvals`
--
ALTER TABLE `workflow_approvals`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `workflow_logs`
--
ALTER TABLE `workflow_logs`
  MODIFY `log_id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `workflow_privileges`
--
ALTER TABLE `workflow_privileges`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `workflow_requests`
--
ALTER TABLE `workflow_requests`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- AUTO_INCREMENT for table `workflow_steps`
--
ALTER TABLE `workflow_steps`
  MODIFY `id` int NOT NULL AUTO_INCREMENT;

--
-- Constraints for dumped tables
--

--
-- Constraints for table `cycle_count`
--
ALTER TABLE `cycle_count`
  ADD CONSTRAINT `cycle_count_ibfk_1` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `cycle_count_item`
--
ALTER TABLE `cycle_count_item`
  ADD CONSTRAINT `cycle_count_item_ibfk_1` FOREIGN KEY (`cycle_count_id`) REFERENCES `cycle_count` (`id`) ON DELETE CASCADE,
  ADD CONSTRAINT `cycle_count_item_ibfk_2` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`);

--
-- Constraints for table `demand_forecast`
--
ALTER TABLE `demand_forecast`
  ADD CONSTRAINT `demand_forecast_ibfk_1` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`),
  ADD CONSTRAINT `demand_forecast_ibfk_2` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `goods_receipt`
--
ALTER TABLE `goods_receipt`
  ADD CONSTRAINT `goods_receipt_ibfk_1` FOREIGN KEY (`po_id`) REFERENCES `purchase_order` (`id`),
  ADD CONSTRAINT `goods_receipt_ibfk_2` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `goods_receipt_item`
--
ALTER TABLE `goods_receipt_item`
  ADD CONSTRAINT `goods_receipt_item_ibfk_1` FOREIGN KEY (`receipt_id`) REFERENCES `goods_receipt` (`id`) ON DELETE CASCADE,
  ADD CONSTRAINT `goods_receipt_item_ibfk_2` FOREIGN KEY (`po_item_id`) REFERENCES `purchase_order_item` (`id`),
  ADD CONSTRAINT `goods_receipt_item_ibfk_3` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`);

--
-- Constraints for table `inventory_alert`
--
ALTER TABLE `inventory_alert`
  ADD CONSTRAINT `inventory_alert_ibfk_1` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`),
  ADD CONSTRAINT `inventory_alert_ibfk_2` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `inventory_snapshot`
--
ALTER TABLE `inventory_snapshot`
  ADD CONSTRAINT `inventory_snapshot_ibfk_1` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`),
  ADD CONSTRAINT `inventory_snapshot_ibfk_2` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `inventory_turnover`
--
ALTER TABLE `inventory_turnover`
  ADD CONSTRAINT `inventory_turnover_ibfk_1` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`),
  ADD CONSTRAINT `inventory_turnover_ibfk_2` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `inventory_valuation`
--
ALTER TABLE `inventory_valuation`
  ADD CONSTRAINT `inventory_valuation_ibfk_1` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`),
  ADD CONSTRAINT `inventory_valuation_ibfk_2` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `product_kit`
--
ALTER TABLE `product_kit`
  ADD CONSTRAINT `product_kit_ibfk_1` FOREIGN KEY (`kit_product_id`) REFERENCES `tblproduct` (`id`) ON DELETE CASCADE,
  ADD CONSTRAINT `product_kit_ibfk_2` FOREIGN KEY (`component_product_id`) REFERENCES `tblproduct` (`id`) ON DELETE CASCADE;

--
-- Constraints for table `purchase_order`
--
ALTER TABLE `purchase_order`
  ADD CONSTRAINT `purchase_order_ibfk_1` FOREIGN KEY (`supplier_id`) REFERENCES `supplier` (`id`),
  ADD CONSTRAINT `purchase_order_ibfk_2` FOREIGN KEY (`requisition_id`) REFERENCES `purchase_requisition` (`id`),
  ADD CONSTRAINT `purchase_order_ibfk_3` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `purchase_order_item`
--
ALTER TABLE `purchase_order_item`
  ADD CONSTRAINT `purchase_order_item_ibfk_1` FOREIGN KEY (`po_id`) REFERENCES `purchase_order` (`id`) ON DELETE CASCADE,
  ADD CONSTRAINT `purchase_order_item_ibfk_2` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`);

--
-- Constraints for table `purchase_requisition_item`
--
ALTER TABLE `purchase_requisition_item`
  ADD CONSTRAINT `purchase_requisition_item_ibfk_1` FOREIGN KEY (`requisition_id`) REFERENCES `purchase_requisition` (`id`) ON DELETE CASCADE,
  ADD CONSTRAINT `purchase_requisition_item_ibfk_2` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`);

--
-- Constraints for table `quality_check`
--
ALTER TABLE `quality_check`
  ADD CONSTRAINT `quality_check_ibfk_1` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`),
  ADD CONSTRAINT `quality_check_ibfk_2` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `reorder_recommendation`
--
ALTER TABLE `reorder_recommendation`
  ADD CONSTRAINT `reorder_recommendation_ibfk_1` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`),
  ADD CONSTRAINT `reorder_recommendation_ibfk_2` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`),
  ADD CONSTRAINT `reorder_recommendation_ibfk_3` FOREIGN KEY (`po_id`) REFERENCES `purchase_order` (`id`);

--
-- Constraints for table `serial_number`
--
ALTER TABLE `serial_number`
  ADD CONSTRAINT `serial_number_ibfk_1` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`),
  ADD CONSTRAINT `serial_number_ibfk_2` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `stock_adjustment`
--
ALTER TABLE `stock_adjustment`
  ADD CONSTRAINT `stock_adjustment_ibfk_1` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `stock_adjustment_item`
--
ALTER TABLE `stock_adjustment_item`
  ADD CONSTRAINT `stock_adjustment_item_ibfk_1` FOREIGN KEY (`adjustment_id`) REFERENCES `stock_adjustment` (`id`) ON DELETE CASCADE,
  ADD CONSTRAINT `stock_adjustment_item_ibfk_2` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`);

--
-- Constraints for table `stock_balance`
--
ALTER TABLE `stock_balance`
  ADD CONSTRAINT `stock_balance_ibfk_1` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`),
  ADD CONSTRAINT `stock_balance_ibfk_2` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `stock_batch`
--
ALTER TABLE `stock_batch`
  ADD CONSTRAINT `stock_batch_ibfk_1` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`),
  ADD CONSTRAINT `stock_batch_ibfk_2` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `stock_movement`
--
ALTER TABLE `stock_movement`
  ADD CONSTRAINT `stock_movement_ibfk_1` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`),
  ADD CONSTRAINT `stock_movement_ibfk_2` FOREIGN KEY (`from_warehouse_id`) REFERENCES `warehouse` (`id`),
  ADD CONSTRAINT `stock_movement_ibfk_3` FOREIGN KEY (`to_warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `stock_transfer`
--
ALTER TABLE `stock_transfer`
  ADD CONSTRAINT `stock_transfer_ibfk_1` FOREIGN KEY (`from_warehouse_id`) REFERENCES `warehouse` (`id`),
  ADD CONSTRAINT `stock_transfer_ibfk_2` FOREIGN KEY (`to_warehouse_id`) REFERENCES `warehouse` (`id`);

--
-- Constraints for table `stock_transfer_item`
--
ALTER TABLE `stock_transfer_item`
  ADD CONSTRAINT `stock_transfer_item_ibfk_1` FOREIGN KEY (`transfer_id`) REFERENCES `stock_transfer` (`id`) ON DELETE CASCADE,
  ADD CONSTRAINT `stock_transfer_item_ibfk_2` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`);

--
-- Constraints for table `supplier_product`
--
ALTER TABLE `supplier_product`
  ADD CONSTRAINT `supplier_product_ibfk_1` FOREIGN KEY (`supplier_id`) REFERENCES `supplier` (`id`) ON DELETE CASCADE,
  ADD CONSTRAINT `supplier_product_ibfk_2` FOREIGN KEY (`product_id`) REFERENCES `tblproduct` (`id`) ON DELETE CASCADE;

--
-- Constraints for table `warehouse_zone`
--
ALTER TABLE `warehouse_zone`
  ADD CONSTRAINT `warehouse_zone_ibfk_1` FOREIGN KEY (`warehouse_id`) REFERENCES `warehouse` (`id`) ON DELETE CASCADE;

--
-- Constraints for table `workflow_approvals`
--
ALTER TABLE `workflow_approvals`
  ADD CONSTRAINT `fk_approval_user` FOREIGN KEY (`approved_by`) REFERENCES `users` (`id`) ON DELETE SET NULL,
  ADD CONSTRAINT `workflow_approvals_ibfk_1` FOREIGN KEY (`workflow_id`) REFERENCES `workflows` (`id`) ON DELETE CASCADE,
  ADD CONSTRAINT `workflow_approvals_ibfk_2` FOREIGN KEY (`step_id`) REFERENCES `workflow_steps` (`id`) ON DELETE CASCADE;

--
-- Constraints for table `workflow_logs`
--
ALTER TABLE `workflow_logs`
  ADD CONSTRAINT `fk_workflow_request_id` FOREIGN KEY (`workflow_request_id`) REFERENCES `workflow_requests` (`id`) ON DELETE CASCADE;

--
-- Constraints for table `workflow_privileges`
--
ALTER TABLE `workflow_privileges`
  ADD CONSTRAINT `workflow_privileges_ibfk_1` FOREIGN KEY (`workflow_id`) REFERENCES `workflows` (`id`) ON DELETE CASCADE;

--
-- Constraints for table `workflow_requests`
--
ALTER TABLE `workflow_requests`
  ADD CONSTRAINT `fk_approved_by` FOREIGN KEY (`approved_by`) REFERENCES `users` (`id`) ON DELETE SET NULL,
  ADD CONSTRAINT `fk_current_step` FOREIGN KEY (`current_step`) REFERENCES `workflow_steps` (`id`) ON DELETE SET NULL,
  ADD CONSTRAINT `fk_requested_by` FOREIGN KEY (`requested_by`) REFERENCES `users` (`id`) ON DELETE SET NULL,
  ADD CONSTRAINT `fk_workflow_id` FOREIGN KEY (`workflow_id`) REFERENCES `workflows` (`id`) ON DELETE SET NULL;

--
-- Constraints for table `workflow_steps`
--
ALTER TABLE `workflow_steps`
  ADD CONSTRAINT `workflow_steps_ibfk_1` FOREIGN KEY (`workflow_id`) REFERENCES `workflows` (`id`) ON DELETE CASCADE;
 