// service-worker.js - FIXED VERSION

const CACHE_NAME = "cyberdesk-cache-v6"; // Increment version
const OFFLINE_FALLBACK = "/index.html";

const ASSETS_TO_CACHE = [
  "/",
  "/index.html",
  "/app.js",
  "/manifest.json",
  
  // Critical JS files - MUST BE CACHED
  "/xd-assets/js/login-scripts.js",
  "/xd-assets/js/user_profile.js",
  "/xd-assets/js/nav-scripts.js",
  "/xd-assets/js/sidenavigation.js",
  "/xd-assets/js/systemSettings.js",
  "/xd-assets/js/products.js",
  "/xd-assets/js/jquery-1.11.3.min.js",
  "/xd-assets/js/mainScript.js",
  
  // KEY FILES - These are your problem files
  "/xd-modules/payment-processing/transact.js",
  "/xd-modules/agent-module/js/transactions.js",
  "/xd-modules/billing-module/billing/js/invoiceCart.js",
  "/xd-modules/billing-module/billing/js/invoicePaymentCart.js",
  
  // Module HTML files
  "/xd-modules/agent-module/side-navigation.html",
  "/xd-modules/agent-module/js-scripts.html",
  "/xd-modules/revenue-accountant/side-navigation.html",
  "/xd-modules/revenue-accountant/js-scripts.html",
  "/xd-modules/billing-module/side-navigation.html",
  "/xd-modules/billing-module/js-scripts.html",
  "/xd-modules/modals.html",
  "/xd-modules/messenger-module/js/messenger.js",
  
  // CSS files
  "/xd-assets/css/w3.css",
  "/xd-assets/fontx-v1.0/style.css",
  "/xd-assets/css/xd-styles.css",
  "/xd-assets/css/printStyles.css",
  "/xd-assets/css/side-navigation.css",
  "/xd-assets/css/imagepreview.css",
  "/xd-assets/css/animations.css",
  "/xd-modules/messenger-module/css/messenger.css",
  
  // Images
  "/xd-assets/img/logo.png",
  "/192x192.png",
  "/512x512.png",
  "/logo.png",
  "/logo.jpg",
 'xd-assets/img/10101.jpg'
];

// ---- INSTALL: Cache assets ----
self.addEventListener('install', event => {
  console.log('🔧 Service Worker installing...');
  event.waitUntil(
    caches.open(CACHE_NAME).then(cache => {
      console.log('📦 Caching assets...');
      return Promise.all(
        ASSETS_TO_CACHE.map(asset => 
          cache.add(asset).catch(err => {
            console.warn("⚠️ Failed to cache:", asset, err.message);
          })
        )
      );
    }).then(() => {
      console.log('✅ Service Worker installed');
      return self.skipWaiting();
    })
  );
});

// ---- ACTIVATE: Remove old caches ----
self.addEventListener('activate', event => {
  console.log('🔄 Service Worker activating...');
  event.waitUntil(
    caches.keys().then(keys =>
      Promise.all(
        keys.map(key => {
          if (key !== CACHE_NAME) {
            console.log("🗑️ Deleting old cache:", key);
            return caches.delete(key);
          }
        })
      )
    ).then(() => {
      console.log('✅ Service Worker activated');
      return self.clients.claim();
    })
  );
});

// ---- FETCH: Network First for .js files, Cache First for others ----
self.addEventListener('fetch', event => {
  const request = event.request;
  const url = new URL(request.url);

  // Skip non-GET requests
  if (request.method !== 'GET') {
    return;
  }

  // Skip chrome-extension and other non-http(s) requests
  if (!url.protocol.startsWith('http')) {
    return;
  }

  // Skip API calls (backend PHP files)
  if (url.pathname.includes('/backend/') || url.pathname.endsWith('.php')) {
    event.respondWith(
      fetch(request).catch(() => {
        return new Response(
          JSON.stringify({ status: 'offline', message: 'No connection' }),
          { headers: { 'Content-Type': 'application/json' } }
        );
      })
    );
    return;
  }

  // Navigation requests (HTML pages)
  if (request.mode === 'navigate') {
    event.respondWith(
      fetch(request).catch(() => caches.match(OFFLINE_FALLBACK))
    );
    return;
  }

  // CRITICAL: Network First for JavaScript files
  if (url.pathname.endsWith('.js')) {
    event.respondWith(
      fetch(request)
        .then(response => {
          // Cache the fresh copy
          const responseClone = response.clone();
          caches.open(CACHE_NAME).then(cache => {
            cache.put(request, responseClone);
          });
          console.log('🌐 Fetched from network:', url.pathname);
          return response;
        })
        .catch(() => {
          console.log('📦 Serving from cache:', url.pathname);
          return caches.match(request).then(cached => {
            if (cached) {
              return cached;
            }
            console.error('❌ Not in cache:', url.pathname);
            return new Response('console.error("File not available offline: ' + url.pathname + '");', {
              headers: { 'Content-Type': 'application/javascript' }
            });
          });
        })
    );
    return;
  }

  // Cache First for everything else (CSS, images, HTML fragments)
  event.respondWith(
    caches.match(request).then(cached => {
      if (cached) {
        console.log('📦 Cache hit:', url.pathname);
        return cached;
      }

      return fetch(request)
        .then(response => {
          if (response.status === 200) {
            const responseClone = response.clone();
            caches.open(CACHE_NAME).then(cache => {
              cache.put(request, responseClone);
            });
          }
          return response;
        })
        .catch(() => {
          console.error('❌ Failed to fetch:', url.pathname);
          return caches.match(OFFLINE_FALLBACK);
        });
    })
  );
});