<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
session_start();

//— Load DB config
require_once '../../xd-assets/backend/config/dbconfig.php';

//— Connect to database
try {
    $conn = new PDO("mysql:host=$servername;dbname=$database", $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    echo json_encode(['status'=>'error','message'=>'DB connect failed: '.$e->getMessage()]);
    exit;
}

//— Verify session
$agentname    = $_SESSION["username"] ?? null;
$userloggedid = $_SESSION["userid"]   ?? null;
$orgid        = $_SESSION["orgid"]    ?? null;
$orgname      = $_SESSION["orgname"]  ?? null;
if (!$agentname || !$userloggedid || !$orgid || !$orgname) {
    echo json_encode(['status'=>'error','message'=>'Session incomplete']);
    exit;
}

//— Read & sanitize inputs
$cartObject          = filter_input(INPUT_POST,"orderDetails",FILTER_UNSAFE_RAW);
$cartArray           = json_decode($cartObject, true);
$amount              = 0;
if (is_array($cartArray)) {
    foreach ($cartArray as $item) {
        $amount += floatval($item['total'] ?? 0);
    }
}
$userid              = filter_input(INPUT_POST,"userid",FILTER_VALIDATE_INT);
$fname               = filter_input(INPUT_POST,"fname",FILTER_SANITIZE_STRING);
$lname               = filter_input(INPUT_POST,"lname",FILTER_SANITIZE_STRING);
$email               = filter_input(INPUT_POST,"email",FILTER_SANITIZE_EMAIL);
$phone               = filter_input(INPUT_POST,"phone",FILTER_SANITIZE_STRING);
$deliveryaddress     = filter_input(INPUT_POST,"deliveryaddress",FILTER_SANITIZE_STRING);
$deliverylat         = filter_input(INPUT_POST,"deliverylat",FILTER_SANITIZE_STRING);
$deliverylng         = filter_input(INPUT_POST,"deliverylng",FILTER_SANITIZE_STRING);
$transtype           = filter_input(INPUT_POST,"transtype",FILTER_SANITIZE_STRING);
$hiddenInvoiceNumber = filter_input(INPUT_POST,"hiddenInvoiceNumber",FILTER_SANITIZE_INT);
$invoiceDueDate      = filter_input(INPUT_POST,"invoiceDueDate",FILTER_SANITIZE_STRING);
$pacra               = filter_input(INPUT_POST,"pacra",FILTER_SANITIZE_STRING);
$tpin                = filter_input(INPUT_POST,"tpin",FILTER_SANITIZE_STRING);
$plotnumber          = filter_input(INPUT_POST,"plotnumber",FILTER_SANITIZE_STRING);
$area                = filter_input(INPUT_POST,"area",FILTER_SANITIZE_STRING);
$nrc                 = filter_input(INPUT_POST,"nrc",FILTER_SANITIZE_STRING);
$businessCategory    = filter_input(INPUT_POST,"businessCategory",FILTER_SANITIZE_STRING);
$comment             = filter_input(INPUT_POST,"comment",FILTER_SANITIZE_STRING);

//— Prepare transaction metadata
$transaction_reference = uniqid('d', true);
$order_code            = uniqid('o', true);
$transaction_date      = date("Y-m-d", strtotime("+2 hours"));
$transaction_time      = date("H:i:s", strtotime("+2 hours"));
$day                   = date('l', strtotime($transaction_date));
$month                 = date("F", strtotime($transaction_date));
$week                  = date("W", strtotime($transaction_date));

//— Fetch CGrate credentials
$stmt = $conn->prepare("
    SELECT public_key, private_key
      FROM api_details
     WHERE details_name='CGRATE ZAMBIA'
       AND currency='KWACHA'
       AND status='Active'
       AND org_id='100'
     LIMIT 1
");
$stmt->execute();
$api = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$api) {
    echo json_encode(['status'=>'error','message'=>'API credentials not found']);
    exit;
}
$cgrateUser = $api['public_key'];
$cgratePass = $api['private_key'];

//— Prepare SOAP call
$soapMobile      = preg_replace('/^0/', '260', $phone);
$paymentReference= uniqid('txn_', true);
$endpoint        = "https://test.543.cgrate.co.zm:8443/Konik/KonikWs";

function buildSoapXml($u, $p, $m, $a, $r) {
    return <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
                  xmlns:kon="http://konik.cgrate.com">
  <soapenv:Header>
    <wsse:Security xmlns:wsse="http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd" soapenv:mustUnderstand="1">
      <wsse:UsernameToken>
        <wsse:Username>{$u}</wsse:Username>
        <wsse:Password Type="http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText">{$p}</wsse:Password>
      </wsse:UsernameToken>
    </wsse:Security>
  </soapenv:Header>
  <soapenv:Body>
    <kon:processCustomerPayment>
      <transactionAmount>{$a}</transactionAmount>
      <customerMobile>{$m}</customerMobile>
      <paymentReference>{$r}</paymentReference>
    </kon:processCustomerPayment>
  </soapenv:Body>
</soapenv:Envelope>
XML;
}

function sendSoap($url, $xml) {
    $hdrs = [
        "Content-Type: text/xml; charset=utf-8",
        "SOAPAction: \"processCustomerPayment\"",
        "Content-Length: ".strlen($xml)
    ];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => $xml,
        CURLOPT_HTTPHEADER     => $hdrs,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_SSLVERSION     => CURL_SSLVERSION_TLSv1_2,
        CURLOPT_CONNECTTIMEOUT => 60,
        CURLOPT_TIMEOUT        => 120,
        CURLOPT_HEADER         => true,
    ]);
    $raw = curl_exec($ch);
    $err = curl_errno($ch) ? curl_error($ch) : null;
    $info = curl_getinfo($ch);
    curl_close($ch);

    return [
        'curl_error'=> $err,
        'http_code' => $info['http_code'] ?? 0,
        'headers'   => $raw !== false ? substr($raw, 0, $info['header_size']) : '',
        'body'      => $raw !== false ? substr($raw, $info['header_size']) : ''
    ];
}

function parseSoap($xml) {
    libxml_use_internal_errors(true);
    if ($s = simplexml_load_string($xml)) {
        $b   = $s->children('http://schemas.xmlsoap.org/soap/envelope/')->Body;
        $ret = $b->children('http://konik.cgrate.com')
                 ->processCustomerPaymentResponse->return;
        return [
            'code'     => (string)$ret->responseCode,
            'message'  => (string)$ret->responseMessage,
            'paymentID'=> (string)$ret->paymentID
        ];
    }
    if (preg_match('/<responseCode>(\d+)<\/responseCode>/', $xml, $m) &&
        preg_match('/<responseMessage>([^<]+)<\/responseMessage>/', $xml, $n)
    ) {
        return [
            'code'     => $m[1],
            'message'  => trim($n[1]),
            'paymentID'=> (preg_match('/<paymentID>([^<]+)<\/paymentID>/', $xml, $p) ? $p[1] : null)
        ];
    }
    return null;
}

//— Execute SOAP call
$soapXml = buildSoapXml($cgrateUser, $cgratePass, $soapMobile, $amount, $paymentReference);
$result  = sendSoap($endpoint, $soapXml);

//— Handle transport error
if ($result['curl_error']) {
    echo json_encode(['status'=>'error','message'=>'Transport error: '.$result['curl_error']]);
    exit;
}

//— Handle HTTP codes
if ($result['http_code'] !== 200) {
    echo json_encode([
        'status'=>'error',
        'message'=>"HTTP {$result['http_code']} from SOAP endpoint",
        'details'=>substr($result['body'], 0, 200)
    ]);
    exit;
}

//— Detect HTML error page
if (preg_match('#<\s*html#i', $result['body'])) {
    echo json_encode([
        'status'=>'error',
        'message'=>'Received HTML error page',
        'details'=>substr($result['body'], 0, 200)
    ]);
    exit;
}

//— Parse SOAP
$parsed = parseSoap($result['body']);
if (!$parsed) {
    echo json_encode([
        'status'=>'error',
        'message'=>'Malformed SOAP response',
        'details'=>substr($result['body'], 0, 200)
    ]);
    exit;
}

//— Business success?
$ok = ($parsed['code']==='0' || stripos($parsed['message'],'success')!==false);
if (!$ok) {
    echo json_encode(['status'=>'error','message'=>"Payment failed: {$parsed['message']}", 'code'=>$parsed['code']]);
    exit;
}

//— Payment succeeded: perform DB work
try {
    $conn->beginTransaction();

    // Insert receipt
    $r = $conn->prepare("INSERT INTO receipts (issuer_id, org_id) VALUES (:a,:o)");
    $r->execute([':a'=>$userloggedid,':o'=>$orgid]);
    $invoiceNumber = $conn->lastInsertId();

    // Insert transaction_details
    foreach ($cartArray as $item) {
        $t = $conn->prepare("
            INSERT INTO transaction_details (
                cfname, clname, cemail, cphone, product_name, product_id, product_code,
                product_category, product_category_code, balance_bf, transaction_amount,
                transaction_balance, quantity, order_code, transaction_reference,
                transaction_status, order_details, user_id, agent_name, agent_id,
                trans_type, debit_or_credit, depot, depot_id, depot_code,
                delivery_address, area, plot_no, nrc, lat, lng,
                transaction_date, transaction_time, transaction_month, transaction_week,
                transaction_day, receipt_no, invoiceNumber, comment, receipt_date,
                payment_mode, bank_name, bank_account_name, business_category,
                pacra, tpin, org_name, org_id
            ) VALUES (
                :fn,:ln,:em,:ph,:pn,:pid,:pcode,
                :cat,:cc,:bf,:ta,:tb,:qty,:oc,:tr,
                'active',:od,:uid,:an,:ai,
                :tt,'Credit',:dep,:depid,:depcode,
                :addr,:ar,:pl,:nr,:lt,:lg,
                :td,:ttm,:mo,:wk,:dy,:r,:iin,
                :cmt,:rd,'DIGITAL','DIGITAL','DIGITAL',
                :bc,:pa,:tp,:on,:oi
            )
        ");
        $t->execute([
            ':fn'=>$fname,':ln'=>$lname,':em'=>$email,':ph'=>$phone,
            ':pn'=>$item['name'],':pid'=>$item['id'],':pcode'=>$item['productcode'],
            ':cat'=>$item['category'],':cc'=>$item['categorycode'],':bf'=>$item['balance_bf'],
            ':ta'=>$item['total'],':tb'=>$item['balanceCF'],':qty'=>$item['count'],
            ':oc'=>$order_code,':tr'=>$transaction_reference,':od'=>$cartObject,
            ':uid'=>$userid,':an'=>$agentname,':ai'=>$userloggedid,':tt'=>$transtype,
            ':dep'=>$item['depot'],':depid'=>$item['depotid'],':depcode'=>$item['depotcode'],
            ':addr'=>$deliveryaddress,':ar'=>$area,':pl'=>$plotnumber,':nr'=>$nrc,
            ':lt'=>$deliverylat,':lg'=>$deliverylng,
            ':td'=>$transaction_date,':ttm'=>$transaction_time,
            ':mo'=>$month,':wk'=>$week,':dy'=>$day,
            ':r'=>$invoiceNumber,':iin'=>$hiddenInvoiceNumber,
            ':cmt'=>$comment,':rd'=>$invoiceDueDate,
            ':bc'=>$businessCategory,':pa'=>$pacra,':tp'=>$tpin,
            ':on'=>$orgname,':oi'=>$orgid
        ]);


        // Update billing_subscription
        // For each item in the cart, update the billing_subscription table to reflect the payment made.
        // The code below selects all billing_subscription records for the client and product with a non-zero balance, ordered by invoice number.
        // It then deducts the payment amount from the oldest balances first (FIFO), updating each record's balance accordingly.
        // If there is any remaining payment amount after clearing all positive balances, it deducts the remainder from the most recent invoice.
        // This ensures that the payment is applied to the oldest outstanding invoices first, and any excess is subtracted from the latest invoice.
        $f = $conn->prepare("
            SELECT b_id,balance FROM billing_subscription
            WHERE client_id=:uid AND product_id=:pid AND balance<>0
            ORDER BY invoice_no ASC
        ");
        $f->execute([':uid'=>$userid,':pid'=>$item['id']]);
        $remaining = floatval($item['total'] ?? 0);
        foreach ($f->fetchAll(PDO::FETCH_ASSOC) as $inv) {
            if ($remaining <= 0) break;
            if ($remaining >= $inv['balance']) {
                $remaining -= $inv['balance'];
                $newBal = 0;
            } else {
                $newBal = $inv['balance'] - $remaining;
                $remaining = 0;
            }
            $u = $conn->prepare("UPDATE billing_subscription SET balance=:nb WHERE b_id=:bid");
            $u->execute([':nb'=>$newBal,':bid'=>$inv['b_id']]);
        }
        if ($remaining > 0) {
            $fl = $conn->prepare("
                SELECT invoice_no,balance FROM billing_subscription
                WHERE client_id=:uid AND product_id=:pid
                ORDER BY invoice_no DESC LIMIT 1
            ");
            $fl->execute([':uid'=>$userid,':pid'=>$item['id']]);
            if ($li = $fl->fetch(PDO::FETCH_ASSOC)) {
                $ul = $conn->prepare("
                    UPDATE billing_subscription
                       SET balance = balance - :rem
                     WHERE client_id=:uid AND product_id=:pid AND invoice_no=:ino
                ");
                $ul->execute([
                    ':rem'=>$remaining,
                    ':uid'=>$userid,
                    ':pid'=>$item['id'],
                    ':ino'=>$li['invoice_no']
                ]);
            }
        } //end of updating billing_subscription datABASE

    }

    // Commit
    $conn->commit();

    echo json_encode([
        'status'=>'created',
        'message'=>'Payment successful and recorded',
        'receiptnumber'=>$invoiceNumber,
        'paymentID'=>$parsed['paymentID'],
        'amount'=>$amount,
        'date'          => date("Y-m-d"),
        'time'          => date("H:i:s"),
        'day'           => date('l')
    ]);
} catch (Exception $e) {
    $conn->rollBack();
    echo json_encode([
        'status'=>'error',
        'message'=>'DB error after payment: '.$e->getMessage()
    ]);
}